/*
 * sc-prim.c -- Implementation of Scheme's primitive string and
 *                  character manipulation routines
 *
 * (C) m.b (Matthias Blume); Jun 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * ident "@(#) sc-prim.c (C) M.Blume, Princeton University, 2.5"
 */

# ident "@(#)sc-prim.c	(C) M.Blume, Princeton University, 2.5"

# include <ctype.h>
# include <string.h>

# include "String.h"
# include "Character.h"
# include "Boolean.h"
# include "Cont.h"
# include "Numeric.h"
# include "Cons.h"
# include "Code.h"
# include "type.h"
# include "except.h"

# include "builtins.tab"

/*ARGSUSED*/
unsigned ScmPrimitiveCharP (unsigned argcnt)
{
  void *tmp = PEEK ();

  SET_TOP (ScmTypeOf (tmp) == ScmType (Character) ? &ScmTrue : &ScmFalse);
  return 0;
}

static unsigned monoton_compar (
  unsigned argcnt, int (* cmp) (void *, void *))
{
  void *cur, *tmp;
  unsigned i;

  if (argcnt < 2)
    error ("too few arguments to primitive char/string comparision");
  for (i = 1; i < argcnt; i++) {
    cur = POP ();
    tmp = PEEK ();
    if ((* cmp) (cur, tmp) == 0) {
      while (++i < argcnt)
	(void) POP ();
      SET_TOP (&ScmFalse);
      return 0;
    }
  }
  SET_TOP (&ScmTrue);
  return 0;
}

static int char_eq_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) && c1 == c2;
}

unsigned ScmPrimitiveCharEqP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_eq_cmp);
}

static int char_lt_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		C_CHAR (c1) < C_CHAR (c2);
}

unsigned ScmPrimitiveCharLtP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_lt_cmp);
}

static int char_gt_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		C_CHAR (c1) > C_CHAR (c2);
}

unsigned ScmPrimitiveCharGtP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_gt_cmp);
}

static int char_le_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		C_CHAR (c1) <= C_CHAR (c2);
}

unsigned ScmPrimitiveCharLeP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_le_cmp);
}

static int char_ge_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		C_CHAR (c1) >= C_CHAR (c2);
}

unsigned ScmPrimitiveCharGeP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_ge_cmp);
}

static int char_ci_eq_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		tolower (C_CHAR (c1)) == tolower (C_CHAR (c2));
}

unsigned ScmPrimitiveCharCiEqP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_ci_eq_cmp);
}

static int char_ci_lt_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		tolower (C_CHAR (c1)) < tolower (C_CHAR (c2));
}

unsigned ScmPrimitiveCharCiLtP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_ci_lt_cmp);
}

static int char_ci_gt_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		tolower (C_CHAR (c1)) > tolower (C_CHAR (c2));
}

unsigned ScmPrimitiveCharCiGtP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_ci_gt_cmp);
}

static int char_ci_le_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		tolower (C_CHAR (c1)) <= tolower (C_CHAR (c2));
}

unsigned ScmPrimitiveCharCiLeP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_ci_le_cmp);
}

static int char_ci_ge_cmp (void *c1, void *c2)
{
  return ScmTypeOf (c1) == ScmType (Character) &&
		tolower (C_CHAR (c1)) >= tolower (C_CHAR (c2));
}

unsigned ScmPrimitiveCharCiGeP (unsigned argcnt)
{
  return monoton_compar (argcnt, char_ci_ge_cmp);
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharAlphabeticP (unsigned argcnt)
{
  void *tmp = PEEK ();

  SET_TOP (ScmTypeOf (tmp) == ScmType (Character) && isalpha (C_CHAR (tmp))
	   ? &ScmTrue : &ScmFalse);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharNumericP (unsigned argcnt)
{
  void *tmp = PEEK ();

  SET_TOP (ScmTypeOf (tmp) == ScmType (Character) && isdigit (C_CHAR (tmp))
	   ? &ScmTrue : &ScmFalse);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharWhitespaceP (unsigned argcnt)
{
  void *tmp = PEEK ();

  SET_TOP (ScmTypeOf (tmp) == ScmType (Character) && isspace (C_CHAR (tmp))
	   ? &ScmTrue : &ScmFalse);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharUpperCaseP (unsigned argcnt)
{
  void *tmp = PEEK ();

  SET_TOP (ScmTypeOf (tmp) == ScmType (Character) && isupper (C_CHAR (tmp))
	   ? &ScmTrue : &ScmFalse);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharLowerCaseP (unsigned argcnt)
{
  void *tmp = PEEK ();

  SET_TOP (ScmTypeOf (tmp) == ScmType (Character) && islower (C_CHAR (tmp))
	   ? &ScmTrue : &ScmFalse);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharToInteger (unsigned argcnt)
{
  void *tmp = PEEK ();
  int c;

  if (ScmTypeOf (tmp) != ScmType (Character))
    error ("bad arg to primitive procedure char->integer: %w", tmp);
  c = C_CHAR (tmp);
  tmp = ScmLongToNumber (c);
  SET_TOP (tmp);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveIntegerToChar (unsigned argcnt)
{
  unsigned c = ScmNumberToUShort (PEEK (), "integer->char");

  if (c > 0377)
    error ("argument to integer->char out of range: %w", PEEK ());
  SET_TOP (&ScmCharacter_array [c]);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharUpcase (unsigned argcnt)
{
  void *tmp = PEEK ();

  if (ScmTypeOf (tmp) != ScmType (Character))
    error ("bad arg to primitive procedure char-upcase: %w", tmp);
  SET_TOP (&ScmCharacter_array [toupper (C_CHAR (tmp))]);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveCharDowncase (unsigned argcnt)
{
  void *tmp = PEEK ();

  if (ScmTypeOf (tmp) != ScmType (Character))
    error ("bad arg to primitive procedure char-downcase: %w", tmp);
  SET_TOP (&ScmCharacter_array [tolower (C_CHAR (tmp))]);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveStringP (unsigned argcnt)
{
  void *tmp = PEEK ();

  SET_TOP (ScmTypeOf (tmp) == ScmType (String) ? &ScmTrue : &ScmFalse);
  return 0;
}

unsigned ScmPrimitiveMakeString (unsigned argcnt)
{
  int fill;
  unsigned len, i;
  ScmString *string;
  void *tmp;

  if (argcnt < 1 || argcnt > 2)
    error ("wrong arg cnt (%u) to primitive procedure make-string",
           (unsigned) argcnt);
  len = ScmNumberToUShort (PEEK (), "make-string");
  SCM_VNEW (string, String, len, char);
  string->length = len;
  if (argcnt == 2) {
    (void) POP ();
    tmp = PEEK ();
    if (ScmTypeOf (tmp) != ScmType (Character))
      error ("bad second arg to primitive procedure make-string: %w", tmp);
    fill = C_CHAR (tmp);
  } else
    fill = ' ';
  for (i = 0; i < len; i++)
    string->array [i] = fill;
  SET_TOP (string);
  return 0;
}

unsigned ScmPrimitiveString (unsigned argcnt)
{
  ScmString *string;
  void *tmp;
  unsigned i;

  SCM_VNEW (string, String, argcnt, char);
  string->length = argcnt;
  for (i = 0; i < argcnt; i++) {
    tmp = POP ();
    if (ScmTypeOf (tmp) != ScmType (Character))
      error ("bad arg to primitive procedure string: %w", tmp);
    string->array [i] = C_CHAR (tmp);
  }
  Push (string);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveStringLength (unsigned argcnt)
{
  ScmString *string = PEEK ();
  void *tmp;

  if (ScmTypeOf (string) != ScmType (String))
    error ("bad arg to primitive procedure string-length: %w", string);
  tmp = ScmLongToNumber (string->length);
  SET_TOP (tmp);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveStringRef (unsigned argcnt)
{
  ScmString *string;
  unsigned k;

  string = POP ();
  k = ScmNumberToUShort (PEEK (), "string-ref");
  if (ScmTypeOf (string) != ScmType (String))
    error ("bad arg to string-ref: %w", string);
  if (k >= string->length)
    error ("string index (%i) out of bounds of %w", (int) k, string);
  SET_TOP (&ScmCharacter_array [(unsigned char) string->array [k]]);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveStringSet (unsigned argcnt)
{
  ScmString *string;
  void *obj;
  unsigned k;

  string = POP ();
  k = ScmNumberToUShort (POP (), "string-set!");
  obj = PEEK ();
  if (ScmTypeOf (string) != ScmType (String))
    error ("bad arg to string-set!: %w", string);
  if (k >= string->length)
    error ("string index (%i) out of bounds of %w", (int) k, string);
  if (ScmTypeOf (obj) != ScmType (Character))
    error ("bad arg to string-set!: %w", obj);
  string->array [k] = C_CHAR (obj);
  SET_TOP (string);
  return 0;
}

static int string_compare (
  ScmString *s1, ScmString *s2, int (* cmp) (unsigned char, unsigned char))
{
  unsigned i;
  int c;

  for (i = 0; i < s1->length && i < s2->length; i++) {
    c = (* cmp) (s1->array [i], s2->array [i]);
    if (c != 0)
      return c;
  }
  return (i < s1->length) ? 1 : (i < s2->length) ? -1 : 0;
}

static int string_cmp (unsigned char c1, unsigned char c2)
{
  return (c1 < c2) ? -1: (c1 == c2) ? 0 : 1;
}

static int string_ci_cmp (unsigned char c1, unsigned char c2)
{
  return string_cmp (tolower (c1), tolower (c2));
}

static int string_eq_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
         ScmTypeOf (s2) == ScmType (String) &&
         string_compare (s1, s2, string_cmp) == 0;
}

unsigned ScmPrimitiveStringEqP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_eq_cmp);
}

static int string_lt_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_cmp) < 0;
}

unsigned ScmPrimitiveStringLtP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_lt_cmp);
}

static int string_gt_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_cmp) > 0;
}

unsigned ScmPrimitiveStringGtP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_gt_cmp);
}

static int string_le_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_cmp) <= 0;
}

unsigned  ScmPrimitiveStringLeP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_le_cmp);
}

static int string_ge_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_cmp) >= 0;
}

unsigned ScmPrimitiveStringGeP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_ge_cmp);
}

static int string_ci_eq_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_ci_cmp) == 0;
}

unsigned ScmPrimitiveStringCiEqP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_ci_eq_cmp);
}

static int string_ci_lt_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_ci_cmp) < 0;
}

unsigned ScmPrimitiveStringCiLtP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_ci_lt_cmp);
}

static int string_ci_gt_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_ci_cmp) > 0;
}

unsigned ScmPrimitiveStringCiGtP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_ci_gt_cmp);
}

static int string_ci_le_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_ci_cmp) <= 0;
}

unsigned ScmPrimitiveStringCiLeP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_ci_le_cmp);
}

static int string_ci_ge_cmp (void *s1, void *s2)
{
  return ScmTypeOf (s1) == ScmType (String) &&
	 ScmTypeOf (s2) == ScmType (String) &&
	 string_compare (s1, s2, string_ci_cmp) >= 0;
}

unsigned ScmPrimitiveStringCiGeP (unsigned argcnt)
{
  return monoton_compar (argcnt, string_ci_ge_cmp);
}

/*ARGSUSED*/
unsigned ScmPrimitiveSubstring (unsigned argcnt)
{
  ScmString *string, *old;
  unsigned s, e, n;

  string = POP ();
  s = ScmNumberToUShort (POP (), "substring (arg1)");
  e = ScmNumberToUShort (PEEK (), "substring (arg2)");
  SET_TOP (string);
  if (ScmTypeOf (string) != ScmType (String))
    error ("bad arg to primitive procedure substring: %w", string);
  if (e < s || e > string->length)
    error ("range violation in primtive procedure (substring %w %i %i)",
	   string, (int) s, (int) e);
  n = e - s;
  SCM_VNEW (string, String, n, char);
  string->length = n;
  old = PEEK ();
  while (n-- > 0)
    string->array [n] = old->array [s + n];
  SET_TOP (string);
  return 0;
}

unsigned ScmPrimitiveStringAppend (unsigned argcnt)
{
  unsigned total;
  unsigned i;
  ScmString *string, *tmp;

  total = 0;
  for (i = 0; i < argcnt; i++) {
    tmp = POS (ScmCC->top - 1 - i);
    if (ScmTypeOf (tmp) != ScmType (String))
      error ("bad arg to primitive procedure string-append: %w", tmp);
    total += tmp->length;
  }
  SCM_VNEW (string, String, total, char);
  string->length = total;
  total = 0;
  while (argcnt-- > 0) {
    tmp = POP ();
    memcpy (string->array + total, tmp->array, tmp->length);
    total += tmp->length;
  }
  Push (string);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveStringToList (unsigned argcnt)
{
  ScmString *string = PEEK ();
  unsigned len;
  ScmCons *cons;

  if (ScmTypeOf (string) != ScmType (String))
    error ("bad arg to primitive procedure string->list: %w", string);
  len = string->length;
  SET_TOP (&ScmNil);
  Push (NULL);
  (void) POP ();
  while (len--) {
    PUSH (string);
    SCM_NEW (cons, Cons);
    string = POP ();
    cons->cdr = PEEK ();
    cons->car = &ScmCharacter_array [(unsigned char) string->array [len]];
    SET_TOP (cons);
  }
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveListToString (unsigned argcnt)
{
  ScmCons *l;
  unsigned long len, i;
  ScmString *string;
  void *tmp;

  len = ScmListLength (PEEK ());
  SCM_VNEW (string, String, len, char);
  string->length = len;
  l = PEEK ();
  for (i = 0; i < len; i++) {
    tmp = l->car;
    l = l->cdr;
    if (ScmTypeOf (tmp) != ScmType (Character))
      error ("list arg to primitive procedure list->string contains: %w", tmp);
    string->array [i] = C_CHAR (tmp);
  }
  SET_TOP (string);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveStringCopy (unsigned argcnt)
{
  ScmString *old = PEEK ();
  ScmString *string;

  if (ScmTypeOf (old) != ScmType (String))
    error ("bad arg to primitive procedure string-copy: %w", old);
  SCM_VNEW (string, String, old->length, char);
  old = PEEK ();
  string->length = old->length;
  memcpy (string->array, old->array, old->length);
  SET_TOP (string);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveStringFill (unsigned argcnt)
{
  ScmString *string = POP ();
  void *tmp;

  tmp = PEEK ();
  if (ScmTypeOf (string) != ScmType (String) ||
      ScmTypeOf (tmp) != ScmType (Character))
    error ("bad arg to primitive procedure (string-fill! %w %w)", string, tmp);
  memset (string->array, C_CHAR (tmp), string->length);
  SET_TOP (string);
  return 0;
}
