/*
 * storext.c -- Implementation of the Scheme type extension hook
 *
 * (C) m.b (Matthias Blume); Apr 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * ident "@(#) storext.c (C) M.Blume, Princeton University, 2.3"
 */

# ident "@(#)storext.c	(C) M.Blume, Princeton University, 2.3"

# include "storage.h"
# include "storext.h"
# include "numops.h"
# include "except.h"

/*
 * I/O operations -- this certainly goes beyond the realm of an all-purpose
 *		     storage module and must be considered part of Scheme.
 *		     Consequently, it has been separated from storage.[ch].
 */

# define Ext(obj) (ScmTypeOf (obj)->extension)

void display_object (void *obj, putc_proc pp, void *cd)
{
  if (obj == NULL)
    putc_string ("#<NULL>", pp, cd);
  else
    (* Ext (obj).display) (obj, pp, cd);
}

void write_object (void *obj, putc_proc pp, void *cd)
{
  if (obj == NULL)
    putc_string ("#<NULL>", pp, cd);
  else
    (* Ext (obj).write) (obj, pp, cd);
}

int equal_object (void *obj1, void *obj2)
{
  equal_proc ep;

  if (obj1 == NULL ||
      obj2 == NULL ||
      (ep = Ext (obj1).equal) == NULL_eq ||
      Ext (obj2).equal == NULL_eq)
    return obj1 == obj2;
  return (* ep) (obj1, obj2);
}

int eqv_object (void *obj1, void *obj2)
{
  equal_proc ep;

  if (obj1 == NULL ||
      obj2 == NULL ||
      (ep = Ext (obj1).eqv) == NULL_eq ||
      Ext (obj2).eqv == NULL_eq)
    return obj1 == obj2;
  return (* ep) (obj1, obj2);
}

void putc_string (const char *s, putc_proc pp, void *cd)
{
  while (*s != '\0')
    (* pp) (*s++, cd);
}

/*
 * All kinds of numerical predicates and operations...
 *   (using dispatch tables)
 *
 * The representation of Scheme numbers is as follows:
 *          |  exact      inexact
 * ---------+---------------------
 * integer  |  integer    real
 * fraction |  fraction   -
 * real     |  -          real
 * complex  |  -          complex
 *
 * All numbers are represented in the ``simplest'' possible type.
 * Therefore a fraction can never be equal to an integer, and a complex
 * can not be equal to anything but a complex.
 */

static binary_pred
  bin_pred_tab [N_NUMBER_CATEGORIES][N_NUMBER_CATEGORIES][N_BINARY_PREDS] = {
    {				/* SCM_NO_NUMBER */
      {				/* SCM_NO_NUMBER x SCM_NO_NUMBER */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_NO_NUMBER x SCM_INTEGER */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_NO_NUMBER x SCM_FRACTION */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_NO_NUMBER x SCM_REAL */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_NO_NUMBER x SCM_COMPLEX */
	bin_pred_error, bin_pred_error,
      },
    },
    {				/* SCM_INTEGER */
      {				/* SCM_INTEGER x SCM_NO_NUMBER */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_INTEGER x SCM_INTEGER */
	int_int_eq, int_int_cmp,
      },
      {				/* SCM_INTEGER x SCM_FRACTION */
	bin_pred_false, int_fract_cmp,
      },
      {				/* SCM_INTEGER x SCM_REAL */
	int_real_eq, int_real_cmp,
      },
      {				/* SCM_INTEGER x SCM_COMPLEX */
	bin_pred_false, bin_pred_error,
      },
    },
    {				/* SCM_FRACTION */
      {				/* SCM_FRACTION x SCM_NO_NUMBER */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_FRACTION x SCM_INTEGER */
	bin_pred_false, fract_int_cmp,
      },
      {				/* SCM_FRACTION x SCM_FRACTION */
	fract_fract_eq, fract_fract_cmp,
      },
      {				/* SCM_FRACTION x SCM_REAL */
	fract_real_eq, fract_real_cmp,
      },
      {				/* SCM_FRACTION x SCM_COMPLEX */
	bin_pred_false, bin_pred_error,
      },
    },
    {				/* SCM_REAL */
      {				/* SCM_REAL x SCM_NO_NUMBER */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_REAL x SCM_INTEGER */
	real_int_eq, real_int_cmp,
      },
      {				/* SCM_REAL x SCM_FRACTION */
	real_fract_eq, real_fract_cmp,
      },
      {				/* SCM_REAL x SCM_REAL */
	real_real_eq, real_real_cmp,
      },
      {				/* SCM_REAL x SCM_COMPLEX */
	bin_pred_false, bin_pred_error,
      },
    },
    {				/* SCM_COMPLEX */
      {				/* SCM_COMPLEX x SCM_NO_NUMBER */
	bin_pred_error, bin_pred_error,
      },
      {				/* SCM_COMPLEX x SCM_INTEGER */
	bin_pred_false, bin_pred_error,
      },
      {				/* SCM_COMPLEX x SCM_FRACTION */
	bin_pred_false, bin_pred_error,
      },
      {				/* SCM_COMPLEX x SCM_REAL */
	bin_pred_false, bin_pred_error,
      },
      {				/* SCM_COMPLEX x SCM_COMPLEX */
	cplx_cplx_eq, bin_pred_error,
      },
    },
  };

static binary_op
  bin_op_tab [N_NUMBER_CATEGORIES][N_NUMBER_CATEGORIES][N_BINARY_OPS] = {
    {				/* SCM_NO_NUMBER */
      {				/* SCM_NO_NUMBER x SCM_NO_NUMBER */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_NO_NUMBER x SCM_INTEGER */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_NO_NUMBER x SCM_FRACTION */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_NO_NUMBER x SCM_REAL */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_NO_NUMBER x SCM_COMPLEX */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
    },
    {				/* SCM_INTEGER */
      {				/* SCM_INTEGER x SCM_NO_NUMBER */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_INTEGER x SCM_INTEGER */
	int_int_add, int_int_sub, int_int_mul, int_int_div,
      },
      {				/* SCM_INTEGER x SCM_FRACTION */
	int_fract_add, int_fract_sub, int_fract_mul, int_fract_div,
      },
      {				/* SCM_INTEGER x SCM_REAL */
	real_real_add, real_real_sub, real_real_mul, real_real_div,
      },
      {				/* SCM_INTEGER x SCM_COMPLEX */
	real_cplx_add, real_cplx_sub, real_cplx_mul, real_cplx_div,
      },
    },
    {				/* SCM_FRACTION */
      {				/* SCM_FRACTION x SCM_NO_NUMBER */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_FRACTION x SCM_INTEGER */
	fract_int_add, fract_int_sub, fract_int_mul, fract_int_div,
      },
      {				/* SCM_FRACTION x SCM_FRACTION */
	fract_fract_add, fract_fract_sub, fract_fract_mul, fract_fract_div,
      },
      {				/* SCM_FRACTION x SCM_REAL */
	real_real_add, real_real_sub, real_real_mul, real_real_div,
      },
      {				/* SCM_FRACTION x SCM_COMPLEX */
	real_cplx_add, real_cplx_sub, real_cplx_mul, real_cplx_div,
      },
    },
    {				/* SCM_REAL */
      {				/* SCM_REAL x SCM_NO_NUMBER */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_REAL x SCM_INTEGER */
	real_real_add, real_real_sub, real_real_mul, real_real_div,
      },
      {				/* SCM_REAL x SCM_FRACTION */
	real_real_add, real_real_sub, real_real_mul, real_real_div,
      },
      {				/* SCM_REAL x SCM_REAL */
	real_real_add, real_real_sub, real_real_mul, real_real_div,
      },
      {				/* SCM_REAL x SCM_COMPLEX */
	real_cplx_add, real_cplx_sub, real_cplx_mul, real_cplx_div,
      },
    },
    {				/* SCM_COMPLEX */
      {				/* SCM_COMPLEX x SCM_NO_NUMBER */
	bin_op_error, bin_op_error, bin_op_error, bin_op_error,
      },
      {				/* SCM_COMPLEX x SCM_INTEGER */
	cplx_real_add, cplx_real_sub, cplx_real_mul, cplx_real_div,
      },
      {				/* SCM_COMPLEX x SCM_FRACTION */
	cplx_real_add, cplx_real_sub, cplx_real_mul, cplx_real_div,
      },
      {				/* SCM_COMPLEX x SCM_REAL */
	cplx_real_add, cplx_real_sub, cplx_real_mul, cplx_real_div,
      },
      {				/* SCM_COMPLEX x SCM_COMPLEX */
	cplx_cplx_add, cplx_cplx_sub, cplx_cplx_mul, cplx_cplx_div,
      },
    },
  };

static unary_pred
  u_pred_tab [N_NUMBER_CATEGORIES][N_UNARY_PREDS] = {
    {				/* SCM_NO_NUMBER */
      u_pred_error, u_pred_error, u_pred_error, u_pred_error,
      u_pred_false, u_pred_false, u_pred_false, u_pred_false,
    },
    {				/* SCM_INTEGER */
      int_zero, int_positive, int_negative, u_pred_true,
      u_pred_true, u_pred_true, u_pred_true, u_pred_true,
    },
    {				/* SCM_FRACTION */
      u_pred_false, fract_positive, fract_negative, u_pred_true,
      u_pred_false, u_pred_true, u_pred_true, u_pred_true,
    },
    {				/* SCM_REAL */
      real_zero, real_positive, real_negative, u_pred_false,
      real_is_int, real_is_int, u_pred_true, u_pred_true,
    },
    {				/* SCM_COMPLEX */
      u_pred_false, u_pred_error, u_pred_error, u_pred_false,
      u_pred_false, u_pred_false, u_pred_false, u_pred_true,
    },
  };

static unary_op
  u_op_tab [N_NUMBER_CATEGORIES][N_UNARY_OPS] = {
    {				/* SCM_NO_NUMBER */
      u_op_error, u_op_error, u_op_error, u_op_error, u_op_error, u_op_error,
      u_op_error, u_op_error, u_op_error, u_op_error, u_op_error, u_op_error,
      u_op_error, u_op_error, u_op_error, u_op_error, u_op_error, u_op_error,
      u_op_error, u_op_error, u_op_error, u_op_error, u_op_error, u_op_error,
    },
    {				/* SCM_INTEGER */
      int_abs, int_negate, int_inverse, u_op_id, int_one, u_op_id, u_op_id,
      u_op_id, u_op_id, real_exp, real_log, real_sin, real_cos, real_tan,
      real_asin, real_acos, real_atan, real_sqrt, int_abs, no_cplx_angle,
      u_op_id, u_op_zero, int_to_real, u_op_id,
    },
    {				/* SCM_FRACTION */
      fract_abs, fract_negate, fract_inverse, fract_numerator,
      fract_denominator, fract_floor, fract_ceiling, fract_truncate,
      fract_round, real_exp, real_log, real_sin, real_cos, real_tan,
      real_asin, real_acos, real_atan, real_sqrt, fract_abs, no_cplx_angle,
      u_op_id, u_op_zero, fract_to_real, u_op_id,
    },
    {				/* SCM_REAL */
      real_abs, real_negate, real_inverse, real_numerator, real_denominator,
      real_floor, real_ceiling, real_truncate, real_round, real_exp, real_log,
      real_sin, real_cos, real_tan, real_asin, real_acos, real_atan, real_sqrt,
      real_abs, no_cplx_angle, u_op_id, u_op_zero, u_op_id, real_to_fract, 
    },
    {				/* SCM_COMPLEX */
      u_op_error, cplx_negate, cplx_inverse, u_op_error, u_op_error,
      u_op_error, u_op_error, u_op_error, u_op_error, cplx_exp, cplx_log,
      cplx_sin, cplx_cos, cplx_tan, cplx_asin, cplx_acos, cplx_atan,
      cplx_sqrt, cplx_magnitude, cplx_angle, cplx_real_part, cplx_imag_part,
      u_op_id, u_op_error,
    },
  };
# define Categ(obj) (Ext (obj).numcat)

# define BIN_PRED(op,txt) txt,
static const char * const bin_pred_names [N_BINARY_PREDS] = {
# include "numops.tab"
};
# undef BIN_PRED

# define BIN_OP(op,txt) txt,
static const char * const bin_op_names [N_BINARY_OPS] = {
# include "numops.tab"
};
# undef BIN_OP

# define U_PRED(op,txt) txt,
static const char * const u_pred_names [N_UNARY_PREDS] = {
# include "numops.tab"
};
# undef U_PRED

# define U_OP(op,txt) txt,
static const char * const u_op_names [N_UNARY_OPS] = {
# include "numops.tab"
};
# undef U_OP

int ScmBinPred (enum binary_pred p, void *x, void *y)
{
  return
    (* bin_pred_tab [Categ (x)][Categ (y)][p])
      (x, y, bin_pred_names [p]);
}

void *ScmBinOp (enum binary_op o, void *x, void *y)
{
  return
    (* bin_op_tab [Categ (x)][Categ (y)][o])
      (x, y, bin_op_names [o]);
}

int ScmUPred (enum unary_pred p, void *x)
{
  return
    (* u_pred_tab [Categ (x)][p])
      (x, u_pred_names [p]);
}

void *ScmUOp (enum unary_op o, void *x)
{
  return
    (* u_op_tab [Categ (x)][o])
      (x, u_op_names [o]);
}

double ScmGetReal (void *x)
{
  return (* Ext (x).to_real) (x);
}

/*
 * the type-independend part of numops.h...
 */
int bin_pred_error (void *x, void *y, const char *s)
{
  error ("binary predicate error: (%s %w %w)", s, x, y);
}

/*ARGSUSED*/
int bin_pred_false (void *x, void *y, const char *s)
{
  return 0;
}

void *bin_op_error (void *x, void *y, const char *s)
{
  error ("binary operator error: (%s %w %w)", s, x, y);
}

int u_pred_error (void *x, const char *s)
{
  error ("unary predicate error: (%s %w)", s, x);
}

/*ARGSUSED*/
int u_pred_false (void *x, const char *s)
{
  return 0;
}

/*ARGSUSED*/
int u_pred_true (void *x, const char *s)
{
  return 1;
}

void *u_op_error (void *x, const char *s)
{
  error ("unary operator error: (%s %w)", s, x);
}

void *u_op_zero (void *x, const char *s)
{
  if (ScmUPred (SCM_EXACT_PRED, x))
    return exact_zero ();
  else
    return inexact_zero ();
}

/*ARGSUSED*/
void *u_op_id (void *x, const char *s)
{
  return x;
}

double cannot_cvt_real (void *x)
{
  error ("cannot convert %w to real", x);
}
