/*
 * misc-prim.c -- Implementation of miscellaneous Scheme primitives
 *
 * (C) m.b (Matthias Blume); May 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * ident "@(#) misc-prim.c (C) M.Blume, Princeton University, 2.9"
 */

# ident "@(#)misc-prim.c	(C) M.Blume, Princeton University, 2.9"

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <errno.h>

# include "storage.h"
# include "storext.h"
# include "Cont.h"
# include "Numeric.h"
# include "String.h"
# include "Boolean.h"
# include "Code.h"
# include "Primitive.h"
# include "Procedure.h"
# include "Symbol.h"
# include "Cons.h"
# include "type.h"
# include "tmpstring.h"
# include "mode.h"
# include "except.h"

# include "builtins.tab"

# define ERRORSTRING (errno == 0 ? "unknown reason" : strerror (errno))

/* Primitive No. 0: */
unsigned ScmVMTrapC (void)
{
  reset ("VM Trap");
}

unsigned ScmVMTrap (unsigned argcnt)
{
  return ScmVMTrapC ();
}

/* Primitive No. 1: */
unsigned ScmVMErrorC (void)
{
  reset ("tried to call error continuation");
}

unsigned ScmVMError (unsigned argcnt)
{
  return ScmVMErrorC ();
}

/* Primitive No. 2: */
unsigned ScmVMGCStrategyC (void)
{
  void *tmp = PEEK ();
  unsigned long bound;

  if (tmp == &ScmFalse)
    reset ("user gc-strategy gives up");
  if (ScmUPred (SCM_INTEGER_PRED, tmp)) {
    bound = ScmNumberToULong (tmp, "ScmVMGCStrategyC");
    MEM_min_heap_size = bound;
  }
  ScmRevertToFatherContinuation (1);
  (void) POP ();
  return 0;
}

unsigned ScmVMGCStrategy (unsigned argcnt)
{
  fatal ("internal error (ScmVMGCStrategy called)");
}

/* Primitive No. 3: */
unsigned ScmVMInterruptC (void)
{
  ScmRevertToFatherContinuation (1);
  (void) POP ();
  return 0;
}

unsigned ScmVMInterrupt (unsigned argcnt)
{
  fatal ("internal error (ScmVMInterrupt called)");
}

unsigned ScmPrimitiveQuit (unsigned argcnt)
{
  int stat = EXIT_SUCCESS;
  if (argcnt == 1)
    stat = ScmNumberToLong (POP (), "quit");
  else if (argcnt > 1)
    error ("wrong argcnt (%u) to primitive procedure quit",
	   (unsigned) argcnt);
  exit (stat);
  /*NOTREACHED*/
}

/*ARGSUSED*/
unsigned ScmPrimitiveDump (unsigned argcnt)
{
  void *tmp;
  ScmString *string;
  FILE *fp;
  char *filename;
  extern const char *dump_prefix; /* defined in main.c */

  tmp = PEEK ();
  if (ScmTypeOf (tmp) != ScmType (String))
    error ("wrong argument type for primitive procedure dump: %w", tmp);
  string = tmp;
  filename = tmpstring (string->array, string->length);
  errno = 0;
  if ((fp = fopen (filename, "wb")) == NULL)
    error ("cannot open file \"%s\" for dump (%s)", filename, ERRORSTRING);
  SET_TOP (&ScmFalse);
  MEM_dump_storage (fp, dump_prefix);
  errno = 0;
  if (fclose (fp) == EOF)
    error ("problems when closing dumpfile %w (%s)", tmp, ERRORSTRING);
  /* dump returns #f after successfully dumping the memory */
  /* see main.c for details how the return value gets patched on boot */
  /* dump returns a list of command-line arguments after boot */
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveExecuteAsm (unsigned argcnt)
{
  void *proc = ScmAsm (PEEK ());

  SET_TOP (proc);
  return 1;
}

/*ARGSUSED*/
unsigned ScmPrimitiveDefineAsm (unsigned argcnt)
{
  (void) ScmAsmDcl (PEEK ());
  SET_TOP (&ScmTrue);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveSystem (unsigned argcnt)
{
  ScmString *string;
  int status;

  string = PEEK ();
  if (ScmTypeOf (string) != ScmType (String))
    error ("bad arg to primitive procedure system: %w", string);
  status = system (tmpstring (string->array, string->length));
  SET_TOP (status == EXIT_SUCCESS ? &ScmTrue : &ScmFalse);
  return 0;
}

static unsigned call_with_mode (int mode_id)
{
  void *mode;

  mode = POP ();
  ScmPushPrimitiveContinuation (mode, 1);
  ScmSetMode (mode_id, ScmCC->u.c.environ);
  /* we allocated enough stack space -- this is safe */
  PUSH (CPOP (ScmCC->father));
  return 1;
}

static unsigned call_with_handler_mode (int mode_id, const char *strg)
{
  if (ScmTypeOf (PEEK ()) != ScmType (Procedure))
    error ("%s: argument is not a compiled LAMBDA: %w", strg, PEEK ());
  return call_with_mode (mode_id);
}

/*ARGSUSED*/
unsigned ScmPrimWithErrorHandler (unsigned argcnt)
{
  return call_with_handler_mode (SCM_ERROR_HANDLER_MODE, "with-error-handler");
}

/*ARGSUSED*/
unsigned ScmPrimWithGCStrategy (unsigned argcnt)
{
  return call_with_mode (SCM_GC_STRATEGY_MODE);
}

/*ARGSUSED*/
unsigned ScmPrimWithIntHandler (unsigned argcnt)
{
  return call_with_handler_mode (SCM_INTERRUPT_MODE, "with-interrupt-handler");
}

/*ARGSUSED*/
unsigned ScmPrimWithTimerExpHandler (unsigned argcnt)
{
  return call_with_handler_mode (SCM_TIMER_EXPIRATION_MODE,
				 "with-timer-expiration-handler");
}

unsigned ScmPrimWithSomethingC (void)
{
  ScmDirtyModeCache (ScmCC->u.c.mode_id);
  ScmRevertToFatherContinuation (1);
  return 0;
}

unsigned ScmPrimitiveTimer (unsigned argcnt)
{
  long l;

  if (argcnt == 0)
    l = -1;
  else if (argcnt == 1) {
    l = ScmNumberToLong (POP (), "timer");
    if (l < 0)
      error ("negative timer unit: %i", (int) l);
  } else
    error ("wrong arg cnt (%u) to primitive procedure timer",
	   (unsigned) argcnt);
  l = ScmTimer (l);
  Push (ScmLongToNumber (l));
  return 0;
}

static void *disassemble (ScmCode *code, unsigned long pc)
{
  const char *opcode;
  int length;
  unsigned short op1, op2, op3;
  long npc;
  void *tmp;
  int i = 3;

  npc = ScmDisassemble (code, pc, &opcode, &length, &op1, &op2, &op3);
  if (npc < 0)
    return &ScmFalse;
  tmp = ScmLongToNumber (pc);
  Push (tmp);
  tmp = ScmLongToNumber (npc);
  Push (tmp);
  tmp = ScmMakeSymbol (opcode, strlen (opcode));
  Push (tmp);
  if (length > 1) {
    i++;
    tmp = ScmLongToNumber (op1);
    Push (tmp);
    if (length > 2) {
      i++;
      tmp = ScmLongToNumber (op2);
      Push (tmp);
      if (length > 3) {
	i++;
	tmp = ScmLongToNumber (op3);
	Push (tmp);
      }
    }
  }
  tmp = &ScmNil;
  while (i-- > 0) {
    Push (tmp);
    SCM_NEW (tmp, Cons);
    ((ScmCons *) tmp)->cdr = POP ();
    ((ScmCons *) tmp)->car = POP ();
  }
  return tmp;
}

/*ARGSUSED*/
unsigned ScmPrimitiveDisassemble (unsigned argcnt)
{
  void *tmp;
  ScmCode *code;
  unsigned long pc;

  tmp = POP ();
  pc = ScmNumberToULong (PEEK (), "disassemble");
  if (ScmTypeOf (tmp) != ScmType (Code))
    error ("disassemble: %w (not a code object)", tmp);
  code = tmp;
  SET_TOP (disassemble (code, pc));
  return 0;
}

enum {
  CCONT_IDX,
  SHARED_IDX,
  STACK_IDX,
  PRIM_CODE_IDX,
  MODEID_PC_IDX,
  MODE_CONST_IDX,
  ENV_INST_IDX,

  INSPECT_VECT_LEN
};

/*ARGSUSED*/
unsigned ScmPrimitiveInspect (unsigned argcnt)
{
  ScmCont *cont;
  void *tmp;
  unsigned i;
  ScmVector *vect, *stack;
  unsigned long pc;

  tmp = POP ();
  i = ScmNumberToUShort (PEEK (), "inspect");
  while (i > 0 && tmp != NULL) {
    i--;
    tmp = ((ScmCont *) tmp)->father;
  }
  if (tmp == NULL) {
    SET_TOP (&ScmFalse);
    return 0;
  }
  cont = tmp;
  vect = NewScmVector (INSPECT_VECT_LEN);
  SET_TOP (vect);
  vect->array [SHARED_IDX] = ScmContIsShared (cont) ? &ScmTrue : &ScmFalse;
  PUSH (cont);			/* there is space */
  stack = NewScmVector (cont->top);
  cont = POP ();
  vect = PEEK ();
  for (i = cont->top; i-- > 0; )
    stack->array [i] = CPOS (cont, i);
  vect->array [STACK_IDX] = stack;
  if (ScmTypeOf (cont) == ScmType (CCont)) {
    vect->array [CCONT_IDX] = &ScmTrue;
    vect->array [PRIM_CODE_IDX] = GetScmPrimitive (cont->u.c.prim_no);
    vect->array [MODE_CONST_IDX] = cont->u.c.mode;
    vect->array [ENV_INST_IDX] = cont->u.c.environ;
    if (cont->u.c.mode == NULL)
      vect->array [MODEID_PC_IDX] = &ScmFalse;
    else {
      tmp = ScmLongToNumber (cont->u.c.mode_id);
      vect = PEEK ();
      vect->array [MODEID_PC_IDX] = tmp;
    }
  } else {
    vect->array [CCONT_IDX] = &ScmFalse;
    vect->array [PRIM_CODE_IDX] = cont->u.s.code;
    vect->array [MODE_CONST_IDX] = cont->u.s.constants;
    pc = cont->u.s.pc;
    tmp = disassemble (cont->u.s.code, pc);
    vect = PEEK ();
    vect->array [ENV_INST_IDX] = tmp;
    tmp = ScmLongToNumber (pc);
    vect = PEEK ();
    vect->array [MODEID_PC_IDX] = tmp;
  }
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveError (unsigned argcnt)
{
  error ("%d", PEEK ());
}

# define CLK2MS(clk) ((long) ((clk)*1000.0/CLOCKS_PER_SEC))

/*ARGSUSED*/
unsigned ScmPrimitiveClock (unsigned argcnt)
{
  void *tmp = ScmLongToNumber (CLK2MS (clock ()));

  Push (tmp);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveGcClock (unsigned argcnt)
{
  void *tmp = ScmLongToNumber (CLK2MS (MEM_total_gc_clock ()));

  Push (tmp);
  return 0;
}

/*ARGSUSED*/
unsigned ScmPrimitiveGetenv (unsigned argcnt)
{
  ScmString *strg;
  char *s;
  void *tmp;
  size_t len;

  tmp = PEEK ();
  if (ScmTypeOf (tmp) != ScmType (String))
    error ("getenv: %w (wrong arg)", tmp);
  strg = tmp;
  s = getenv (tmpstring (strg->array, strg->length));
  if (s == NULL)
    SET_TOP (&ScmFalse);
  else {
    len = strlen (s);
    SCM_VNEW (strg, String, len, char);
    strg->length = len;
    strncpy (strg->array, s, len);
    SET_TOP (strg);
  }
  return 0;
}

/* Primitive No. 5 */
/*ARGSUSED*/
unsigned ScmPrimCWValues (unsigned argcnt)
{
  void *thunk = POP ();
  void *cont = PEEK ();
  SET_TOP (thunk);
  ScmPushPrimitiveContinuation (cont, 1);
  PUSH (CPOP (ScmCC->father));	/* this is safe here */
  return 1;
}

unsigned ScmPrimCWValuesC (void)
{
  unsigned cnt;
  Push (ScmCC->u.c.environ);
  cnt = ScmCC->top;
  ScmRevertToFatherContinuation (cnt);
  return cnt;
}

unsigned ScmPrimitiveValues (unsigned argcnt)
{
  if (argcnt != 1 && ScmMultiCont (ScmCC) == 0)
    error ("values: continuation doesn't accept %u arguments",
	   (unsigned) argcnt);
  return 0;
}
