_Chapter(Debugging)
_label(chapter:debug)

The facilities described in this chapter are highly
implementation-specific, and subject to change without notice.
Facilities described here are intended either for use directly as
commands, or as utilities for user-written debugging subsystems.

_Section(Errors)
_index(errors)
_label(section:errorsInterface)        

When the implementation detects an error condition, a message is printed and
a read-eval-print loop is entered in the dynamic context in which
the error occurred.  A number of facilities are available at this
point for debugging and for recovering from the error.

Errors are usually detected in conditions which are left
undefined by the manual (see sections _Ref(section:undefinedSemantics)
and _Ref(section:errors)).  These conditions include the following:

_begin_Itemize
_Item()    An unbound variable is referenced.

_Item()    A procedure is called with too few or too many arguments.

_Item()    A non-procedure is called.

_Item()    An object of the wrong type is passed to a system procedure.

_Item()    A string or vector index is out of range.

_Item()    A generic operation is invoked on an object with no method
    to handle it.

_Item()    The external representation of an object being parsed by _tc(READ-OBJECT)
    or a read macro is syntactically incorrect.  (This kind of error
    is called a _Fit(read error).)

_Item()    An object being interpreted as an expression by a compiler or a
    macro expander is syntactically incorrect.  (This kind of error is
    called a _Fit(syntax error).)

_Item()    A call to _tc(ERROR) or _tc(UNDEFINED-EFFECT) occurs.
_end_Itemize

Once inside the read-eval-print loop, one may examine the current
environment (e.g., examining variables or the stack) using the
read-eval-print loop (section _Ref(section:repl)) or the inspector
(section _Ref(section:inspector)).

Typical actions after an error occurs include:
_begin_Itemize
_Item()The user corrects the error by returning a new value
with _tc(RET).

_Item()The user goes up one command level by typing the end-of-file character
(see page _Pageref(end-of-file)).

_Item()The user throws to top level by calling _tc((RESET)) (page _Pageref(RESET)).
_end_Itemize

Example:
_begin_Example
_Ftt(_cGT()) (CADR 3) _LNL

_star()_star() Error: attempting to take CADR of 3 _LNL
_Ftt(_cGT())_Ftt(_cGT()) (RET '(A B)) _LNL
BA _LNL
_Ftt(_cGT()) (PLUS 3 4) _LNL

_star()_star() Error: variable PLUS has no value _LNL
_Ftt(_cGT())_Ftt(_cGT()) (RET ADD) _LNL
7
_end_Example

_DComment(Talk about errors from compiled code.)

_Section(Debugging utilities)

_Comdef(`TRACE',`Special form',
`(TRACE _Fit(variable)) _yl() _Fit(undefined)')
_begin_Desc
Reassigns _Fit(variable), whose value should be a procedure,
to be a new procedure which prints information on the
_tc((DEBUG-OUTPUT)) port whenever it is called or returns a value.
_end_Desc 

_Comdef(`UNTRACE',`Special form',
`(UNTRACE _Fit(variable)) _yl() _Fit(undefined)')
_begin_Desc
If _Fit(variable) has a traced procedure as its value,
_tc(UNTRACE) restores it to its original value.
Otherwise it prints a warning and does nothing.
_end_Desc 

_Comdef(`PP',`Special form',
`(PP _Fit(procedure)) _yl() _Fit(undefined)')
_begin_Desc
Prints the definition of _Fit(procedure) on the terminal output port.
If the source code is not available, then it prints the name of a file
where it can be found.
(See also _tc(PRETTY-PRINT), page _Pageref(PRETTY-PRINT),
and _tc(WHERE-DEFINED), page _Pageref(WHERE-DEFINED).)
_end_Desc 

_Comdef(`BACKTRACE',` ',
`(BACKTRACE) _yl() _Fit(undefined)')
_label(BACKTRACE)
_begin_Desc
Prints a one-line summary describing each continuation on the stack.
_index(continuations)
See also _tc(DEBUG), page _Pageref(DEBUG).
_end_Desc 

_Section(The inspector)
_label(section:inspector)

_tc(CRAWL), also known as the _iix(inspector),
is a stack and structure inspector.
It consists of a command loop and a set of commands.
The inspector keeps track of a _Fit(current object), and a stack of
objects which have previously been current objects.  Some inspector
commands move from one object to another.

The command loop operates as follows: the current object is
_qu(summarized) (that is, printed or displayed somehow); a prompt is
printed; a line is read; and any commands on the line are executed.
One can give one or more commands on a single input line.

The current object may be any object.  If it is a continuation (stack frame),
the name of the procedure which contains the continuation's return point
is printed, and the prompt is _qu(_tc(debug:)).  Otherwise, the object
is printed (up to a maximum of one line of output),
and the prompt is _qu(_tc(crawl:)).
_index(stack)
_index(frames)
_index(continuations)

The meanings of some commands vary depending on what kind of object
the current object is,  and not all commands are
appropriate for all kinds of objects.

_Comdef(`DEBUG',` ',
`(DEBUG) _yl() _Fit(undefined)')
_label(DEBUG)
_begin_Desc
_tc((DEBUG)) enters the inspector.  The current object becomes a continuation
near the top of the stack (more precisely, the continuation to which
the value passed to _tc(RET) will be given).
_end_Desc 

_Comdef(`CRAWL',` ',
`(CRAWL _Fit(object)) _yl() _Fit(undefined)')
_begin_Desc
Enters the inspector.  The current object becomes _Fit(object).
_end_Desc 

Inspector commands:

_begin_Table
_Item(_tc(?))	_Fit(Help:) prints a list of inspector commands, with one-line
summaries.

_Item(_tc(Q))	_Fit(Quit:) exits out of the inspector, usually back to the
read-eval-print loop.

_Item(_tc(U))	_Fit(Up:) pops the stack of saved objects.  The current object is
forgotten, and the inspector moves to the previous current object.

_Item(_tc(D))	_Fit(Down:) if the current object is a continuation,
moves to the next continuation deeper in the stack, that is, to the
continuation that
was pushed prior to the current continuation.

_Item(_tc(X))   _Fit(Exhibit:) prints useful information about the current object.  The exact behavior of this varies depending on the type of the object.
For structures, all of the structure's components are displayed.
For continuations, any saved values are printed.  Integers are printed
in various radices.

For some kinds of object, such as structures and vectors, the contents
of an object will be displayed in a menu form, as a sequence of lines of
the form
_begin_Example
_exqu(_tc([_Fit(key)] _Fit(component)))
_end_Example
In this case, giving _Fit(key) as an inspector command will move to
_Fit(component), which then becomes the current object.

_Item(_Fit(integer))	_Fit(Select element:) in the case of a list or vector,
this moves to the appropriate component.  For example, if the current object
is a list, then _tc(3) as an inspector command is the same as _tc(A CADDDR):
it moves to the list's fourth element.

_Item(_Fit(selector))	_Fit(Select component:) if the current object is a structure,
typing a selector name will move to that component of the structure.

_Item(_tc(B))	_Fit(Breakpoint:) enters a read-eval-print loop.  Executing
_tc((RET)) will return back into the inspector.  The read-eval-print
loop will execute either in _tc((REPL-ENV)), or, if the current object
is a continuation or procedure created by interpreted code (i.e. code
compiled by the standard compiler, as opposed to TC), in the lexical
environment of the object.  Inside the breakpoint loop, the system
variable _tc(*OBJ*) is bound to the current object.

_Item(_tc(C))	_Fit(Crawl:) moves to a new object.  The
object is obtained by evaluating an input expression, which is
read either from the command line or in response to a prompt.
The environment of the evaluation is one appropriate to
the current object, as with the _tc(B) command.

_Item(_tc(E))	_Fit(Evaluate:) evaluates an expression.  The expression is read
and evaluated as
with the _tc(C) command.  The result of the evaluation is printed, but
the current object remains the same.

_Item(_tc(A))	_Fit(Apply:) applies a procedure to the current object, and move to the
result of that call.  An expression evaluating to the procedure is
read as with the _tc(C) command.

_Item(_tc(M))	_Fit(Macroexpand:) performs one macro expansion on the current
object, which should be a list, and the current object becomes the result
of the macro expansion.
_index(macros)

_Item(_tc(P))	_Fit(Pretty-print:) prints the current object using the pretty
printer.
If the current object is a continuation, then this command tries to
print the expression to which the value returned by the continuation
is to be supplied.

_Item(_tc(R))	_Fit(Return:) if the current object is a continuation, returns a
value to it.  The value is obtained as with the _tc(C),
_tc(E), and _tc(A) commands.

_Item(_tc(W))	_Fit(Where-defined:) prints the result of calling _tc(WHERE-DEFINED)
on the current object.

_Item(_tc(V))	_Fit(Unit:) moves to a template's or procedure's
unit.  Units are not documented, but the _tc(X) command works with them.
This command is intended primarily for the use of the _T implementors.
_end_Table

The control stack which one inspects by invoking the
inspector with _tc((DEBUG)) is a sequence of continuations, or stack
frames.  These differ from activation records in implementations of
languages such as C or Lisp in that they represent future computations
rather than past calls.  A full call history is not easily
available in tail-recursive languages such a _T and Scheme.
This is implementation-dependent, of course, and future _T
implementations may maintain call histories for debugging purposes.

Each continuation represents a control point which will make use of
the value returned to the continuation.  Usually these control points
correspond to the arguments in a call, the predicate in an _tc(IF)
or _tc(COND), or any but the last subform of a block.  For example,
when evaluating a call, a continuation is constructed for each argument
in the call, because these values will be used when the call actually
occurs.  In the example below, the continuations into _tc(FACT) were
mostly delivering values to the second argument position in the call
to _tc(*).

The three-column synopsis that the inspector prints for continuations
is the same as that printed by _tc(BACKTRACE) (page
_Pageref(BACKTRACE)).  The first column is the name of the procedure
into which control will return, or _tc((anonymous)) if no procedure
name is available (as for procedures created by anonymous top-level
_tc(LAMBDA)-expressions instead of by _tc(DEFINE)).  The second column
is the name of the source file containing the procedure, and the third
column is relevant source code, if available.

Here is a sample interaction with the inspector.
Commentary is in Roman font on the right.
The terms _qu(frame), _qu(stack frame), and _qu(continuation) are
used interchangeably.
_DComment(dorab:todo -- ick -- what a mess)
_begin_Example
_Ftt(_cGT()) (define (fact n)                      _rm(Define factorial function.) _LNL
    (cond ((= n 0) i)                   _rm(Bug: _tc(i) instead of _tc(1).) _LNL
          (else (* n (fact (- n 1)))))) _LNL
_cH()_cBo()Procedure 120 FACT _cBc() _LNL
_Ftt(_cGT()) (fact 4) _LNL

** Error: variable I is unbound         _rm(Error detected by interpreter.) _LNL
_Ftt(_cGT())_Ftt(_cGT()) (debug) _LNL
_cH()_cBo()Continuation 121_cBc()                     _rm(Current object is this frame.) _LNL
 BIND-INTERNAL         THROW            _rm(Internal to the implementation.) _LNL
debug: d                                _rm(Go down one frame.) _LNL
_cH()_cBo()Dynamic-state-transition 122_cBc()         _rm(Also internal, keep going.) _LNL
debug: d _LNL
_cH()_cBo()Continuation 123_cBc() _LNL
 FACT                  ()         I     _rm(Okay, this looks good.) _LNL
debug: d _LNL
_cH()_cBo()Continuation 124_cBc() _LNL
 FACT                  ()         (* N (FACT (- N 1))) _LNL
debug: e n                              _rm(Evaluate _tc(N) in this frame.) _LNL
1                                       _rm(Value is 1.) _LNL
debug: d _LNL
_cH()_cBo()Continuation 125_cBc() _LNL
 FACT                  ()         (* N (FACT (- N 1))) _LNL
debug: e n                              _rm(Value is 2 in this frame.) _LNL
2
debug: d d                              _rm(Go down two frames.) _LNL
_cH()_cBo()Continuation 126_cBc() _LNL
 FACT                  ()         (* N (FACT (- N 1))) _LNL
debug: d _LNL
_cH()_cBo()Continuation 127_cBc() _LNL
 READ-EVAL-PRINT-LOOP  REPL             _rm(_tc(FACT)'s caller (i.e. top level).) _LNL
debug: u u u u u                        _rm(Up five frames.) _LNL
_cH()_cBo()Continuation 123_cBc() _LNL
 FACT                  ()         I _LNL
debug: r 1                              _rm(Return the value 1 to this frame.) _LNL
24                                      _rm(Execution proceeds, and the value) _LNL
_Ftt(_cGT())                                       _rm(24 comes out.) _LNL
_end_Example

_Section(Debugging primitives)

This section describes routines which may be useful in writing debugging
aids.  Note that they are not part of the language, and therefore should be
avoided in _qu(ordinary) programs.  Relying on these routines may lead
to programs which behave differently depending on how the programs
are compiled (TC or standard compiler), or which fail to work
across releases.

_Comdef(`WHERE-DEFINED',`Operation',
`(WHERE-DEFINED _Fit(object)) _yl() _Fit(filename)')
_label(WHERE-DEFINED)
_begin_Desc
Tries to find a filename for the source file wherein
_Fit(object) (usually a procedure) is defined.
_end_Desc 

_Comdef(`IDENTIFICATION',`Operation',
`(IDENTIFICATION _Fit(object)) _yl() _Fit(symbol) _rm(or) _Fit(false)')
_begin_Desc
If appropriate, this returns a symbol naming a variable which, in some
environment, might be defined to be the _Fit(object).  If no such
identification is appropriate, _tc(IDENTIFICATION) returns false.  This
behavior is heuristic, not contractual; for no value is _tc(IDENTIFICATION)
required to return non-null.

_begin_ExampleTabular
(IDENTIFICATION CADR)                _colsep()_evalto()_colsep  CADR _LNL
(LET ((X CADR)) (IDENTIFICATION X))  _colsep()_evalto()_colsep  CADR
_end_ExampleTabular
_end_Desc 

_Comdef(`ARGSPECTRUM',`Operation',
`(ARGSPECTRUM _Fit(procedure)) _yl() _Fit(pair)')
_begin_Desc
Returns an _Fit(argspectrum) for _Fit(procedure).  An argspectrum is a pair
_wt((_Fit(min) . _Fit(max))) which describes the number of arguments that
_Fit(procedure) expects to receive when called.  _Fit(Min) is always an
integer giving the minimum number of arguments expected; _Fit(max) is
either an integer giving a maximum, or it is _tc(()), meaning that exactly
_Fit(min) arguments are required, or it is _tc(T), meaning that any number
of arguments (but at least _Fit(min)) are acceptable.  _DComment(This convention
seems pretty random; suggest something better.)
_end_Desc 

_Comdef(`DISCLOSE',` ',
`(DISCLOSE _Fit(procedure)) _yl() _Fit(list) _rm(or) _Fit(false)')
_begin_Desc
Attempts to reconstruct a source expression from which _Fit(procedure)
may have been compiled.  This may work for code compiled using the
standard compiler, but is likely to fail to work for code compiled using
_orbit().  Returns false if no source code can be obtained.
_end_Desc 

_Comdef(`GET-ENVIRONMENT',` ',
`(GET-ENVIRONMENT _Fit(procedure)) _yl() _Fit(environment) _rm(or) _Fit(false)')
_begin_Desc
Attempts to reconstruct an environment in which _Fit(procedure)
may have been loaded (or run).  This may work for code compiled using the
standard compiler, but is likely to fail to work for code compiled using
TC.  Returns false if no environment can be obtained.
_end_Desc 

_Comdef(`STRUCTURE-TYPE',` ',
`(STRUCTURE-TYPE _Fit(object)) _yl() _Fit(stype) _rm(or) _Fit(false)')
_begin_Desc
If _Fit(object) is a structure, this returns the structure type of which
it is an instance.  If _Fit(object) is not a structure, it returns false.
_tc(STRUCTURE-TYPE) is to be used with care since it may violate the
data protection otherwise provided by structure types.  That is, anyone
who is given a structure may find out about its internals.
_end_Desc 

_Comdef(`WALK-SYMBOLS',` ',
`(WALK-SYMBOLS _Fit(procedure)) _yl() _Fit(undefined)')
_begin_Desc
Calls _Fit(procedure) on every accessible symbol.
_end_Desc 

_Section(Miscellaneous)

_Comdef(`*T-VERSION-NUMBER*',` ',
`*T-VERSION-NUMBER* _yl() _Fit(integer)')
_begin_Desc
An integer which gives the version number of the currently running _T
implementation.  The integer has the form
_begin_Example
(+ (* _Fit(major-version-number) 10) _Fit(minor-version-number))
_end_Example
For example, in _Tv(3.1), _tc(*T-VERSION-NUMBER*) is _tc(31).  Knowledge
of the version number may be useful in dealing with
incompatibilities between _T releases, so that programs may
conditionally adjust their state according to the version, and thus
be able to run in both older and newer releases.
_begin_Inset(Caution:)
This may soon change to _tc(T-VERSION-NUMBER).
_end_Inset
_end_Desc 

_Comdef(`T-IMPLEMENTATION-ENV',` ',
`T-IMPLEMENTATION-ENV _yl() _Fit(locale)')
_label(T-IMPLEMENTATION-ENV)
_begin_Desc
In _Tv(2) called _tix(*T-IMPLEMENTATION-ENV*).
This environment contains variables internal to the implementation of
_T.
_begin_Example
(IMPORT T-IMPLEMENTATION-ENV _cP()_cP()PAIR-TAG)
_end_Example
_end_Desc

_Comdef(`ORBIT-ENV',` ',
`ORBIT-ENV _yl() _Fit(locale)')
_begin_Desc
In _Tv(2) called _tix(*TC-ENV*).
This environment contains variables internal to the implementation
of _orbit().
_end_Desc

_Comdef(`GC',` ',
`(GC) _yl() _Fit(undefined)')
_begin_Desc
Invokes the garbage collector._index(garbage collection)
Garbage collection is a low-level process by which the memory used by
objects which are no longer accessible is reclaimed for use by new
objects.
Ordinarily, garbage collection is invoked asynchronously as the need
arises, so explicit calls to _tc(GC) are unnecessary.

A side-effect of garbage collection is that any ports created by
_tc(OPEN) which are both inaccessible and still open, are closed.

See also section _Ref(OBJECT-HASH).
_end_Desc 

_Comdef(`GC-STATS',` ',
`(GC-STATS) _yl() _Fit(undefined)')
_begin_Desc
Prints some statistics about the most recent garbage collection.
_end_Desc 

_Comdef(`GC-NOISILY?',`Settable',
`(GC-NOISILY?) _yl() _Fit(boolean)')
_begin_Desc
Switch, initially true.  If true, then the garbage collector will print
messages when invoked.  If false, it will do its work silently.
_end_Desc 

_Comdef(`TIME',`',
`(TIME _Fsem(expression count)) _yl _Fsem(undefined)')
_begin_Desc
_wt((TIME _Fit(expression count))) computes _Fit(expression count)
times and prints the virtual time used.  The _Fit(count) argument is optional.
_end_Desc
