/*
 *
 *  p o r t . c			-- ports implementation
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *            Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 17-Feb-1993 12:27
 * Last file update: 31-Mar-1994 14:53
 *
 */

#include "stk.h"

#define OUTP(p) 	(OPORTP(p) || OSPORTP(p))
#define INP(p)  	(IPORTP(p) || ISPORTP(p))
#define F_READ  	01
#define F_WRITE 	02

/* external vars */
SCM curr_iport, curr_oport, curr_eport, eof_object;

static SCM makeport(char *name, int type, int error)
{
  SCM z;
  FILE *f;
  long flag;
  char * full_name;

  flag = no_interrupt(1);

  full_name = CHARS(internal_expand_file_name(name));

  if ((f = fopen(full_name,(type==tc_iport)? "r" : "w")) == NULL) {
    if (error) err("could not open file", NIL);
    else return ntruth;
  }
  NEWCELL(z, type);
  z->storage_as.port.f    = f;
  z->storage_as.port.name = (char *) must_malloc(strlen(full_name)+1);
  strcpy(z->storage_as.port.name, full_name);
  
  no_interrupt(flag);
  return(z);
}
  
static SCM verify_port(char *who, SCM port, int mode)
{
  char buff[100];

  if (port == UNBOUND) 
    port = (mode&F_WRITE) ? curr_oport: curr_iport; /* test write 'cause of flush */

  if (port->storage_as.port.f == NULL) {
    sprintf(buff, "%s: port is closed", who);
    err(buff, port);
  }
  if ((mode & F_READ)  && INP(port))  return port; /* not else. It can be both */
  if ((mode & F_WRITE) && OUTP(port)) return port;
Error:
  sprintf(buff, "%s: bad port", who);
  err(buff, port);
}

static void closeport(SCM port)
{
  long flag = no_interrupt(1);

  if (port->storage_as.port.f) {
    fclose(port->storage_as.port.f); 
    port->storage_as.port.f = (FILE *) NULL;
  }
  no_interrupt(flag);
}

void freeport(SCM port)
{
   long flag = no_interrupt(1);

   closeport(port);
   if (*(port->storage_as.port.name)) {
     free(port->storage_as.port.name);
     port->storage_as.port.name = "";
   }
   no_interrupt(flag);
}

void init_standard_ports(void)
{
  NEWCELL(curr_iport, tc_iport);
  curr_iport->storage_as.port.name = "*stdin*"; 
  curr_iport->storage_as.port.f    = stdin;
  gc_protect(curr_iport);

  NEWCELL(curr_oport, tc_oport);
  curr_oport->storage_as.port.name = "*stdout*"; 
  curr_oport->storage_as.port.f    = stdout;
  gc_protect(curr_oport);
 
  NEWCELL(curr_eport, tc_oport);
  curr_eport->storage_as.port.name = "*stderr*"; 
  curr_eport->storage_as.port.f    = stderr;
  gc_protect(curr_eport);
  
  NEWCELL(eof_object, tc_eof);
  gc_protect(eof_object);
}


SCM loadfile(char *fname, int err_if_absent)
{
  SCM form;
  FILE *f;
  int c;
  char *full_name;

  full_name = CHARS(internal_expand_file_name(fname));
  f         = fopen(full_name, "r");

  if (f == NULL) {
    if (err_if_absent) 
      err("load: cannot open file", makestrg(strlen(fname), fname));
    return ntruth;
  }

  /* Just read one character. Assume that file is an object if this 
   * character is a control one. Here, I don't try to see if the file magic 
   * number has a particular value, since I'm not nure that all platforms
   * use identical conventions 
   */
  c = Getc(f); Ungetc(c, f);
  if (c != EOF &&  ((iscntrl(c)&& c!= '\n') || !isascii(c)))
    load_object_file(fname);
  else {
    /* file seems not to be an object file. Try to load it as a Scheme file */
    for( ; ; ) {
      form = lreadf(f, FALSE);
      if EQ(form, eof_object) break;
      leval(form, NIL);
    }
  }
  fclose(f);
  return(err_if_absent? UNDEFINED: truth);
}

PRIMITIVE input_portp(SCM port)
{
  return IPORTP(port)? truth: ntruth;
}

PRIMITIVE output_portp(SCM port)
{
  return OPORTP(port)? truth: ntruth;
}

PRIMITIVE current_input_port(void)
{
  return curr_iport;
}

PRIMITIVE current_output_port(void)
{
  return curr_oport;
}

PRIMITIVE current_error_port(void)
{
  return curr_eport;
}

PRIMITIVE with_input_from_file(SCM string, SCM thunk)
{
  jmp_buf env, *prev_env = top_jmp_buf;
  SCM result, prev_iport = curr_iport;
  int prev_context 	 = error_context;
  int k;

  if (NSTRINGP(string))  
    err("with-input-from-file: bad string", string);
  if (NTYPEP(thunk, tc_subr_0) && NTYPEP(thunk, tc_closure))
    err("with-input-from-file: bad thunk", thunk);

  curr_iport = UNBOUND; 	/* will not be changed if opening fails */

  if ((k = setjmp(env)) == 0) {
    top_jmp_buf = &env;
    curr_iport  = makeport(CHARS(string), tc_iport, 1);
    result      = apply(thunk, NIL);
  }
  /* restore normal error jmpbuf  and current input port*/
  if (curr_iport != UNBOUND) closeport(curr_iport);
  curr_iport    = prev_iport;
  top_jmp_buf   = prev_env;
  error_context = prev_context;

  if (k) /*propagate error */ longjmp(*top_jmp_buf, k);
  return result;
}

PRIMITIVE with_output_to_file(SCM string, SCM thunk)
{
  jmp_buf env, *prev_env = top_jmp_buf;
  SCM result, prev_oport = curr_oport;
  int prev_context       = error_context;
  int k;

  if (NSTRINGP(string)) 
    err("with-output-from-file: bad string", string);
  if (NTYPEP(thunk, tc_subr_0) && NTYPEP(thunk, tc_closure))
    err("with-output-from-file: bad thunk", thunk);

  curr_oport = UNBOUND;		/* will not be changed if opening fails */

  if ((k = setjmp(env)) == 0) {
    top_jmp_buf = &env;
    curr_oport  = makeport(CHARS(string), tc_oport, 1);
    result      = apply(thunk, NIL);
  }
  /* restore normal error jmpbuf  and current output port*/
  if (curr_oport != UNBOUND) closeport(curr_oport);
  curr_oport    = prev_oport;
  top_jmp_buf   = prev_env;
  error_context = prev_context;

  if (k) /*propagate error */ longjmp(*top_jmp_buf, k);
  return result;
}


PRIMITIVE open_input_file(SCM filename)
{
  if (NSTRINGP(filename)) err("open-input-file: bad file name", filename);
  return makeport(CHARS(filename), tc_iport, TRUE);
}

PRIMITIVE open_output_file(SCM filename)
{
  if (NSTRINGP(filename)) err("open-output-file: bad file name", filename);
  return makeport(CHARS(filename), tc_oport, TRUE); 
}

PRIMITIVE close_input_port(SCM port)
{
  if (NIPORTP(port)) err("close-input-port: not an input port", port);
  closeport(port);

  return UNDEFINED;
}

PRIMITIVE close_output_port(SCM port)
{
  if (NOPORTP(port)) err("close-input-port: not an output port", port);
  closeport(port);

  return UNDEFINED;
}

PRIMITIVE lread(SCM port)
{
  port = verify_port("read", port, F_READ);
  return(lreadf(port->storage_as.port.f, FALSE));
}

PRIMITIVE read_char(SCM port)
{
  int c;

  port = verify_port("read-char", port, F_READ);
  c = Getc(port->storage_as.port.f);
  return (c == EOF) ? eof_object : makechar(c);
}

PRIMITIVE peek_char(SCM port)
{
  int c;

  port = verify_port("peek-char", port, F_READ);
  c = Getc(port->storage_as.port.f);
  Ungetc(c, port->storage_as.port.f);
  return (c == EOF) ? eof_object : makechar(c);
}

PRIMITIVE eof_objectp(SCM obj)
{
  return (obj == eof_object)? truth : ntruth;
}

PRIMITIVE char_readyp(SCM port)
{
  port = verify_port("char-ready?", port, F_READ);
  if (Eof(port->storage_as.port.f)) return truth;
  if (ISPORTP(port)) /* !eof -> */ return truth;
  else
    if (isatty(fileno(port->storage_as.port.f)))
      return truth;
    else
      return truth;
}

PRIMITIVE lwrite(SCM expr, SCM port)
{
  port = verify_port("write", port, F_WRITE);
  lprint(expr, port->storage_as.port.f, WRT_MODE);
  return UNDEFINED;
}

PRIMITIVE display(SCM expr, SCM port)
{
  port = verify_port("display", port, F_WRITE);
  lprint(expr, port->storage_as.port.f, DSP_MODE);
  return UNDEFINED;
}

PRIMITIVE newline(SCM port)
{
  port = verify_port("newline", port, F_WRITE);
  Putc('\n', port->storage_as.port.f);
  return UNDEFINED;
}

PRIMITIVE write_char(SCM c, SCM port)
{
  if (NCHARP(c)) err("write-char: not a character", c);
  port = verify_port("write-char", port, F_WRITE);
  Putc(CHAR(c), port->storage_as.port.f);
  return UNDEFINED;
}

/*
 * The name `scheme_load' is needed because of a symbol table conflict
 * in libc. This is bogus, but what do you do.
 */
PRIMITIVE scheme_load(SCM filename)
{
  if (NSTRINGP(filename)) err("load: bad file name", filename); 
  return loadfile(CHARS(filename), 1);
}


/*
 *
 * STk bonus
 *
 */

static SCM internal_format(SCM l, int error)  /* a very simple and poor format */ 
{
  SCM port, fmt;
  int len = llength(l);
  int format_in_string = 0;
  char *p;
  FILE *f;

  if (error) {
    if (len < 1) err("error: Bad list of parameters", l);
    format_in_string = 1;
    port = open_output_string();
    len -= 1;
  }
  else {
    if (len < 2) err("format: Bad list of parameters", l);
    port = CAR(l); l = CDR(l);
    len -= 2;
  }
  fmt  = CAR(l); l = CDR(l);

  if (BOOLEANP(port)){
    if (port == truth) port = curr_oport;
    else {
      format_in_string = 1;
      port= open_output_string();
    }
  }
  
  verify_port(error? "error": "format", port, F_WRITE);
  if (NSTRINGP(fmt)) err("format: bad format string", fmt);

  f = port->storage_as.port.f;

  for(p=CHARS(fmt); *p; p++) {
    if (*p == '~') {
      switch(*(++p)) {
        case 'S':
        case 's':
        case 'A':
        case 'a': if (len-- > 0) {
                    lprint(CAR(l), f, (tolower(*p) == 's')? WRT_MODE: DSP_MODE);
                    l = CDR(l);
                  }
                  else err("format: too much ~ in format string", l); 
	          continue;
        case '%': Putc('\n', f);
                  continue;
        case '~': Putc('~', f);
                  continue;
        default:  Putc('~',  f);
                  /* NO BREAK */
      }
    }
    Putc(*p, f);
  }

  if (NNULLP(l)) err("format: too few ~ in format string", l);

  return format_in_string ? get_output_string(port) : UNDEFINED;
}

PRIMITIVE format(SCM l)
{
  return internal_format(l, FALSE);
}

PRIMITIVE lerror(SCM l)
{
  err(CHARS(internal_format(l, TRUE)), NIL);
  return UNDEFINED; 	/* for compiler */
}

PRIMITIVE try_load(SCM filename)
{
  if (NSTRINGP(filename)) err("try-load: bad file name", filename); 

  return loadfile(CHARS(filename), FALSE);
}

PRIMITIVE open_file(SCM filename, SCM mode)
{
  int type;

  if (NSTRINGP(filename)) err("open-file: bad file name", filename); 
  if (NSTRINGP(mode))     err("open-file: bad mode", mode);

  type = strchr(CHARS(mode), 'r') ? tc_iport : tc_oport;
  return(makeport(CHARS(filename), type, FALSE));
}

PRIMITIVE close_port(SCM port)
{
  switch (TYPE(port)) {
    case tc_iport:
    case tc_oport: closeport(port);
    case tc_isport:
    case tc_osport: break;
    default:        err("close-port: bad port", port);
  }
  return UNDEFINED;
}

PRIMITIVE read_line(SCM port)
{
  FILE *f;
  int c, i, size = 128;
  char *buff = (char *) must_malloc(size);
  SCM res;

  port = verify_port("read-line", port, F_READ);
  f = port->storage_as.port.f;
  for (i = 0; ; i++) {
    switch (c = Getc(f)) {
      case EOF:  if (i == 0) { free(buff); return eof_object; }
      case '\n': res = makestrg(i, buff); free(buff); return res;
      default:   if (i == size) {
	           size += size / 2;
		   buff = must_realloc(buff, size);
		 }
	         buff[i] = c;
    }
  }
}

PRIMITIVE lflush(SCM port)
{
  port = verify_port("flush", port, F_WRITE|F_READ);
  fflush(port->storage_as.port.f);
  return UNDEFINED;
}

