/*
 *
 * e v a l . c				-- The evaluator
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: 23-Oct-1993 21:37
 * Last file update:  1-Jun-1994 11:41
 */

#include "stk.h"


/*
 * Eval stack
 * 
 * The eval stack is a stack of the argumenet passed toeval. This stack permits
 * to facilitate debugging  of Scheme programs. Its contents is displayed 
 * when an error occurs.
 * Note that "eval_stack" does'nt need to be protected since it contains 
 * pointers which are themselves copies of the eval C routine. Eval parameters
 * will be marked as all the objects which are in the C stack
 */
#define INITIAL_STACK_SIZE 	10
#define RETURN(x)		{tmp = (x); goto Out; }

static SCM *eval_stack	     = NULL;
static SCM *env_stack        = NULL;
static int eval_stack_ptr    = -1;
static int eval_stack_size   = 0;


void show_eval_stack(int depth)
{
  int i = 0, j;

  fprintf(stderr, "\nCurrent eval stack:\n__________________\n");
  for (j=eval_stack_ptr; j >= 0; j--) {
    fprintf(stderr, "%3d    ", i);
    lprint(luncode(eval_stack[j]), stderr, WRT_MODE);
    putc('\n', stderr);
    if (++i >= depth) {
      if (j > 0) fprintf(stderr, "...\n");
      break; 
    } 
  }
}

void reset_eval_stack(void)
{
  eval_stack_ptr = -1;
}


PRIMITIVE get_eval_stack(void)
{
  long i, j;
  SCM z = makevect(eval_stack_ptr+1, NULL);
  
  for (i=0, j=eval_stack_ptr; j >= 0; i++, j--)
    VECT(z)[i] = eval_stack[j];
  
  return vector2list(z);
}

PRIMITIVE get_env_stack(void)
{
  long i, j;
  SCM z = makevect(eval_stack_ptr+1, NULL);
  
  for (i=0, j=eval_stack_ptr; j >= 0; i++, j--)
    /* Avoid to create an environmant for each vector item */
    VECT(z)[i] = (i>0 && equal(env_stack[j], env_stack[j+1])==truth)
      			? VECT(z)[i-1]
			: makeenv(env_stack[j]);
  return vector2list(z);
}

SCM top_env_stack(void)
{
  return makeenv((eval_stack_ptr >= 0)? env_stack[eval_stack_ptr]: NIL);
}


static SCM eval_args(SCM l, SCM env)
{
  SCM result,v1,v2;

  if (NULLP(l)) return NIL;

  if (NCONSP(l)) err("bad syntax argument list", l);

  v1 = result = cons(EVALCAR(l), NIL);
  
  for(v2=CDR(l); CONSP(v2); v2=CDR(v2)) {
    v1 = CDR(v1) = cons(EVALCAR(v2),NIL);
  }
  if (NNULLP(v2)) err("bad syntax argument list", l);

  return result;
}

static SCM eval_cond(SCM *pform, SCM env)
{
  SCM l, clause, tmp, res = truth;

  for (l=*pform; NNULLP(l); l = CDR(l)) {
    clause = CAR(l);
    /* We are sure that clause is a cons here (see syntax_cond) */
    if (EQ(CAR(clause), sym_else) || (res=EVALCAR(clause)) != ntruth) {
      tmp = CDR(clause);
      if (NULLP(tmp))  SYNTAX_RETURN(res, ntruth);
      if (NCONSP(tmp)) goto Error;

      if (EQ(CAR(tmp), sym_imply)) {
	/* Clause is ((condition) => function) */
	if (llength(tmp) != 2) err("cond: malformed `=>'", tmp);
	SYNTAX_RETURN(apply(EVALCAR(CDR(tmp)), cons(res, NIL)), ntruth);
      }
      else {
	for( ; NNULLP(CDR(tmp)); tmp=CDR(tmp))
	  EVALCAR(tmp);
	SYNTAX_RETURN(CAR(tmp), truth);
      }
    }
  }
  SYNTAX_RETURN(UNDEFINED, ntruth);
Error:
  err("cond: bad clause body", clause);
}


SCM leval(SCM x, SCM env)
{
  register SCM tmp, fct;
  register int len;
  long eval_stack_save = eval_stack_ptr;
  
Top:
  /* Retain current expression in the eval stack */
  if (eval_stack_ptr >= eval_stack_size-1){
    /* make the eval stack larger */
    if (eval_stack == NULL) {
      /* First call to eval. Allocate eval_stack */
      eval_stack_size = INITIAL_STACK_SIZE;
      eval_stack      = must_malloc(eval_stack_size * sizeof(SCM));
      env_stack	      = must_malloc(eval_stack_size * sizeof(SCM));
      eval_stack_ptr  = -1;
    }
    else {
      eval_stack_size += eval_stack_size / 2;
      eval_stack       = must_realloc(eval_stack, eval_stack_size*sizeof(SCM));
      env_stack	       = must_realloc(env_stack,  eval_stack_size*sizeof(SCM));
    }
  }
  eval_stack[++eval_stack_ptr] = x;
  env_stack[eval_stack_ptr]    = env;

  /* Eval */
  switch TYPE(x) {
    case tc_symbol:
         RETURN(*varlookup(x, env, 1));
    case tc_globalvar:
	 RETURN(VCELL(VCELL(x)));
    case tc_localvar:
	 RETURN(localvalue(x, env));
    case tc_cons: {
         /* Find length of the parameter list */
	 for (len=0, tmp=CDR(x); NNULLP(tmp); len++, tmp=CDR(tmp))
	   if (NCONSP(tmp)) err("eval: malformed list", x);
	 
	 /* Evaluate the first argument of this list (without calling eval) */
	 tmp = CAR(x);
	 switch TYPE(tmp) {
           case tc_symbol:
	        fct=*varlookup(x, env, 1);
		break;
	   case tc_cons:
		fct = EVAL(tmp); break;
	   case tc_globalvar:
		fct = VCELL(VCELL(tmp)); break;
	   case tc_localvar:
		fct = localvalue(tmp, env); break;
           default:
		fct = tmp;
  	 }

	 /* apply parameters to fct */
	 tmp = CDR(x);
	 switch (TYPE(fct)) {
	   case tc_subr_0:
	        if (len == 0) RETURN(SUBR0(fct)());
		goto Error;
	   case tc_subr_1:
		if (len == 1) RETURN(SUBRF(fct)(EVALCAR(tmp)));
		goto Error;
	   case tc_subr_2:
		if (len == 2) RETURN(SUBRF(fct)(EVALCAR(tmp), 
						EVALCAR(CDR(tmp))));
		goto Error;
	   case tc_subr_3:
		if (len == 3) RETURN(SUBRF(fct)(EVALCAR(tmp),
						EVALCAR(CDR(tmp)),
						EVALCAR(CDR(CDR(tmp)))));
		goto Error;
	   case tc_subr_0_or_1:
		switch (len) {
		  case 0: RETURN(SUBRF(fct)(UNBOUND));
		  case 1: RETURN(SUBRF(fct)(EVALCAR(tmp)));
		  default: goto Error;
		}
	   case tc_subr_1_or_2:
		switch (len) {
		  case 1: RETURN(SUBRF(fct)(EVALCAR(tmp), UNBOUND));
		  case 2: RETURN(SUBRF(fct)(EVALCAR(tmp), 
					    EVALCAR(CDR(tmp))));
		  default: goto Error;
		}
	   case tc_subr_2_or_3:
		switch (len) {
		  case 2: RETURN(SUBRF(fct)(EVALCAR(tmp), 
					    EVALCAR(CDR(tmp)),
					    UNBOUND));
		  case 3: RETURN(SUBRF(fct)(EVALCAR(tmp), 
					    EVALCAR(CDR(tmp)),
					    EVALCAR(CDR(CDR(tmp)))));
		  default: goto Error;
		}
	   case tc_fsubr:
		RETURN(SUBRF(fct)(tmp, env));
	   case tc_syntax:
		if (SUBRF(fct)(&x, env) == truth) goto Top;
		RETURN(x);
	   case tc_lsubr:
		RETURN(SUBRF(fct)(eval_args(tmp, env)));
#ifdef USE_TK
	  case tc_tkcommand:
	    	RETURN(execute_Tk_lib_cmd(fct, tmp, env, 1));
#endif
	   case tc_closure:
		env = extend_env(CAR(fct->storage_as.closure.code),
				 eval_args(tmp, env),
				 fct->storage_as.closure.env,
				 x); 
		tmp = CDR(fct->storage_as.closure.code);
		/* NOBREAK */
Begin:	   case tc_begin:
		for( ; CONSP(CDR(tmp)); tmp=CDR(tmp))
		  EVALCAR(tmp);
		x = CAR(tmp);
	        goto Top;
	   case tc_cont:
		if (len == 1) throw(fct, EVALCAR(tmp));
		goto Error;
	   case tc_let:
		env = fast_extend_env(CAR(tmp), eval_args(CAR(CDR(tmp)),env), env);
		tmp = CDR(CDR(tmp));
		goto Begin;
	   case tc_letstar:
		{
		  SCM l1=CAR(tmp), l2=CAR(CDR(tmp));
		  /* Create a rib to avoid that internal def be seen as global  */
		  env = fast_extend_env(NIL, NIL, env); 
		  for ( ; NNULLP(l1); l1=CDR(l1), l2=CDR(l2))
		    env = fast_extend_env(cons(CAR(l1), NIL), 
					  cons(EVALCAR(l2), NIL), env);
		  tmp =  CDR(CDR(tmp));
		  goto Begin;
		}
	   case tc_letrec:
		{
		  SCM bindings = NIL, l1=CAR(tmp), l2=CAR(CDR(tmp));
		  
		  /* Make a binding list an extend current with it */
		  for (len=llength(l1); len; len--) bindings=cons(UNBOUND,bindings);
		  env = fast_extend_env(l1, bindings, env);

		  /* Eval init forms in the new environment */
		  for (l1 = CAR(tmp); NNULLP(l1); l1=CDR(l1), l2=CDR(l2))
		    *(varlookup(CAR(l1), env, 0)) = EVALCAR(l2);

		  /* Evaluate body */
		  tmp =  CDR(CDR(tmp));
		  goto Begin;
		}
           case tc_macro:
	        x = apply(fct->storage_as.macro.code, x);
	        goto Top;
	   case tc_quote:
		RETURN(CAR(tmp));
	   case tc_lambda:
		NEWCELL(x, tc_closure);
		x->storage_as.closure.env  = env;
		x->storage_as.closure.code = tmp;
		RETURN(x);
	   case tc_if:
		x = NEQ(EVALCAR(tmp), ntruth) ? CAR(CDR(tmp))
					      : CAR(CDR(CDR(tmp)));
		goto Top;
	   case tc_setq:
		*(varlookup(CAR(tmp), env, 0)) = EVALCAR(CDR(tmp));
#ifdef USE_TK
		Tcl_ChangeValue(PNAME(CAR(tmp)));
#endif
		RETURN(UNDEFINED);
	   case tc_cond:
		/* Don't use tmp because 
		 *     1) it's in a register 
		 *     2) we can arrive from tc_syntax 
		 */
		x = CDR(x); /* x is a "normal" var */
		if (eval_cond(&x, env) == truth) goto Top;
		RETURN(x);
	   case tc_and:
		if (!len) RETURN(truth);
		for (--len ; len; len--, tmp=CDR(tmp))
		  if (EVALCAR(tmp) == ntruth) RETURN(ntruth);
		x=CAR(tmp);
		goto Top;
	   case tc_or:
		if (!len) RETURN(ntruth);
		for (--len; len; len--, tmp=CDR(tmp))
		  if ((fct=EVALCAR(tmp)) != ntruth) RETURN(fct);
		x=CAR(tmp);
		goto Top;
           default:
		if (EXTENDEDP(fct)) {
		  if (extended_eval_parameters(fct)==truth) 
		    tmp = eval_args(tmp, env);
		  RETURN(extended_apply(fct, tmp, env));
		}
	        err("eval: bad function in ", x);
	 }
       }
     default:
       RETURN(x);
     }
Out:
  eval_stack_ptr = eval_stack_save;
  return tmp;

Error:
  err("eval: Bad number of parameters", x);
}


SCM apply(SCM fct, SCM param)
{
  switch TYPE(fct) {
    case tc_subr_0:
         return SUBR0(fct)();
    case tc_subr_1:
	 if  (llength(param) == 1)return SUBRF(fct)(CAR(param));
	 break;
    case tc_subr_2:
	 if (llength(param) == 2)
	   return SUBRF(fct)(CAR(param), CAR(CDR(param)));
	 break;
    case tc_subr_3:
	 if (llength(param) == 3)
	   return SUBRF(fct)(CAR(param), CAR(CDR(param)), CAR(CDR(CDR(param))));
	 break;
    case tc_subr_0_or_1:
	 switch (llength(param)) {
	   case 0: return SUBRF(fct)(UNBOUND);
	   case 1: return SUBRF(fct)(CAR(param));
	 }	 
    case tc_subr_1_or_2:
	 switch (llength(param)) {
	   case 1: return SUBRF(fct)(CAR(param), UNBOUND);
	   case 2: return SUBRF(fct)(CAR(param), CAR(CDR(param)));
	 }

    case tc_subr_2_or_3:
	 switch (llength(param)) {
	   case 2: return SUBRF(fct)(CAR(param), CAR(CDR(param)));
	   case 3: return SUBRF(fct)(CAR(param), CAR(CDR(param)), 
				     CAR(CDR(CDR(param))));
	 }
    case tc_fsubr:
    case tc_lsubr:
	 return SUBRF(fct)(param);
    case tc_cont:
	 if (llength(param) == 1)
	   throw(fct, CAR(param));	 
    case tc_closure: { 
         SCM env = extend_env(CAR(fct->storage_as.closure.code),
			      param,
			      fct->storage_as.closure.env,
			      fct);

	 return EVAL(cons(sym_progn, CDR(fct->storage_as.closure.code)));
       }
#ifdef USE_TK
    case tc_tkcommand:
      	return execute_Tk_lib_cmd(fct, param, NIL, 0);
#endif

    default:
	 if (EXTENDEDP(fct)) 
	   if (extended_procedurep(fct)) 
	     return extended_apply(fct, param, UNBOUND);
	 err("apply: bad procedure", fct);
  }
error:
  err("apply: bad number of arguments to apply ", cons(fct,param));
}

PRIMITIVE user_eval(SCM expr, SCM env)
{
  if (env == UNBOUND) env = makeenv(NIL);
  else 
    if (NENVP(env)) err("eval: bad environment", env);

  /* If expr is a list, make a copy of it to avoid the user to see it modified
   * (i.e. "recoded") when eval returns
   */
  if (CONSP(expr)) expr = copy_tree(expr);
  return leval(expr, env->storage_as.env.data);
}

PRIMITIVE eval_string(SCM str, SCM env)
{
  SCM result;

  if (env == UNBOUND) env = makeenv(NIL);
  else 
    if (NENVP(env)) err("eval-string: bad environment", env);

  if (NSTRINGP(str)) err("eval-string: Bad string", str);
  result = internal_eval_string(CHARS(str), 
				ERR_READ_FROM_STRING, 
				env->storage_as.env.data);
  return result == EVAL_ERROR? UNDEFINED: result;
}

