/*
 *
 * s t r . c				-- Strings management
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: ??????
 * Last file update: 21-Dec-1993 07:52
 */

#include <ctype.h>
#include "stk.h"

static int stringcomp(SCM s1, SCM s2)
{
  if (NSTRINGP(s1)) err("comparing string: bad string", s1); 
  if (NSTRINGP(s2)) err("comparing string: bad string", s2);

  return strcmp(CHARS(s1), CHARS(s2));
}

static int stringcompi(SCM s1, SCM s2)
{
  if (NSTRINGP(s1)) err("comparing string: bad string", s1);
  if (NSTRINGP(s2)) err("comparing string: bad string", s2);

  return strcmpi(CHARS(s1), CHARS(s2));
}

SCM makestrg(int len, char *init)
{
  long flag;
  SCM  z;

  flag = no_interrupt(1);
  NEWCELL(z, tc_string);

  z->storage_as.string.dim  = len;
  z->storage_as.string.data = (char *) must_malloc(len+1); 
  z->storage_as.string.data[len] = 0;

  if (init) strncpy(z->storage_as.string.data, init, len);
  no_interrupt(flag);
  return z;
}


/**** Section 6.7 ****/

PRIMITIVE stringp(SCM obj)
{
  return STRINGP(obj) ? truth: ntruth;
}

PRIMITIVE make_string(SCM len, SCM init_char)
{
  long k;
  SCM z;

  if ((k=integer_value(len)) < 0) err("make-string: bad string length", len);

  z = makestrg(k, NULL);
      
  if (init_char != UNBOUND) {
    if (CHARP(init_char)) {
      char c = CHAR(init_char);
      int j;

      for(j=0 ;j<k; j++) z->storage_as.string.data[j] = c;
    }
    else
      err("make-string: initializing char not valid", init_char); }
  return z;
}

PRIMITIVE lstring(SCM l)
{
  int j, len = llength(l);
  SCM tmp, z;

  if (len < 0) err("string: bad list", l);
  z = makestrg(len, NULL);

  /* copy element in newly allocated string */
  for (j=0; j < len; j++, l=CDR(l)) {
    tmp = CAR(l);
    if (NCHARP(tmp)) err("string: bad element", tmp);
    CHARS(z)[j] = CHAR(tmp);
  }
  return z;
}

PRIMITIVE string_length(SCM str)
{
  if (NSTRINGP(str)) err("string-length: not a string", str);
  return makeinteger(str->storage_as.string.dim);
}

PRIMITIVE string_ref(SCM str, SCM index)
{
  long k;

  if (NSTRINGP(str))    	    err("string-ref: not a string", str);
  if ((k=integer_value(index)) < 0) err("string-ref: bad index", index);

  if (k >= str->storage_as.string.dim) 
    err("string-ref: index out of bounds", index);
  return makechar(CHARS(str)[k]);
}

PRIMITIVE string_set(SCM str, SCM index, SCM value)
{
  long k;

  if (NSTRINGP(str))		    err("string-set!: not a string", str); 
  if ((k=integer_value(index)) < 0) err("string-set!: bad index", index); 
  
  if (k >= str->storage_as.string.dim) 
    err("string-set!: index out of bounds", index);
    
  if (NCHARP(value)) err("string-set!: value is not a char", value);
    
  CHARS(str)[k] = CHAR(value);
  return UNDEFINED;
}

PRIMITIVE streq   (SCM s1, SCM s2){return (stringcomp(s1,s2)==0)? truth: ntruth;}
PRIMITIVE strless (SCM s1, SCM s2){return (stringcomp(s1,s2)<0) ? truth: ntruth;}
PRIMITIVE strgt   (SCM s1, SCM s2){return (stringcomp(s1,s2)>0) ? truth: ntruth;} 
PRIMITIVE strlesse(SCM s1, SCM s2){return (stringcomp(s1,s2)<=0)? truth: ntruth;}
PRIMITIVE strgte  (SCM s1, SCM s2){return (stringcomp(s1,s2)>=0)? truth: ntruth;}

PRIMITIVE streqi   (SCM s1, SCM s2){return (stringcompi(s1,s2)==0)? truth: ntruth;}
PRIMITIVE strlessi (SCM s1, SCM s2){return (stringcompi(s1,s2)<0) ? truth: ntruth;}
PRIMITIVE strgti   (SCM s1, SCM s2){return (stringcompi(s1,s2)>0) ? truth: ntruth;} 
PRIMITIVE strlessei(SCM s1, SCM s2){return (stringcompi(s1,s2)<=0)? truth: ntruth;}
PRIMITIVE strgtei  (SCM s1, SCM s2){return (stringcompi(s1,s2)>=0)? truth: ntruth;}

PRIMITIVE substring(SCM string, SCM start, SCM end)
{
  int len, from, to;

  if (NSTRINGP(string)) err("substring: not a string", string);
  if ((from=integer_value(start))==LONG_MIN) err("substring: not an integer",start);
  if ((to=integer_value(end)) == LONG_MIN)   err("substring: not an integer", end);

  if (0 <= from && from <= to && to <= string->storage_as.string.dim)
    return makestrg(to - from, CHARS(string)+from);

  err("substring: bad bounds", cons(start, end));
}

PRIMITIVE string_append(SCM l)
{
  int i, total=0, len = llength(l);
  SCM z, tmp = l;
  char *p;
  
  /* Compute total length of resulting string */
  for (i = 0; i < len; i++) {
    if (NSTRINGP(CAR(tmp)))
      err("string-append: bad string", CAR(tmp));
    total += CAR(tmp)->storage_as.string.dim;
    tmp = CDR(tmp);
  }

  /* Allocate result */
  z = makestrg(total, NULL);
  p = CHARS(z);
  
  /* copy strings */
  for (i=0; i < len; i++) {
    strcpy(p, CHARS(CAR(l)));
    p += CAR(l)->storage_as.string.dim;
    l = CDR(l);
  }
  return z;
}

PRIMITIVE string2list(SCM str)
{
  int j, len;
  SCM tmp, z = NIL;

  if (NSTRINGP(str)) err("string->list: not a string", str);
  len = str->storage_as.string.dim;

  for (j=0; j<len; j++) {
    if (j == 0)
      tmp = z = cons(makechar(CHARS(str)[j]), NIL);
    else 
      tmp = CDR(tmp) = cons(makechar(CHARS(str)[j]), NIL);
  }
  return z;
}

PRIMITIVE list2string(SCM l)
{
  int j=0, len = llength(l);
  SCM tmp, z;

  if (len < 0) err("list->string: bad list", l);
  z = makestrg(len, NULL);
  for ( ; NNULLP(l); l=CDR(l)) {
    if (NCHARP(CAR(l))) err("list->string: not a character", CAR(l));
    CHARS(z)[j++] = CHAR(CAR(l));
  }
  return z;
}

PRIMITIVE string_copy(SCM str)
{
  if (NSTRINGP(str)) err("string-copy: not a string", str);
  return makestrg(strlen(CHARS(str)), CHARS(str));
}

PRIMITIVE string_fill(SCM str, SCM c)
{
  int len, i;
  char c_char;

  if (NSTRINGP(str)) err("string-fill: not a string", str);
  if (NCHARP(c))     err("string-fill: not a char", c);

  len = str->storage_as.string.dim;
  c_char = CHAR(c);

  for (i = 0; i < len; i++)
    CHARS(str)[i] = c_char;
  return UNDEFINED;
}


/*
 * 
 * STk bonus
 *
 */

PRIMITIVE string_findp(SCM s1, SCM s2)
{
  if (NSTRINGP(s1)) err("string-find?: bad string",s1);
  if (NSTRINGP(s2)) err("string-find?: bad string",s2);
  
  return strstr(CHARS(s2), CHARS(s1)) ? truth: ntruth;
}

PRIMITIVE string_lower(SCM s)
{
  SCM z;
  register char *p, *q;

  if (NSTRINGP(s)) err("string-lower: not a string", s);
  z = makestrg(strlen(CHARS(s)), NULL);

  for (p=CHARS(s), q=CHARS(z); *p; p++, q++) *q = tolower(*p);
  return z;
}

PRIMITIVE string_upper(SCM s)
{
  SCM z;
  register char *p, *q;

  if (NSTRINGP(s)) err("string-upper: not a string", s);
  z = makestrg(strlen(CHARS(s)), NULL);

  for (p=CHARS(s), q=CHARS(z); *p; p++, q++) *q = toupper(*p);
  return z;
}

  
