
(require 'record)
(require 'pretty-print)
(require 'stdio)
(require 'common-list-functions)
(require 'generic-write)
(define pp pretty-print)
(define (pp2 v) (pretty-print v) v)

;;;
;;; things scm is missing
;;;

(define (delq! item list)
  (cond ((null? list) '())
	((eq? item (car list)) (cdr list))
	(else
	 (set-cdr! list (delq! item (cdr list)))
	 list)))

(define (intersect a b)
  (if (null? a)
      '()
      (if (memq (car a) b)
	  (cons (car a) (intersect (cdr a) b))
	  (intersect (cdr a) b))))

;;;
;;; entry points
;;; 

(define (bg-test sexp)
  (let ((tree (bg-build-pass1-tree sexp '())))
    (bg-meta-evaluate! tree)))

(define (bg-pass1-analyze! node redo opt)
  (bg-env-analyze! node redo #f)
  (bg-triviality-analyze! node redo)
  (if opt (bg-effects-analyze! node redo))
  node)

;;;
;;; rabbit flags that were carried over to bugs
;;;
(define *testing* #f)

;;;
;;; Various truths about the environment.
;;;

(define bg-forbidden-global-assignments '(a-prim prim-top))
(define bg-primitive-functions '(a-prim a-prim2 prim-top))
(define (bg-primitive? symbol)
  (not (not (memq symbol bg-primitive-functions))))
(define (bg-known-affected func-name) #f)
(define (bg-known-effects func-name) #f)
(define (bg-okay-to-fold? func-name) #f)
(define (bg-meta-apply fn-name arglist)
  (bg-internal-error "Don't know how to meta-apply yet." fn-name))

;;; utils

; non-specific value
(define bg-nothing 'bg-nothing)

(define bg-errobj '())

(define (bg-internal-error msg obj)
  (set! bg-errobj obj)
  (error msg))

(define (bg-sexp-warning msg sexp)
  (printf "Warning! %s\\n" msg)
  (pp sexp))

(define (bg-sexp-error msg sexp)
  (printf "Error! %s\\n" msg)
  (pp sexp)
  (error "Bugs quits."))

(define bg-make-identifier-family
  (lambda (prefix)
    (let ((counter 0))
      (lambda ()
	(set! counter (1+ counter))
	(string->symbol (string-append prefix (number->string counter)))))))


 
(define bg-empty 'bg-empty)		; for uninitialized node slots.
(define (bg-empty? val) (eq? val bg-empty))


;;; This mirrors rabbit's type NODE.
(define bg-node-type
  (make-record-type "bg-node"
		    '(name		; a gensym which names the node's value
		      name-var		; a var properties node with that name
		      sexp		; source sexp (debugging)
		      env		; environment of the node (debugging)
		      free-vars		; free vars
		      set!-vars		; assigned vars
		      trivial?		; #t if trivialy compilable node
		      effects		; side effects maybe manifested here
		      affected		; side affects maybe affecting here
		      provably-effects
		      provably-affected
		      meta-evaled?	; for the meta-evaluator
		      meta-substituted?)))	; did meta-substitute?
					; rabbit also had `form', but
					; we use sub-types

(define bg-node? (record-predicate bg-node-type))
(define bg-node-id (bg-make-identifier-family "node-"))
(define bg-node-name (record-accessor bg-node-type 'name))
(define bg-node-name-var (record-accessor bg-node-type 'name-var))
(define bg-set-node-name-var! (record-modifier bg-node-type 'name-var))
; node variables should only be used as local variables
(define (bg-node-var node)
  (if (bg-empty? (bg-node-name-var node))
      (bg-set-node-name-var!
       node (bg-make-local-cvar-properties (bg-node-name node))))
  (bg-node-name-var node))
(define bg-node-sexp (record-accessor bg-node-type 'sexp))
(define bg-node-env (record-accessor bg-node-type 'env))
(define bg-node-free-vars (record-accessor bg-node-type 'free-vars))
(define bg-node-set!-vars (record-accessor bg-node-type 'set!-vars))
(define bg-node-trivial? (record-accessor bg-node-type 'trivial?))
(define bg-node-effects (record-accessor bg-node-type 'effects))
(define bg-node-affected (record-accessor bg-node-type 'affected))
(define bg-node-provably-effects
  (record-accessor bg-node-type 'provably-effects))
(define bg-node-provably-affected
  (record-accessor bg-node-type 'provably-affected))
(define bg-node-meta-evaled? (record-accessor bg-node-type 'meta-evaled?))
(define bg-node-meta-substituted?
  (record-accessor bg-node-type 'meta-substituted?))


(define (record-generic-modifier type)
  (lambda (record field-name val)
    (vector-set! record
		 (comlist:position field-name (record:rtd-vfields type)))))

(define (record-generic-accessor type)
  (lambda (record field-name)
    (vector-ref record
		(comlist:position field-name (record:rtd-vfields type)))))


(define bg-alter-node! (record-generic-modifier bg-node-type))

;;; bg-node-subtype-constructor is record-constructor for subtypes of
;;; bg-node.  It defines the rule for initializing the node fields and
;;; corresponds loosely to rabbit's NODIFY.  The resulting constructors expect
;;; a source sexp and environment list as the first two arguments.
;;; These are only for debugging.  The remaining arguments are defined
;;; in the same way as record-constructor.  By default, fields are
;;; initialized to bg-empty.

(define record-constructor-w/fill
  (lambda (rtd default explicit)
    (let* ((defaulting (comlist:set-difference (record:rtd-fields rtd)
					       explicit))
	   (defargs (map (lambda (x) default) defaulting))
	   (full-fields (append explicit defaulting))
	   (constructor (record-constructor rtd full-fields)))
      (lambda initials
	(apply constructor (append! initials defargs))))))

(define (bg-node-subtype-constructor type . fields)
  (let ((constructor
	 (record-constructor-w/fill type bg-empty
				    (append '(name
					      meta-evaled?
					      meta-substituted?
					      sexp
					      env)
					    (if (null? fields)
						'()
						(car fields))))))
    (lambda (sexp env . field-vals)
      (apply constructor (append (list (bg-node-id) 	; name
				       #f 	      	; meta-evaled?
				       #f		; meta-substituted?
				       sexp
				       env)
				 field-vals))))) 	; subtype fields
							      
(define (bg-make-node-sub-type name fields)
  (make-record-sub-type name fields bg-node-type))

(define bg-constant-type
  (bg-make-node-sub-type "bg-constant" '(value)))

(define bg-make-constant
  (bg-node-subtype-constructor bg-constant-type '(value)))

(define bg-constant? (record-predicate bg-constant-type))
(define bg-constant-value (record-accessor bg-constant-type 'value))

;;; Whereas rabbit used the property lists of unique symbols
;;; to hold global data about vars, bugs uses a record.
;;;
;;; This results in circular structures which make debugging 
;;; something of a pain (esp with lame schemes that will print them
;;; forever).  Since abstract accessors are used everywhere, we store
;;; some of the properties in a `hidden' record accessed by a closure
;;; stored in the visible record.

(define bg-hidden-var-properties-type
 (make-record-type "bg-hidden-var-properties"
		    '(binding		; the node binding this var
		      read-refs		; list of nodes refering to it
		      write-refs	; list of set! nodes assigning to it
		      fn-pos-refs	; read-refs in the function position

		      ; The following property is unused until after the 
		      ; program is converted to CPS.
		      known-function	; for constant bindings to fns.
		      )))
(define bg-make-hidden-var-properties
  (let ((construct
	 (record-constructor-w/fill bg-hidden-var-properties-type
				    '() '(known-function))))
    (lambda () (construct #f))))

(define bg-hidden-var-properties-binding
  (record-accessor bg-hidden-var-properties-type 'binding))
(define bg-hidden-var-properties-read-refs
  (record-accessor bg-hidden-var-properties-type 'read-refs))
(define bg-hidden-var-properties-fn-pos-refs
  (record-accessor bg-hidden-var-properties-type 'fn-pos-refs))
(define bg-hidden-var-properties-write-refs
  (record-accessor bg-hidden-var-properties-type 'write-refs))
(define bg-hidden-var-properties-known-function
  (record-accessor bg-hidden-var-properties-type 'known-function))
(define bg-alter-hidden-var-properties!
  (record-generic-modifier bg-hidden-var-properties-type))
(define bg-get-hidden-var-property
  (record-generic-accessor bg-hidden-var-properties-type))

(define bg-var-properties-type
 (make-record-type "bg-var-properties"
		    '(name		; symbol used in program for
					; this var
		      alpha-name	; a unique symbolic name for 
					; this var
		      global?		; well, is it?
		      hidden		; thunk that returns a hidden-var-props

		      ; set-before-strict? is used to assist known-function
		      ; folding in the case of variables bound to functions
		      ; that could only defined within the scope of the
		      ; variable.  (as w/ a letrec)
		      set-before-strict?

		      ; The following properties are unused until after the 
		      ; program is converted to CPS.
		      non-fn-referenced? ; i.e., in other than the fn position
		      offset		; position in a frame or #f
		      frame
		      )))

(define bg-var-properties? (record-predicate bg-var-properties-type))
(define bg-var-properties-name (record-accessor bg-var-properties-type 'name))
(define bg-var-properties-offset
  (record-accessor bg-var-properties-type 'offset))
(define bg-var-properties-alpha-name
  (record-accessor bg-var-properties-type 'alpha-name))
(define bg-var-properties-global?
  (record-accessor bg-var-properties-type 'global?))
(define bg-var-properties-set-before-strict?
  (record-accessor bg-var-properties-type 'set-before-strict?))
(define bg-var-properties-non-fn-referenced?
  (record-accessor bg-var-properties-type 'non-fn-referenced?))
(define (bg-var-properties-known-function prop)
  (bg-hidden-var-properties-known-function
   (bg-var-properties-hidden prop)))
(define bg-var-properties-frame
  (record-accessor bg-var-properties-type 'frame))
(define bg-var-properties-hidden
  (let ((get-thunk (record-accessor bg-var-properties-type 'hidden)))
    (lambda (props)
      ((get-thunk props)))))
(define bg-alter-var-properties!
  (record-generic-modifier bg-var-properties-type))
(define (bg-set-var-binding-node! prop node)
  (bg-alter-hidden-var-properties! (bg-var-properties-hidden prop)
				   'binding node))
(define (bg-set-var-known-function! prop node)
  (bg-alter-hidden-var-properties! (bg-var-properties-hidden prop)
				   'known-function node))

; bg-add-var-xref corresponds to rabbits addprop.  it's used to add to a 
; node to the list attached to read-refs, write-refs, or fn-pos-refs.
(define (bg-add-var-xref! properties prop xref-node)
  (let* ((hidden-props (bg-var-properties-hidden properties))
	 (curval (bg-get-hidden-var-property hidden-props prop))
	 (newval (if (memq xref-node curval) curval (cons xref-node curval))))
    (bg-alter-hidden-var-properties! hidden-props prop newval)))

; This undoes bg-add-var-xref!.  It corresponds to DELPROP in rabbit.

(define (bg-remove-var-xref! properties prop xref-node)
  (let* ((hidden-props (bg-var-properties-hidden properties))
	 (curval (bg-get-hidden-var-property hidden-props prop))
	 (newval (delq! xref-node curval)))
    (bg-alter-hidden-var-properties! hidden-props prop newval)))

(define (bg-var-properties-binding props)
  (bg-hidden-var-properties-binding (bg-var-properties-hidden props)))

(define (bg-var-properties-read-refs props)
  (bg-hidden-var-properties-read-refs (bg-var-properties-hidden props)))
(define (bg-var-properties-fn-pos-refs props)
  (bg-hidden-var-properties-fn-pos-refs (bg-var-properties-hidden props)))
(define (bg-var-properties-write-refs props)
  (bg-hidden-var-properties-write-refs (bg-var-properties-hidden props)))

(define bg-var-properties-constructor
  (let ((construct (record-constructor bg-var-properties-type)))
    (lambda (name alpha-name global?)
      (let* ((hidden (bg-make-hidden-var-properties))
	     (thunk (lambda () hidden)))
	(construct name alpha-name global? thunk #f #f #f #f)))))

(define bg-make-global-var-properties
  (let ((prop-assoc '()))
    (lambda (name)
      (let ((cached (assoc name prop-assoc)))
	(if cached
	    (cdr cached)
	    (let ((prop (bg-var-properties-constructor name name #t)))
	      (set! prop-assoc (cons (cons name prop) prop-assoc))
	      prop))))))

(define bg-var-name (bg-make-identifier-family "var-"))

(define bg-make-local-var-properties
  (lambda (user-name)
    (bg-var-properties-constructor user-name (bg-var-name) #f)))

(define bg-var-ref-type
  (bg-make-node-sub-type "bg-var-ref"
			'(properties)))	; see above

(define bg-make-var-ref
  (let ((constructor (bg-node-subtype-constructor bg-var-ref-type
						  '(properties))))
    (lambda (sexp env props)
      (let ((node (constructor sexp env props)))
	(bg-add-var-xref! props 'read-refs node)
	node))))

(define bg-var-ref? (record-predicate bg-var-ref-type))
(define bg-var-ref-properties
  (record-accessor bg-var-ref-type 'properties))

(define (bg-var-ref-name vr)
  (bg-var-properties-name (bg-var-ref-properties vr)))
(define (bg-var-ref-alpha-name vr)
  (bg-var-properties-alpha-name (bg-var-ref-properties vr)))
(define (bg-var-ref-to-global? vr)
  (bg-var-properties-global? (bg-var-ref-properties vr)))


(define bg-if-type
  (bg-make-node-sub-type "bg-if"
			'(pred con alt)))

(define bg-make-if
  (bg-node-subtype-constructor bg-if-type '(pred con alt)))
(define bg-if? (record-predicate bg-if-type))
(define bg-if-pred (record-accessor bg-if-type 'pred))
(define bg-if-con (record-accessor bg-if-type 'con))
(define bg-if-alt (record-accessor bg-if-type 'alt))
(define bg-alter-if! (record-generic-modifier bg-if-type))

(define bg-set!-type
  (bg-make-node-sub-type "bg-set!"
			'(var	; var-properties record
			  body)))

(define bg-make-set!-
  (let ((constructor (bg-node-subtype-constructor bg-set!-type
						  '(var body))))
    (lambda (sexp env var-props body)
      (let ((node (constructor sexp env var-props body)))
	(bg-add-var-xref!  var-props 'write-refs node)
	node))))

(define bg-set!-? (record-predicate bg-set!-type))
(define bg-set!-var (record-accessor bg-set!-type 'var))
(define bg-set!-body (record-accessor bg-set!-type 'body))
(define (bg-set!-global? node)
  (bg-var-properties-global? (bg-set!-var node)))
(define bg-alter-set! (record-generic-modifier bg-set!-type))

(define bg-lambda-type
  (bg-make-node-sub-type "bg-lambda"
			'(vars-bound	; var-properties records for bound vars
			  body)))
(define bg-make-lambda
  (let ((constructor (bg-node-subtype-constructor bg-lambda-type
						  '(vars-bound body))))
    (lambda (sexp env bound body)
      (let ((node (constructor sexp env bound body)))
	(map (lambda (prop) (bg-set-var-binding-node! prop node))
	     bound)
	node))))

(define bg-lambda? (record-predicate bg-lambda-type))
(define bg-lambda-vars-bound (record-accessor bg-lambda-type 'vars-bound))
(define bg-lambda-body (record-accessor bg-lambda-type 'body))
(define bg-alter-lambda! (record-generic-modifier bg-lambda-type))

;;; rabbit had a node for catch (roughly, a call/cc that binds a variable).
;;; w/ bugs, call-with-current-continuation is hand compiled and elsewhere
;;; works like a normal function.  if a may-return-twice side effect is added,
;;; then it may be desirable to add some call/cc knowledge to bugs.

;;; fixme: is a labels node needed (letrec? can be done with macros...)

(define bg-combination-type
  (bg-make-node-sub-type "bg-combination"
			'(args		; subforms
			  warn?)))	; eval-order warning justified?

(define bg-make-combination
  (let ((constructor
	 (bg-node-subtype-constructor bg-combination-type '(args warn?))))
    (lambda (sexp env args)
      (constructor sexp env args #f))))
(define bg-combination? (record-predicate bg-combination-type))
(define bg-combination-args (record-accessor bg-combination-type 'args))
(define bg-combination-warn? (record-accessor bg-combination-type 'warn?))

;; Parsing sexps:
;; These are specificly for use in bg.  They have preconditions that
;; are rather arbitrary (e.g. bg-combination-exp? assumes that
;; bg-if-exp? is false).

; var references
(define (bg-var-exp? exp) (symbol? exp))
(define (bg-var-exp-name exp) exp)

; quote and self-evaluating
(define (bg-quote-exp? exp)
  (or (and (list? exp) (eq? (car exp) 'quote))
      (number? exp)
      (string? exp)
      (boolean? exp)
      (char? exp)))

(define (bg-quote-exp-quoted exp)
  (if (list? exp)
      (if (not (= 2 (length exp)))
	  (bg-sexp-warning
	   "Too many arguments to QUOTE (extras ignored)." exp)
	  (cadr exp))
      exp))				; alt must be self-evaluating

; ifs
(define (bg-if-exp? expression)
  (if (and (list? expression)
	   (eq? (car expression) 'if))
      (let ((len (length expression)))
	(if (or (< len 3) (> len 4))
	    (bg-sexp-error "Ill formed if." expression))
	#t)
      #f))

(define (bg-if-exp-pred expression) (cadr expression))
(define (bg-if-exp-con expression) (caddr expression))
(define (bg-if-exp-alt expression)
  (if (= (length expression) 4)
      (cadddr expression)
      #f))

; assignment
(define (bg-set!-exp? expression)
  (if (and (list? expression)
	   (eq? (car expression) 'set!))
      (if (not (and (= (length expression) 3)
		    (symbol? (cadr expression))))
	  (bg-sexp-error "Ill formed set!." expression)
	  (if (memq (cadr expression) bg-forbidden-global-assignments)
	      (bg-sexp-error "Illegal assignment to reserved global."
			     expression)
	      #t))
      #f))

(define (bg-set!-exp-var expression)
  (cadr expression))

(define (bg-set!-exp-expression expression)
  (caddr expression))


; lambda
(define (bg-var-set? parameter-list)
  (define (non-syms-or-duplicates? list)
    (and (not (null? list))
	 (or (not (symbol? (car list)))
	     (memq (car list) (cdr list))
	     (non-syms-or-duplicates? list))))
  (and (list? parameter-list)
       (non-syms-or-duplicates? parameter-list)))

(define (bg-lambda-exp? expression)
  (if (and (list? expression)
	   (eq? (car expression) 'lambda))
      ; fixme: will have to handle other kinds of arglist
      (if (and (not (= (length expression) 3))
	       (bg-var-set? expression))
	  (bg-sexp-error "Ill formed lambda." expression)
	  #t)
      #f))

; fixme: handle other kinds of formals
(define (bg-lambda-exp-variables expression) (cadr expression))
(define (bg-lambda-exp-body expression) (caddr expression))

(define (bg-combination-exp? expression)
  (list? expression))

(define (bg-combination-exp-args exp) exp)
;;; 
;;; bg-build-tree converts an sexp into a tree of nodes.
;;; In the process it alpha-renames variables to unique var-properties records.
;;; It corresponds to the rabbit function ALPHATIZE.
;;;

(define (bg-resolve-variable name translation-alist)
  (let ((binding (assoc name translation-alist)))
    (if binding
	(cdr binding)
	(bg-make-global-var-properties name))))

(define (bg-build-pass1-tree sexp environment)
  (cond
   ; variable references
   ((bg-var-exp? sexp)
	 (bg-make-var-ref sexp environment
				(bg-resolve-variable (bg-var-exp-name sexp)
						     environment)))
   ; self evaluating forms and QUOTE
   ((bg-quote-exp? sexp)
    (bg-make-constant sexp environment (bg-quote-exp-quoted sexp)))

   ; ifs
   ((bg-if-exp? sexp)
    (bg-make-if
     sexp environment
     (bg-build-pass1-tree (bg-if-exp-pred sexp) environment)
     (bg-build-pass1-tree (bg-if-exp-con sexp) environment)
     (bg-build-pass1-tree (bg-if-exp-alt sexp) environment)))

   ; assignement
   ((bg-set!-exp? sexp)
    (bg-make-set!-
     sexp environment
     (bg-resolve-variable (bg-set!-exp-var sexp) environment)
     (bg-build-pass1-tree (bg-set!-exp-expression sexp) environment)))


   ; lambda
   ;   i think this will be the only construct that introduces bindings
   ;   (unlike rabbit which also had labels, catch, and block)
   ((bg-lambda-exp? sexp)
    (let* ((formals (bg-lambda-exp-variables sexp))
	   (var-props (map bg-make-local-var-properties formals))
	   (enclosed-env (append (map cons formals var-props) environment)))
      (bg-make-lambda
       sexp environment
       var-props
       (bg-build-pass1-tree (bg-lambda-exp-body sexp) enclosed-env))))

   ; combinations
   ((bg-combination-exp? sexp)
    (bg-make-combination sexp environment
			 (map
			  (lambda (exp) (bg-build-pass1-tree exp environment))
			  (bg-combination-exp-args sexp))))

   ; Later stages of the compiler use build-pass1-tree as a convenient
   ; constructor for nodes (i.e., for a complicated node structure they build
   ; a sexp and pass that through build-pass1-tree).  For convenience, we allow
   ; that code to use node structures as subexpressions in sexps.
   ((bg-node? sexp) sexp)

   (else (bg-sexp-error "Perplexing sexp." sexp))

   ))

(define (bg-build-global-tree sexp)
  (bg-build-pass1-tree sexp '()))

;;;
;;; As a handy debugging aid, this function converts a node tree back into an
;;; sexp (which can then be printed).  There is a flag that determines whether
;;; to use the unique names or user names for variables carried over from the
;;; original program.  This corresponds to SEXPRFY in rabbit.
;;;
;;; If an optional list of fields is provided, then each part of the
;;; sexp is turned into a pair ( VALS . SEXP ).  VALS is a list of pairs
;;; of field names and values.

(define (bg-node->sexp node user-names? . fields-args)
  (define (bg-self-evaluating? value)
    (or (number? value)
	(string? value)
	(boolean? value)
	(char? value)))

  (let ((fields (if (null? fields-args) '() (car fields-args))))
    (define (wrap node sexp)
      (if (null? fields)
	  sexp
	  (let* ((type (record-type-descriptor node))
		 (type-fields (record-type-field-names type))
		 (access (record-generic-accessor type))
		 (valid-fields
		  (remove-if-not (lambda (x) (memq x type-fields)) fields)))
	    (cons (map (lambda (key)
			 (cons key (access node key)))
		       valid-fields)
		  sexp))))

    (define (props->name var-props)
      (if user-names?
	  (bg-var-properties-name var-props)
	  (bg-var-properties-alpha-name var-props)))

    (wrap node 
	  (cond
	   ((bg-constant? node)
	    (if (bg-self-evaluating? (bg-constant-value node))
		(bg-constant-value node)
		(list 'quote (bg-constant-value node))))
	   
	   ((bg-var-ref? node)
	    (props->name (bg-var-ref-properties node)))
	   
	   ((bg-if? node)
	    (list 'if (bg-node->sexp (bg-if-pred node) user-names? fields)
		  (bg-node->sexp (bg-if-con node) user-names? fields)
		  (bg-node->sexp (bg-if-alt node) user-names? fields)))
	   
	   ((bg-set!-? node)
	    (list 'set!
		  (props->name (bg-set!-var node))
		  (bg-node->sexp (bg-set!-body node) user-names? fields)))
	   
	   ((bg-lambda? node)
	    (list 'lambda
		  (map props->name (bg-lambda-vars-bound node))
		  (bg-node->sexp (bg-lambda-body node) user-names? fields)))
	   
	   ((bg-combination? node)
	    (map (lambda (sexp) (bg-node->sexp sexp user-names? fields))
		 (bg-combination-args node)))
	   
	   (else
	    (bg-internal-error "bg-node->sexp: curious node." node))))))



;;; Environment analysis.
;;;
;;; pass1-analysis and optimization use the next several pages of functions.
;;; They are coded in slavish imitation of the original rabbit and the rabbit
;;; thesis would be invaluable to understanding them.
;;;

;;; For nodes encountered we fill in:
;;;	free-vars
;;;	set!-vars
;;;
;;; The redo-this flag is used from within the optimizer after it makes
;;; structural changes to already-analyzed parts of the program. 
;;;
;;; the fn-pos? var is #t if the node being analyzed occurs in the fn position.
;;; 
;;;
;;; The rabbit version had to add various properties to variable references.
;;; Bug's constructors for var-refs, set! nodes, and lambdas do this at
;;; construction time.
;;;     
(define bg-redo-this #t)
(define bg-redo-all 'bg-redo-all)

(define (bg-env-analyze! node redo-this fn-pos?)
  (if (or redo-this (bg-empty? (bg-node-free-vars node)))
      (let ((redo-kids (if (eq? redo-this bg-redo-all) bg-redo-all #f)))
	(cond 
	 ((bg-constant? node) (bg-alter-node! node
					      'free-vars '()
					      'set!-vars '()))

	 ((bg-var-ref? node)
	  (bg-alter-node! node
			  'free-vars (if (bg-var-ref-to-global? node)
					 '()
					 (list (bg-var-ref-properties
						node)))
			 'set!-vars '())
	  (bg-add-var-xref! (bg-var-ref-properties node)
			    'fn-pos-refs node))

	 ((bg-lambda? node)
	  (let ((body (bg-lambda-body node)))
	    (bg-env-analyze! body redo-kids #f)
	    (bg-alter-node! node
			    'free-vars
			    (set-difference (bg-node-free-vars body)
					    (bg-lambda-vars-bound node))
			    'set!-vars
			    (set-difference (bg-node-set!-vars body)
					    (bg-lambda-vars-bound node)))))

	 ((bg-if? node)
	  (let ((pred (bg-if-pred node))
		(con (bg-if-con node))
		(alt (bg-if-alt node)))
	    (bg-env-analyze! pred redo-kids #f)
	    (bg-env-analyze! con redo-kids #f)
	    (bg-env-analyze! alt redo-kids #f)
	    (bg-alter-node! node
			    'free-vars (union (bg-node-free-vars pred)
					 (union (bg-node-free-vars con)
						(bg-node-free-vars alt)))
			    'set!-vars (union (bg-node-set!-vars pred)
					  (union (bg-node-set!-vars con)
						 (bg-node-set!-vars alt))))))

	 ((bg-set!-? node)
	  (let ((body (bg-set!-body node))
		(var (bg-set!-var node)))
	    (bg-env-analyze! body redo-kids #f)
	    (if (bg-var-properties-global? var)
		(bg-alter-node! node
				'free-vars (bg-node-free-vars body)
				'set!-vars (bg-node-set!-vars body))
		(bg-alter-node! node
				'free-vars
				(adjoin var (bg-node-free-vars body))
				'set!-vars
				(adjoin var (bg-node-set!-vars body))))))

	 ((bg-combination? node)
	  (let ((args (bg-combination-args node)))
	    (bg-env-analyze! (car args) redo-kids #t)
	    (map (lambda (x) (bg-env-analyze! x redo-kids #f)) (cdr args))
	    (do ((a args (cdr a))
		 (r '() (union r (bg-node-free-vars (car a))))
		 (s '() (union s (bg-node-set!-vars (car a)))))
		((null? a)
		 (bg-alter-node! node
				 'free-vars  r
				 'set!-vars  s))))))))
  bg-nothing)


;;; triviality analysis

;;; for nodes encountered we fill in:
;;;	trivial?

;;; A combination is trivial iff all arguments are trivial, and
;;; the function can be proved to be trivial.  We assume closures
;;; to be non-trivial in this context, so that the convert function
;;; will be forced to examine them.

(define (bg-triviality-analyze! node redo-this)
  (if (or redo-this (bg-empty? (bg-node-trivial? node)))
      (let ((redo (if (eq? redo-this bg-redo-all) bg-redo-all #f)))
	(cond
	 ((bg-constant? node)
	  (bg-alter-node! node 'trivial? #t))
	 ((bg-var-ref? node)
	  (bg-alter-node! node 'trivial? #t))
	 ((bg-lambda? node)
	  (bg-triviality-analyze! (bg-lambda-body node) redo)
	  (bg-alter-node! node 'trivial? #f))
	 ((bg-if? node)
	  (bg-triviality-analyze! (bg-if-pred node) redo)
	  (bg-triviality-analyze! (bg-if-con node) redo)
	  (bg-triviality-analyze! (bg-if-alt node) redo)
	  (bg-alter-node! node
			  'trivial?
			  (and (bg-node-trivial? (bg-if-pred node))
			       (bg-node-trivial? (bg-if-con node))
			       (bg-node-trivial? (bg-if-alt node)))))
	 ((bg-set!-? node)
	  (bg-triviality-analyze! (bg-set!-body node) redo)
	  (bg-alter-node! node
			  'trivial? (bg-node-trivial? (bg-set!-body node))))
	 ((bg-combination? node)
	  (let ((args (bg-combination-args node)))
	    (bg-triviality-analyze! (car args) redo)
	    (do ((a (cdr args) (cdr a))
		 (sw #t (and sw (bg-node-trivial? (car a)))))
		((null? a)
		 (bg-alter-node! node
				 'trivial?
				 (and sw (bg-trivial-fn? (car args)))))
	      (bg-triviality-analyze! (car a) redo)))))))
  bg-nothing)

(define (bg-trivial-fn? fn)
  (or (and (bg-var-ref? fn)
	   (bg-primitive? (bg-var-ref-name fn)))
      (and (bg-lambda? fn)
	   (bg-node-trivial? (bg-lambda-body fn)))))




;;; side-effects analysis
;;; for nodes encountered we fill in:  
;;;   effects, affected, provably-effects, provably-affected
;;;
;;; a set of side effects may be either bg-no-effect or bg-any-effect, 
;;; or a set.

(define bg-no-effect 'bg-no-effect)
(define bg-any-effect 'bg-any-effect)

; side effect names:
; global-set!
; set!
; file
; cons
; 


(define (bg-effects-analyze! node redo-this)
  (if (or redo-this (bg-empty? (bg-node-effects node)))
      (let ((redo (if (eq? redo-this bg-redo-all) bg-redo-all #f)))
	(cond
	 ((bg-constant? node)
	  (bg-alter-node! node
			  'effects 'bg-no-effect
			  'affected 'bg-no-effect
			  'provably-effects 'bg-no-effect
			  'provably-affected 'bg-no-effect))

	 ((bg-var-ref? node)
	  (let ((a (cond ((bg-var-ref-to-global? node) '(global-set!))
			 ((null? (bg-var-properties-write-refs
				  (bg-var-ref-properties node)))
			  bg-no-effect)
			 (else '(set!)))))
	    (bg-alter-node! node
			    'effects bg-no-effect
			    'affected a
			    'provably-effects bg-no-effect
			    'provably-affected a)))

	 ((bg-lambda? node)
	  (bg-effects-analyze! (bg-lambda-body node) redo)
	  (bg-alter-node! node
			  'effects '(cons)
			  'affected '()
			  'provably-effects '(cons)
			  'provably-affected '()))

	 ((bg-if? node)
	  (bg-effects-analyze-if! node redo))

	 ((bg-set!-? node)
	  (bg-effects-analyze! (bg-set!-body node) redo)
	  (let ((set!-effects (if (bg-set!-global? node)
				  '(global-set!)
				  '(set!))))
	    (bg-alter-node! node
			    'effects
			    (bg-effects-union
			     set!-effects
			     (bg-node-effects (bg-set!-body node)))

			    'affected
			    (bg-node-affected (bg-set!-body node))

			    'provably-effects
			    (bg-effects-union
			     set!-effects
			     (bg-node-provably-effects (bg-set!-body node)))

			    'provably-affected
			    (bg-node-provably-affected (bg-set!-body node)))))
	 
	 ((bg-combination? node)
	  (bg-effects-analyze-combination! node redo))))))

(define (bg-effects-union a b)
  (cond ((eq? a bg-no-effect) b)
	((eq? b bg-no-effect) a)
	((eq? a bg-any-effect) bg-any-effect)
	((eq? b bg-any-effect) bg-any-effect)
	(else (union a b))))

(define (bg-effects-analyze-if! node redo)
  (bg-effects-analyze! (bg-if-pred node) redo)
  (bg-effects-analyze! (bg-if-con node) redo)
  (bg-effects-analyze! (bg-if-alt node) redo)

  (bg-alter-node! node
		  'effects
		  (bg-effects-union
		   (bg-node-effects (bg-if-pred node))
		   (bg-effects-union
		    (bg-node-effects (bg-if-con node))
		    (bg-node-effects (bg-if-alt node))))

		  'affected
		  (bg-effects-union
		   (bg-node-affected (bg-if-pred node))
		   (bg-effects-union
		    (bg-node-affected (bg-if-con node))
		    (bg-node-affected (bg-if-alt node))))

		  'provably-effects
		  (bg-effects-union
		   (bg-node-provably-effects (bg-if-pred node))
		   (bg-effects-union
		    (bg-node-provably-effects (bg-if-con node))
		    (bg-node-provably-effects (bg-if-alt node))))

		  'provably-affected
		  (bg-effects-union
		   (bg-node-provably-affected (bg-if-pred node))
		   (bg-effects-union
		    (bg-node-provably-affected (bg-if-con node))
		    (bg-node-provably-affected (bg-if-alt node))))))


(define *bg-check-eval-order* #f)

(define (bg-effects-analyze-combination! node redo)
  (let ((args (bg-combination-args node)))
    (bg-effects-analyze! (car args) redo)
    (do ((a (cdr args) (cdr a))
	 (ef bg-no-effect (bg-effects-union ef (bg-node-effects (car a))))
	 (af bg-no-effect (bg-effects-union af (bg-node-affected (car a))))
	 (pef bg-no-effect
	      (bg-effects-union pef (bg-node-provably-effects (car a))))
	 (paf bg-no-effect
	      (bg-effects-union paf (bg-node-provably-affected (car a)))))

	((null? a)
	 (if *bg-check-eval-order*
	     (bg-check-eval-order node))
	 (cond ((bg-var-ref? (car args))
		(let* ((v (bg-var-ref-properties (car args)))
		       (vet (bg-known-effects (bg-var-properties-name v)))
		       (vat (bg-known-affected (bg-var-properties-name v)))
		       (ve (or vet '()))
		       (va (or vat '())))
		  (bg-alter-node! node
				  'effects
				  (if vet
				      (bg-effects-union ef ve)
				      bg-any-effect)

				  'affected
				  (if vat
				      (bg-effects-union af va)
				      bg-any-effect)

				  'provably-effects
				  (bg-effects-union pef ve)

				  'provably-affected
				  (bg-effects-union paf va))))

	       ((bg-lambda? (car args))
		(let ((b (bg-lambda-body (car args))))
		  (bg-alter-node! node
				  'effects
				  (bg-effects-union ef (bg-node-effects b))

				  'affected
				  (bg-effects-union af (bg-node-affected b))

				  'provably-effects
				  (bg-effects-union
				   pef (bg-node-provably-effects b))

				  'provably-affected
				  (bg-effects-union
				   paf (bg-node-provably-affected b)))))

	       (else
		(bg-alter-node! node
				'effects bg-any-effect
				
				'affected bg-any-effect
				
				'provably-effects
				(bg-effects-union
				 pef (bg-node-provably-effects (car args)))
				
				'provably-affected
				(bg-effects-union
				 paf
				 (bg-node-provably-affected (car args)))))))
      (bg-effects-analyze! (car a) redo))))

(define (bg-check-eval-order node)
  (bg-internal-error
   "bg-check-eval-order: eval order dependency checking not implemented."
   node))

;;; This routine is used to undo any pass 1 analysis on a node.

(define (bg-erase-node! node) (bg-erase-nodes! node #f))
(define (bg-erase-all-nodes! node) (bg-erase-nodes! node #t))

(define (bg-erase-nodes! node all?)
  (or (bg-node? node)
      (bg-internal-error "cannot erase a non-node" node))
  (cond
   ((bg-constant? node))

   ((bg-var-ref? node)
    (bg-remove-var-xref! (bg-var-ref-properties node) 'read-refs node)
    (bg-remove-var-xref! (bg-var-ref-properties node) 'fn-pos-refs node))

   ((bg-lambda? node)
    (if all? (bg-erase-all-nodes! (bg-lambda-body node)))
    (if (not *testing*)
	(map (lambda (v)
	       (bg-alter-hidden-var-properties! (bg-var-properties-hidden v)
						'binding bg-empty))
	     (bg-lambda-vars-bound node))))

   ((bg-if? node)
    (if all?
	(begin
	  (bg-erase-all-nodes! (bg-if-pred node))
	  (bg-erase-all-nodes! (bg-if-con node))
	  (bg-erase-all-nodes! (bg-if-alt node)))))

   ((bg-set!-? node)
    (if all? (bg-erase-all-nodes! (bg-set!-body node)))
    (bg-remove-var-xref! (bg-set!-var node) 'write-refs  node))
   
   ((bg-combination? node)
    (if all? (map (lambda (a) (bg-erase-all-nodes! a))
		  (bg-combination-args node))))))


;;; The value of meta-evaluate is the (possibly new) node resulting from the
;;; given one. 

(define *fudge* t)			; switch to control meta-if-fudge
(define *dead-count* 0)			; count of dead-code eliminations

(define (bg-meta-evaluate! node)
  (if (bg-node-meta-evaled? node)
      node
      (cond
       ((bg-constant? node)
	(bg-reanalyze1! node)
	(bg-alter-node! node 'meta-evaled? #t)
	node)

       ((bg-var-ref? node)
	(bg-reanalyze1! node)
	(bg-alter-node! node 'meta-evaled? #t)
	node)

       ((bg-lambda? node)
	(bg-alter-lambda! node 'body (bg-meta-evaluate! (bg-lambda-body node)))
	(bg-reanalyze1! node)
	(bg-alter-node! node 'meta-evaled? #t)
	node)

       ((bg-if? node)
	(bg-alter-if! node
		      'pred (bg-meta-evaluate! (bg-if-pred node))
		      'con (bg-meta-evaluate! (bg-if-con node))
		      'alt (bg-meta-evaluate! (bg-if-alt node)))

	(if (and *fudge* (bg-if? (bg-if-pred node)))
	    (bg-meta-if-fudge! node)
	    (if (bg-constant? (bg-if-pred node))
		(let ((con (bg-if-con node))
		      (alt (bg-if-alt node))
		      (val (bg-constant-value (bg-if-pred node))))
		  (bg-erase-node! node)
		  (bg-erase-all-nodes! (bg-if-pred node))
		  (set! *dead-count* (1+ *dead-count*))
		  (if val
		      (begin (bg-erase-all-nodes! alt) con)
		      (begin (bg-erase-all-nodes! con) alt)))
		(begin (bg-reanalyze1! node)
		       (bg-alter-node! node 'meta-evaled? #t)
		       node))))

       ((bg-set!-? node)
	(bg-alter-set! node 'body (bg-meta-evaluate! (bg-set!-body node)))
	(bg-reanalyze1! node)
	(bg-alter-node! node 'meta-evaled? #t)
	node)

       ((bg-combination? node)
	(let ((fn (car (bg-combination-args node))))
	  (cond ((and (bg-var-ref? fn)
		      (bg-trivial-fn? fn))
		 (bg-meta-combination-trivfn! node))

		((bg-lambda? fn)
		 (bg-meta-combination-lambda! node))

		(else (do ((a (bg-combination-args node) (cdr a)))
			  ((null? a))
			(set-car! a (bg-meta-evaluate! (car a))))
		      (bg-reanalyze1! node)
		      (bg-alter-node! node 'meta-evaled? #t)
		      node)))))))

;;; transform (if (if a b c) d e) into:
;;;    ((lambda (d1 e1)
;;;	        (if a (if b (d1) (e1)) (if c (d1) (e1))))
;;;     (lambda () d)
;;;     (lambda () e))

(define *fudge-count* 0)		; count of if-fudges

(define bg-meta-con-id (bg-make-identifier-family "meta-con-"))
(define bg-meta-alt-id (bg-make-identifier-family "meta-alt-"))

(define (bg-meta-if-fudge! node)
  (let ((pred (bg-if-pred node)))
    (let ((n (bg-build-pass1-tree
	      (let ((convar (bg-meta-con-id))
		    (altvar (bg-meta-alt-id)))
		`((lambda (,convar ,altvar)
		    (if ,(bg-if-pred pred)
			(if ,(bg-if-con pred)
			    (,convar)
			    (,altvar))
			(if ,(bg-if-alt pred)
			    (,convar)
			    (,altvar))))
		  (lambda () ,(bg-if-con node))
		  (lambda () ,(bg-if-alt node))))
	      (bg-node-env node))))	;doesn't matter
      (bg-erase-node! node)
      (bg-erase-node! (bg-if-pred node))
      (set! *fudge-count* (1+ *fudge-count*))
      (bg-meta-evaluate! n))))

;;; reduce a combination with a side-effect-less trivial
;;; function and constant arguments to a constant.

(define *fold-count* 0)			; count of constant foldings

(define (bg-meta-combination-trivfn! node)
  (let ((args (bg-combination-args node)))
    (set-car! args (bg-meta-evaluate! (car args)))
    (do ((a (cdr args) (cdr a))
	 (const? (let ((fnname (bg-var-ref-name (car args))))
		   (or (and (eq? (bg-known-effects fnname) bg-no-effect)
			    (eq? (bg-known-affected fnname) bg-no-effect))
		       (bg-okay-to-fold? fnname)))
		 (and const? (bg-constant? (car a)))))
	((null? a)
	 (if const?
	     (let ((val
		    (bg-meta-apply (bg-var-ref-name (car args))
				   (map (lambda (x)
					  (bg-constant-value x))
					(cdr args)))))
		  (bg-erase-all-nodes! node)
		  (set! *fold-count* (1+ *fold-count*))
		  (bg-meta-evaluate!
		   (bg-build-pass1-tree `(quote ,val) '())))

	     (begin (bg-reanalyze1! node)
		    (bg-alter-node! node 'meta-evaled? #t)
		    node)))
      
      (set-car! a (bg-meta-evaluate! (car a))))))

(define *flush-args* #t)		;switch to control variable elimination
(define *flush-count* 0)		;count of variables eliminated
(define *convert-count* 0)		;count of full beta-conversions

(define (bg-meta-combination-lambda! node)
  (let ((args (bg-combination-args node)))
    (do ((a (cdr args) (cdr a)))
	((null? a))
      (set-car! a (bg-meta-evaluate! (car a)))
      (bg-alter-node! (car a) 'meta-substituted? #f))
    (let ((fn (car args)))
      (do ((v (bg-lambda-vars-bound fn) (cdr v))
	   (a (cdr args) (cdr a))
	   (b (bg-meta-evaluate! (bg-lambda-body fn))
	      (if (bg-subst-candidate? (car a) (car v) b)
		  (bg-meta-substitute! (car a) (car v) b)
		  b)))
	  ((null? v)
	   (bg-alter-lambda! fn 'body (bg-meta-evaluate! b))
	   (do ((v (bg-lambda-vars-bound fn) (cdr v))
		(a (cdr args) (cdr a)))
	       ((null? a))
	     (if (and *flush-args*
		      (null? (bg-var-properties-read-refs (car v)))
		      (null? (bg-var-properties-write-refs (car v)))
		      (or (bg-effectless-except-cons?
			   (bg-node-effects (car a)))
			  (bg-node-meta-substituted? (car a))))
		 (begin
		   (if (or (memq v (bg-node-free-vars (bg-lambda-body fn)))
			   (memq v (bg-node-set!-vars (bg-lambda-body fn))))
		       (bg-internal-error
			"reanalysis lost - bg-meta-combination-lambda!"
			node))
		   (delq! (car a) args)	; don't want to delete from the car
		   (bg-erase-all-nodes! (car a))
		   (set! *flush-count* (1+ *flush-count*))
		   (bg-alter-lambda! fn
				     'vars-bound
				     (delq! (car v)
					    (bg-lambda-vars-bound fn))))))

	   (if (null? (bg-lambda-vars-bound fn))
	       (begin
		 (or (null? (cdr args))
		     (bg-internal-error
		      "too many args in bg-meta-combination-lambda!"
		      node))
		 (let ((bod (bg-lambda-body fn)))
		   (bg-erase-node! (car args))
		   (bg-erase-node! node)
		   (set! *convert-count* (1+ *convert-count*))
		   bod))

	       (begin (bg-reanalyze1! (car args))
		      (bg-alter-node! (car args) 'meta-evaled? #t)
		      (bg-reanalyze1! node)
		      (bg-alter-node! node 'meta-evaled? #t)
		      node)))))))

(define *substitute* #t)		;switch to control substitution
(define *single-subst* #t)		;switch to control substitution of
					;expressions with side effects 
(define *lambda-subst* t)		;switch to control substitution of
					;lambda-expressions 

(define (bg-subst-candidate? arg var bod)
;  (printf "subst?\\n")
;  (pp (bg-node->sexp arg #t '(effects affected)))
;  (printf "for\\n")
;  (pp (bg-var-properties-name var))
;  (printf "in\\n")
;  (pp (bg-node->sexp bod #t '(effects affected)))
;  (printf "yields\\n") (pp2 of value computed below)
  (and *substitute*
       (null? (bg-var-properties-write-refs var))	;be paranoid for now
       
       (or (and *single-subst*
		(>= 1 (length (bg-var-properties-read-refs var))))
	   
	   (or (bg-constant? arg)
	       (bg-var-ref? arg))
	   
	   (and *lambda-subst*
		(bg-lambda? arg)
		(or (null? (cdr (bg-var-properties-read-refs var)))
		    (and (>= 1 (length (bg-var-properties-fn-pos-refs var)))
			 (let ((b (bg-lambda-body arg)))
			   (or (bg-constant? b)
			       (bg-var-ref? b)
			       (and (bg-combination? b)
				    (not (> (length (cdr (bg-combination-args b)))
					    (length (bg-lambda-vars-bound arg))))
				    (do ((a (bg-combination-args b) (cdr a))
					 (p #t
					    (and p
						 (or (bg-constant? (car a))
						     (bg-var-ref? (car a))))))
					((null? a) p)))))))))))

(define (bg-reanalyze1! node)
  (bg-pass1-analyze! node *reanalyze* #t))

(define *reanalyze* 'once)


;;; fixme
;;; Below is a long comment about META-SUBSTITUTE from rabbit.
;;; It's a little hard to follow, but talks about a bug 
;;; with CATCH returning multiple times.
;;; When bugs learns about CALL/CC this will have to be grokked. 
;;; I think the problem can be fixed by adding a RETURNS-TWICE
;;; side effect that can be ommitted from trivial expressions, 
;;; presumed for most everything else.  This won't behave quite like other 
;;; side effects, and that might be what GLS means by `this is a dynamic
;;; problem.... '

;;; Here we determine, for each variable node whose var is the one
;;; given, whether it is possible to substitute in for it; this is
;;; determined on the basis of side effects.  This is done by
;;; walking the program, stopping when a side-effect begins it.
;;; A substitution is made iff is variable node is reached in the walk.

;;; There is a bug in this theory to the effect that a catch
;;; which returns multiply can cause an expression external
;;; to the catch to be evaluated twice.  This is a dynamic problem
;;; which cannot be resolved at compile time, and so we shall
;;; ignore it for now.

;;; We also reset the bg-meta-evaled? flag on all nodes which have a
;;; substitution at or below them, so that the meta-evaluator will
;;; re-penetrate to substitution points, which may admit further
;;; optimizations.

(define (bg-effects-intersect a b)
  (cond ((eq? a bg-any-effect) b)
	((eq? b bg-any-effect) a)
	((eq? a bg-no-effect) a)
	((eq? b bg-no-effect) b)
	(else (intersect a b))))

(define (bg-effectless? x)
  (or (null? x) (eq? x bg-no-effect)))

(define (bg-effectless-except-cons? x)
  (or (bg-effectless? x) (equal? x '(cons))))

(define (bg-passable? node effects affected)
  (begin
    (if (bg-empty? (bg-node-effects node))
	(bg-internal-error "pass 1 analysis missing - bg-passable?" node))
    (and (bg-effectless? (bg-effects-intersect
			  effects
			  (bg-node-affected node)))
	 (bg-effectless? (bg-effects-intersect affected
					       (bg-node-effects node)))
	 (bg-effectless-except-cons?
	  (bg-effects-intersect effects (bg-node-effects node))))))

(define *subst-count* 0)		;count of substitutions
(define *lambda-body-subst* t)		;switch to control substitution in
					;lambda bodies 
(define *lambda-body-subst-try-count* 0) ;count thereof - tries
(define *lambda-body-subst-count* 0) ;count thereof - successes


(define (bg-meta-substitute! arg var bod)
  (let ((effects (bg-node-effects arg))
	(affected (bg-node-affected arg)))
    (if (bg-empty? effects)
	(bg-internal-error "pass 1 analysis screwed up - bg-meta-substitute!"
			   arg))
    (letrec ((substitute
	      (lambda (node)
		(if (or (bg-empty? (bg-node-free-vars node))
			(not (memq var (bg-node-free-vars node)))) ;efficiency
		    node
		    (cond
		     ((bg-constant? node))

		     ((bg-var-ref? node)
		      (if (eq? (bg-var-ref-properties node) var)
			  (begin (bg-erase-all-nodes! node)
				 (set! *subst-count* (1+ *subst-count*))
				 (bg-alter-node! arg
						 'meta-substituted? #t)
				 (bg-copy-code arg))
			  node))

		     ((bg-lambda? node)
		      (if (and (bg-effectless-except-cons? effects)
			       (bg-effectless? affected))
			  (bg-alter-lambda! node
					    'body
					    (substitute
					     (bg-lambda-body node))))

		      (if (bg-node-meta-evaled? node)
			  (bg-alter-node! node
					  'meta-evaled?
					  (bg-node-meta-evaled?
					   (bg-lambda-body node))))
		      node)

		     ((bg-if? node)
		      (bg-alter-if! node
				    'pred (substitute (bg-if-pred node)))
		      (if (bg-passable? (bg-if-pred node) effects affected)
			  (bg-alter-if! node
					'con (substitute (bg-if-con node))
					'alt (substitute (bg-if-alt node))))
		      (if (bg-node-meta-evaled? node)
			  (bg-alter-node!
			   node
			   'meta-evaled?
			   (and (bg-node-meta-evaled? (bg-if-pred node))
				(bg-node-meta-evaled? (bg-if-con node))
				(bg-node-meta-evaled? (bg-if-alt node)))))
		      node)
		     
		     ((bg-set!-? node)
		      (bg-alter-set! node
				     'body (substitute (bg-set!-body node)))
		      (if (bg-node-meta-evaled? node)
			  (bg-alter-node! node
					  'meta-evaled?
					  (bg-node-meta-evaled?
					   (bg-set!-body node))))
		      node)
		     
		     ((bg-combination? node)
		      (let ((args (bg-combination-args node)))
			(do ((a args (cdr a))
			     (x #t (and x (bg-passable? (car a)
						       effects affected))))
			    ((null? a)
			     (if x (do ((a (cdr args) (cdr a)))
				       ((null? a))
				     (set-car! a (substitute (car a)))))
			     (if (and *lambda-body-subst*
				      (bg-lambda? (car args)))
				 (let ((fn (car args)))
				   (set! *lambda-body-subst-try-count*
					 (1+ *lambda-body-subst-try-count*))
				   (if x
				       (begin
					 (set!
					  *lambda-body-subst-count*
					  (1+ *lambda-body-subst-count*))
					 
					 (bg-alter-lambda!
					  fn
					  'body
					  (substitute
					   (bg-lambda-body fn)))))
				   
				   (if (bg-node-meta-evaled? (car args))
				       (bg-alter-node!
					(car args)
					'meta-evaled?
					(bg-node-meta-evaled?
					 (bg-lambda-body fn)))))
				 (if x
				     (set-car! args
					       (substitute (car args)))))))
			(do ((a args (cdr a))
			     (mp #t (and mp (bg-node-meta-evaled? (car a)))))
			    ((null? a)
			     (if (bg-node-meta-evaled? node)
				 (bg-alter-node! node
						 'meta-evaled? mp)))))
		      node))))))
      (substitute bod))))

(define (bg-copy-code node)
  (bg-reanalyze1! (bg-copy-nodes node (bg-node-env node) '())))

(define (copy-var-property prop)
  (if (bg-var-properties-global? prop)
      prop
      (bg-make-local-var-properties (bg-var-properties-name prop))))

(define (bg-copy-nodes node env rnl)
  (cond
   ((bg-constant? node)
    (bg-make-constant (bg-node-sexp node)
		      env
		      (bg-constant-value node)))

   ((bg-var-ref? node)
    (bg-make-var-ref
     (bg-node-sexp node) env
     (let ((slot (assq (bg-var-ref-properties node) rnl)))
       (if slot
	   (cadr slot)
	   (bg-var-ref-properties node)))))

   ((bg-lambda? node)
    (let ((vars (map copy-var-property (bg-lambda-vars-bound node))))
      (bg-make-lambda (bg-node-sexp node) env
		      vars
		      (bg-copy-nodes
		       (bg-lambda-body node)
		       (append (map (lambda (a b)
				      (list (bg-var-properties-name a) b))
				    (bg-lambda-vars-bound node) vars)
			       env)
		       (append (map list (bg-lambda-vars-bound node) vars)
			       rnl)))))
   
   ((bg-if? node)
    (bg-make-if (bg-node-sexp node) env
		(bg-copy-nodes (bg-if-pred node) env rnl)
		(bg-copy-nodes (bg-if-con node) env rnl)
		(bg-copy-nodes (bg-if-alt node) env rnl)))

   ((bg-set!-? node)
    (bg-make-set!- (bg-node-sexp node) env
		   (let ((slot (assq (bg-set!-var node) rnl)))
		     (if slot (cadr slot) (bg-set!-var node)))
		   (bg-copy-nodes (bg-set!-body node) env rnl)))

   ((bg-combination? node)
    (bg-make-combination (bg-node-sexp node) env
			 (map (lambda (n) (bg-copy-nodes n env rnl))
			      (bg-combination-args node)
			      )))))


;;;
;;; The function below has no analog in rabbit.  It's purpose is to try to
;;; prove that variables set!'ed exactly once, are set!'ed before ever being
;;; referenced.  Later, this fact will be used for known-function optimizations
;;; (closure elimination).  It could be used for additional argument folding of
;;; the sort done in the meta evaluator but that is to hairy for now.
;;;
;;; cnode: the node being analyzed in a tree walk
;;; agenda: variables bound above, set!'ed once somewhere, and definately not
;;;         yet referenced. 
;;;
(define (bg-set!-analyze! cnode) hey-bogon)


;;; conversion to continuation-passing style

;;; this involves making a complete copy of the program in terms
;;; of the following new data structures:

(define bg-cnode-type
  (make-record-type "bg-cnode"
		    '(; environment (a list of variables, debugging only)
		      env
		      
		      ;	variables bound above and referenced below this cnode
		      refs

		      ; variables referred to at or below this cnode by
		      ; closures 
		      clovars)))

(define bg-cnode-env (record-accessor bg-cnode-type 'env))
(define bg-cnode-refs (record-accessor bg-cnode-type 'refs))
(define bg-cnode-clovars (record-accessor bg-cnode-type 'clovars))
(define bg-cnode? (record-predicate bg-cnode-type))
(define bg-alter-cnode! (record-generic-modifier bg-cnode-type))
(define (bg-make-cnode-sub-type name fields)
  (make-record-sub-type name fields bg-cnode-type))
(define (bg-cnode-subtype-constructor type . fields)
  (record-constructor-w/fill type bg-empty
			     (if (null? fields) '() (car fields))))



(define bg-trivial-type
  (bg-make-cnode-sub-type "bg-trivial" '(node))) ; node is a pass-1 node tree
(define bg-make-trivial (bg-cnode-subtype-constructor bg-trivial-type '(node)))
(define bg-trivial-node (record-accessor bg-trivial-type 'node))
(define bg-trivial? (record-predicate bg-trivial-type))

; The same var-properties structure is used for both the CPS and pass1 forms of
; the program.
;
(define bg-make-local-cvar-properties
  (lambda (name)
    (bg-var-properties-constructor name name #f)))

(define bg-cvar-ref-type
  (bg-make-cnode-sub-type "bg-cvar-ref"
			  '(properties)	; see above
			  ))
(define bg-make-cvar-ref
  (bg-cnode-subtype-constructor bg-cvar-ref-type '(properties)))
(define bg-cvar-ref-properties
  (record-accessor bg-cvar-ref-type 'properties))
(define bg-cvar-ref? (record-predicate bg-cvar-ref-type))
(define bg-cvar-ref-properties
  (record-accessor bg-cvar-ref-type 'properties))

(define (bg-cvar-ref-name vr)
  (bg-var-properties-name (bg-cvar-ref-properties vr)))
(define (bg-cvar-ref-alpha-name vr)
  (bg-var-properties-alpha-name (bg-cvar-ref-properties vr)))
(define (bg-cvar-ref-to-global? vr)
  (bg-var-properties-global? (bg-cvar-ref-properties vr)))


(define bg-clambda-type
  (bg-make-cnode-sub-type
   "bg-clambda"
   '(vars
     body

     ;closure
     ;  #t => needn't make a full closure of this
     ;	clambda.  may be 'noclose or 'ezclose (the former
     ;	meaning no closure is necessary at all, the latter
     ;	that the closure is merely the environment).
     closure

     ; comp-env: for noclose type functions, we can compute at each invocation
     ; the closed environment from the then active environment.  comp-env
     ; remembers that environment for the code generator.
     comp-env

     ;tvars: the variables which are passed through temp locations
     ;	on entry.  non-nil only if closure='noclose; then is
     ;	normally the lambda vars, but may be decreased
     ;	to account for args which are themselves known noclose's,
     ;	or whose corresponding parameters are never referenced.
     ;	the temp vars involved start in number at dep.
     tvars

     ;name:  the prog tag used to label the final output code for the clambda
     name

     ;consenv: the `consed environment` when the clambda is evaluated
     consenv

     ;closerefs: variables referenced by the clambda which are not in
     ;	the consed environment at evaluation time, and so must be
     ;	added to consenv at that point to make the closure
     closerefs

     ; set!-vars: the elements of vars which are ever seen in a cset!
     set!-vars

     ; frame: if any variable bound here are enclosed below, this will
     ; point to a frame description (defined way below).
     frame

     ; closure?: correcting rabbit's ``design error''.
     cont?
     )))

(define bg-clambda-vars (record-accessor bg-clambda-type 'vars))
(define bg-clambda-body (record-accessor bg-clambda-type 'body))
(define bg-clambda-closure (record-accessor bg-clambda-type 'closure))
(define bg-clambda-comp-env (record-accessor bg-clambda-type 'comp-env))
(define bg-clambda-tvars (record-accessor bg-clambda-type 'tvars))
(define bg-clambda-name (record-accessor bg-clambda-type 'name))
(define bg-clambda-frame (record-accessor bg-clambda-type 'frame))
(define bg-clambda-consenv (record-accessor bg-clambda-type 'consenv))
(define bg-clambda-closerefs (record-accessor bg-clambda-type 'closerefs))
(define bg-clambda-set!-vars (record-accessor bg-clambda-type 'set!-vars))
(define bg-clambda-cont? (record-accessor bg-clambda-type 'cont?))
(define bg-clambda-constructor
  (bg-cnode-subtype-constructor bg-clambda-type
				'(vars body closure comp-env cont? frame)))
(define bg-alter-clambda! (record-generic-modifier bg-clambda-type))

(define (bg-make-clambda vars body)
  (bg-clambda-constructor vars body #f #f #f #f))
; Note that this predicate matches both normal clambdas and clambdas which are
; continuations.  
(define bg-clambda? (record-predicate bg-clambda-type))

(define (bg-make-continuation var body)
  (bg-clambda-constructor (list var) body #f #f #t #f))
(define (bg-continuation? node)
  (and (bg-clambda? node) (bg-clambda-cont? node)))
(define (bg-continuation-var c)
  (car (bg-clambda-vars c)))
(define bg-continuation-body bg-clambda-body)


(define bg-cif-type
  (bg-make-cnode-sub-type "bg-cif" '(pred con alt)))
(define bg-make-cif (bg-cnode-subtype-constructor bg-cif-type '(pred con alt)))
(define bg-cif? (record-predicate bg-cif-type))
(define bg-cif-pred (record-accessor bg-cif-type 'pred))
(define bg-cif-con (record-accessor bg-cif-type 'con))
(define bg-cif-alt (record-accessor bg-cif-type 'alt))


(define bg-cset!-type
  (bg-make-cnode-sub-type "bg-cset!" '(cont var body)))
(define bg-make-cset!
  (bg-cnode-subtype-constructor bg-cset!-type '(cont var body)))
(define bg-cset!-? (record-predicate bg-cset!-type))
(define bg-cset!-var (record-accessor bg-cset!-type 'var))
(define bg-cset!-body (record-accessor bg-cset!-type 'body))
(define bg-cset!-cont (record-accessor bg-cset!-type 'cont))
(define (bg-cset!-global? node)
  (bg-var-properties-global? (bg-cset!-var node)))

; because bugs merges clambda and continuation, it also merges ccombination and
; return. 
(define bg-ccombination-type
  (bg-make-cnode-sub-type "bg-ccombination" '(args return?)))

(define bg-ccombination-constructor
  (bg-cnode-subtype-constructor bg-ccombination-type '(args return?)))

(define (bg-make-ccombination args)
  (bg-ccombination-constructor args #f))
(define bg-ccombination? (record-predicate bg-ccombination-type))
(define bg-ccombination-args (record-accessor bg-ccombination-type 'args))
(define bg-ccombination-return?
  (record-accessor bg-ccombination-type 'return?))

(define (bg-make-return cont cnode)
  (bg-ccombination-constructor (list cont cnode) #t))
(define (bg-return? cnode)
  (and (bg-ccombination? cnode)
       (bg-ccombination-return? cnode)))
(define (bg-return-val cnode)
  (cadr (bg-ccombination-args cnode)))
(define (bg-return-cont cnode)
  (car (bg-ccombination-args cnode)))


(define (bg-convert node cont mp)
  (or (boolean? mp) (bg-internal-error "bg-convert: lists aren't bool" mp))
  (if (bg-empty? (bg-node-trivial? node))
      (bg-internal-error "pass 1 analysis missing" node))
  (or (eq? (bg-node-meta-evaled? node) mp)
      (bg-internal-error "meta-evaluation screwed up metap" node))

  (cond
   ((bg-constant? node)
    (or (bg-node-trivial? node)
	(bg-internal-error "non-trivial constant" node))
    (bg-build-return (bg-make-trivial node) cont))

   ((bg-var-ref? node)
    (or (bg-node-trivial? node)
	(bg-internal-error "non-trivial variable"))
    (bg-build-return (bg-make-trivial node) cont))

   ((bg-lambda? node)
    (bg-build-return (bg-convert-lambda node #f mp) cont))

   ((bg-if? node)
    (or cont (bg-internal-error "missing continuation to if" node))
    (bg-convert-if node cont mp))

   ((bg-set!-? node)
    (or cont (bg-internal-error "missing continuation to set!" node))
    (bg-convert-set! node cont mp))

   ((bg-combination? node)
    (or cont (bg-internal-error "missing continuation to combination" node))
    (bg-convert-combination node cont mp))))

(define (bg-build-return cnode cont)
  (if cont
      (bg-make-return cont cnode)
      cnode))

(define bg-cont-name (bg-make-identifier-family "cont-"))

(define (bg-make-cont-var)
  (bg-make-local-cvar-properties (bg-cont-name)))

(define (bg-convert-lambda node cname mp)
  (let ((cv (bg-make-cont-var)))
    (bg-make-clambda (cons cv (bg-lambda-vars-bound node))
		     (bg-convert (bg-lambda-body node)
				 (bg-make-cvar-ref (or cname cv))
				 mp))))

;;; issues for converting if:
;;; (1) if whole if is trivial, may just create a ctrivial.
;;; (2) if continuation is non-cvariable, must bind a variable to it.
;;; (3) if predicate is trivial, may just stick it in simple cif.

(define (bg-convert-if node cont mp)
  (if (bg-node-trivial? node)
      (bg-build-return (bg-make-trivial node) cont)
      (let* ((cvar (if (bg-cvar-ref? cont)
		       #f
		       (bg-make-cont-var)))
	     (pvar (if (bg-node-trivial? (bg-if-pred node))
		       #f
		       (bg-node-var (bg-if-pred node))))
	     (icont (if cvar
			(bg-make-cvar-ref cvar)
			cont))
	     (ipred (if pvar
			(bg-make-cvar-ref pvar)
			(bg-make-trivial (bg-if-pred node))))

	     (cif (bg-make-cif ipred
			       (bg-convert (bg-if-con node) icont mp)
			       (bg-convert
				(bg-if-alt node)
				(bg-make-cvar-ref
				 (bg-cvar-ref-properties icont))
				mp)))
	     (foo (if pvar (bg-convert (bg-if-pred node)
				       (bg-make-continuation pvar cif)
				       mp)
		      cif)))
	(if cvar
	    (bg-make-ccombination (list (bg-make-clambda (list cvar) foo)
					cont))
	    foo))))

(define (bg-convert-set! node cont mp)
  (if (bg-node-trivial? node)
      (bg-build-return (bg-make-trivial node) cont)
      (bg-convert (bg-set!-body node)
		  (let ((nm (bg-node-var (bg-set!-body node))))
		    (bg-make-continuation
		     nm
		     (bg-make-cset! cont (bg-set!-var node)
				    (bg-make-cvar-ref nm))))
		  mp)))


;;; issues for converting combinations:
;;; (1) trivial argument evaluations are delayed and are not bound to the
;;; variable of a continuation.  we assume thereby that the compiler is
;;; permitted to evaluate operands in any order.
;;; (2) all non-delayable computations are assigned names and strung out with
;;; continuations. 
;;; (3) if cont is a cvariable and the combination is ((lambda ...) ...) then
;;; when converting the lambda-expression we arrange for its body to refer to
;;; the cvariable cont rather than to its own continuation.  this crock
;;; effectively performs the optimization of substituting one variable for
;;; another, only on continuation variables (which couldn't be caught by
;;; meta-evaluate). 

(define (bg-convert-combination node cont mp)
  (if (bg-node-trivial? node)
      (bg-build-return (bg-make-trivial node) cont)
      (do ((a (bg-combination-args node) (cdr a))
	   (delay-flags '()
			(cons (or (bg-node-trivial? (car a))
				  (bg-lambda? (car a)))
			      delay-flags)))
	  ((null? a)
	   (do ((a (reverse (bg-combination-args node)) (cdr a))
		(d delay-flags (cdr d))
		(f (bg-make-ccombination
		    (do ((a (reverse (bg-combination-args node)) (cdr a))
			 (d delay-flags (cdr d))
			 (z '() (cons
				 (if (car d)
				     (if (bg-lambda? (car a))
					 (bg-convert-lambda
					  (car a)
					  (and (null? (cdr a))
					       (bg-cvar-ref? cont)
					       (bg-cvar-ref-properties cont))
					  mp)
					 (bg-make-trivial (car a)))
				     (bg-make-cvar-ref (bg-node-var (car a))))
				 z)))
			((null? a) (cons (car z) (cons cont (cdr z))))))
		   (if (car d) f
		       (bg-convert
			(car a)
			(bg-make-continuation (bg-node-var (car a)) f)
			mp))))
	       ((null? a) f))))))

(define bg-burble #f)
(define (bg-cnode->sexp cnode . field-args)
  (let* ((fields (if (null? field-args) '() (car field-args)))
	 (varlists (if (>= (length field-args) 2) (cadr field-args) '()))
	 (var-props (if (>= (length field-args) 3) (caddr field-args) '()))
	 (name-lambdas? (and (>= (length field-args) 4) (cadddr field-args)))
	 (var-rules
	  (map (lambda (x) (cons x (record-accessor bg-var-properties-type x)))
	       var-props))
	 (var-wrap (lambda (var)
		     (map (lambda (rule) (cons (car rule) ((cdr rule) var)))
			  var-rules))))
			  
    ; isn't this beautiful code? 
    (define (wrap cnode sexp)
      (if (and (null? fields) (null? varlists))
	  sexp
	  (let* ((type (record-type-descriptor cnode))
		 (type-fields (record-type-field-names type))
		 (access (record-generic-accessor type))
		 (valid-fields
		  (remove-if-not (lambda (x) (memq x type-fields)) fields))
		 (valid-va-fields
		  (remove-if-not (lambda (x) (memq x type-fields)) varlists))
		 (answer (list (map (lambda (key)
				      (cons key (access cnode key)))
				    valid-fields)
			       (map
				(lambda (key)
				  (cons key (map var-wrap
						 (or (access cnode key) '()))))
				valid-va-fields)
			       sexp)))
	    (if (null? (cadr answer))
		(set! answer (cons (car answer) (cddr answer))))
	    (if (null? (car answer))
		(set! answer (cdr answer)))
	    (if (= 1 (length answer))
		(car answer)
		answer))))
    (wrap cnode
     (cond 
      ((bg-trivial? cnode)
       (if bg-burble
	   `(trivial ,(bg-node->sexp (bg-trivial-node cnode) #f fields))
	   (bg-node->sexp (bg-trivial-node cnode) #f fields)))
      
      ((bg-cvar-ref? cnode)
       (bg-cvar-ref-alpha-name cnode))
      
      ((bg-continuation? cnode)
       `(,(if bg-burble 'continuation 'lambda)
	 ,@(if name-lambdas? (list (bg-clambda-name cnode)) '())
	 ,(map bg-var-properties-alpha-name
	       (bg-clambda-vars cnode))
	 ,(bg-cnode->sexp (bg-clambda-body cnode)
			  fields varlists var-props name-lambdas?)))
      
      ((bg-clambda? cnode)
       `(,(if bg-burble 'clambda 'lambda)
	 ,@(if name-lambdas? (list (bg-clambda-name cnode)) '())
	 ,(map bg-var-properties-alpha-name (bg-clambda-vars cnode))
	 ,(bg-cnode->sexp (bg-clambda-body cnode)
			  fields varlists var-props name-lambdas?)))
      
      ((bg-cif? cnode)
       `(,(if bg-burble 'cif 'if)
	 ,(bg-cnode->sexp (bg-cif-pred cnode)
			  fields varlists var-props name-lambdas?) 
	 ,(bg-cnode->sexp (bg-cif-con cnode)
			  fields varlists var-props name-lambdas?)
	 ,(bg-cnode->sexp (bg-cif-alt cnode)
			  fields varlists var-props name-lambdas?)))
      
      ((bg-cset!-? cnode)
       `(,(if bg-burble 'cset! 'set!)
	 ,(bg-cnode->sexp (bg-cset!-cont cnode)
			  fields varlists var-props name-lambdas?)
	 ,(bg-var-properties-alpha-name (bg-cset!-var cnode))
	 ,(bg-cnode->sexp (bg-cset!-body cnode)
			  fields varlists var-props name-lambdas?)))
      
      ((bg-return? cnode)
       `(,@(if bg-burble '(return) '())
	 ,(bg-cnode->sexp (bg-return-cont cnode)
			  fields varlists var-props name-lambdas?)
	 ,(bg-cnode->sexp (bg-return-val cnode)
			  fields varlists var-props name-lambdas?)))
      
      ((bg-ccombination? cnode)
       (map (lambda (x)
	      (bg-cnode->sexp x fields varlists var-props name-lambdas?))
	    (bg-ccombination-args cnode)))))))


;;; environment analysis for cps version

;;; we wish to determine the environment at each cnode,
;;; and determine what variables are bound above and
;;; referred to below each cnode.

;;; for each cnode we fill in these slots:
;;;	env	the environment seen at that cnode (a list of vars)
;;;	refs	variables bound above and referred to below that cnode
;;; for each variable referred to in non-function position
;;; by a cvariable or ctrivial cnode we give a non-nil value to the property:
;;;	non-fn-referenced?

;;; fnp is non-nil iff cnode occurs in functional position

(define (bg-cenv-analyze! cnode env fn?)
  (bg-alter-cnode! cnode 'env env)
  (cond
   ((bg-trivial? cnode)
    (bg-cenv-triv-analyze! (bg-trivial-node cnode) fn?)
    (bg-alter-cnode! cnode 'refs (bg-node-free-vars (bg-trivial-node cnode))))

   ((bg-cvar-ref? cnode)
    (let ((v (bg-cvar-ref-properties cnode)))
      (bg-add-var-xref! v 'read-refs  cnode)
      (or fn? (bg-add-var-xref! v 'fn-pos-refs cnode))
      (or fn? (bg-alter-var-properties! v 'non-fn-referenced? #t))
      (bg-alter-cnode! cnode
		      'refs (and (memq v env)
				 (list (bg-cvar-ref-properties cnode))))))

   ; this is good for continuations as well as other lambdas
   ((bg-clambda? cnode)
    (let ((b (bg-clambda-body cnode)))
      (bg-cenv-analyze! b (append (bg-clambda-vars cnode) env) #f)
      (let ((refs (set-difference (bg-cnode-refs b) (bg-clambda-vars cnode))))
	(bg-alter-cnode! cnode 'refs refs))))

   ((bg-cif? cnode)
    (let ((pred (bg-cif-pred cnode))
	  (con (bg-cif-con cnode))
	  (alt (bg-cif-alt cnode)))
      (bg-cenv-analyze! pred env #f)
      (bg-cenv-analyze! con env #f)
      (bg-cenv-analyze! alt env #f)
      (bg-alter-cnode! cnode
		       'refs (union (bg-cnode-refs pred)
				    (union (bg-cnode-refs con)
					   (bg-cnode-refs alt))))))

   ((bg-cset!-? cnode)
    (let ((v (bg-cset!-var cnode))
	  (cn (bg-cset!-cont cnode))
	  (b (bg-cset!-body cnode)))
      (bg-alter-var-properties! (bg-cset!-var cnode) 'non-fn-referenced? #t)
      (bg-cenv-analyze! cn env #t)
      (bg-cenv-analyze! b env #f)
      (bg-alter-cnode! cnode
		       'refs (let ((r (union (bg-cnode-refs cn)
					     (bg-cnode-refs b))))
			       (if (memq v env) (adjoin v r) r)))))

   ((bg-return? cnode)
    (let ((c (bg-return-cont cnode))
	  (v (bg-return-val cnode)))
      (bg-cenv-analyze! c env #t)
      (bg-cenv-analyze! v env #f)
      (bg-alter-cnode! cnode
		       'refs (union (bg-cnode-refs c) (bg-cnode-refs v)))))

   ((bg-ccombination? cnode)
    (let ((args (bg-ccombination-args cnode)))
      (bg-cenv-analyze! (car args) env #t)
      (cond ((and (bg-trivial? (car args))
		  (bg-var-ref? (bg-trivial-node (car args)))
		  (bg-trivial-fn?
		   (bg-var-ref-properties (bg-trivial-node (car args)))))
	     (bg-cenv-analyze! (cdr args) env #t)
	     (cenv-ccombination-analyze! cnode
					env
					(cddr args)
					(union (bg-cnode-refs (car args))
					       (bg-cnode-refs (cadr args)))))
	    (else (cenv-ccombination-analyze! cnode
					     env
					     (cdr args)
					     (bg-cnode-refs (car args)))))))))

;;; This function must go through and locate variables appearing in
;;; non-function position. 

(define (bg-cenv-triv-analyze! node fn?)
  (cond
   ((bg-constant? node) bg-nothing)

   ((bg-var-ref? node)
    (or fn? (bg-alter-var-properties! (bg-var-ref-properties node)
				      'non-fn-referenced? #t)))

   ((bg-lambda? node)
    (or fn?
	(bg-internal-error "trivial closure - bg-cenv-triv-analyze!" node))
    (bg-cenv-triv-analyze! (bg-lambda-body node) #f))

   ((bg-if? node)
    (bg-cenv-triv-analyze! (bg-if-pred node) #f)
    (bg-cenv-triv-analyze! (bg-if-con node) #f)
    (bg-cenv-triv-analyze! (bg-if-alt node) #f))			      

   ((bg-set!-? node)
    (bg-alter-var-properties! (bg-set!-var node) 'non-fn-referenced? #t)
    (bg-cenv-triv-analyze! (bg-set!-body node) #f))

   ((bg-combination? node)
    (do ((a (bg-combination-args node) (cdr a))
	 (f #t #f))
	((null? a))
      (bg-cenv-triv-analyze! (car a) f)))))

(define (cenv-ccombination-analyze! cnode env args frefs)
  (do ((a args (cdr a))
       (r frefs (union r (bg-cnode-refs (car a)))))
      ((null? a)
       (bg-alter-cnode! cnode 'refs r))
    (bg-cenv-analyze! (car a) env #f)))


;;; binding analysis.

;;; for each cnode we fill in:
;;;	clovars		the set of variables referred to by closures
;;;			at or below this node (should always be a
;;;			subset of refs)
;;; for each clambda and continuation we fill in:
;;;	fnp	non-nil iff referenced only as a function.
;;;		will be 'ezclose if referred to by a closure,
;;;		and otherwise 'noclose.
;;;	tvars	variables passed through temp locations when calling
;;;		this function
;;;	name	the name of the function (used for the prog tag)
;;; for each clabels we fill in:
;;;	easy	reflects fnp status of all the labelled functions
;;; for each variable which always denotes a certain function we
;;; put the properties:
;;;	known-function		iff the variable is never aset
;;; the value of the known-function property is the cnode for
;;; the function definition.
;;; for each labels variable in a labels of the 'ezclose variety
;;; we put the property:
;;;	labels-function
;;; to indicate that its `easy` closure must be cdr'd to get the
;;; correct environment (see produce-labels).
;;; name, if non-nil, is a suggested name for the function

(define (bg-bind-analyze! cnode closure name)
  (cond
   ((bg-trivial? cnode)
    (bg-alter-cnode! cnode 'clovars '()))

   ((bg-cvar-ref? cnode)
    (bg-alter-cnode! cnode 'clovars '()))

;   ((bg-continuation? cnode)
;    (bg-bind-analyze-continuation! cnode closure name))
;  ``BIND-ANALYZE-CONTINUATION is entirely analogous to BIND-ANALYZE-LAMBDA.''
;					- GLS

   ((bg-clambda? cnode)
    (bg-bind-analyze-clambda! cnode closure name))

   ((bg-cif? cnode)
    (bg-bind-analyze-cif! cnode))

   ((bg-cset!-? cnode)
    (bg-bind-analyze-cset! cnode))

   ((bg-return? cnode)
    (bg-bind-analyze-return! cnode))

   ((bg-ccombination? cnode)
    (bg-bind-analyze-ccombination! cnode))))


(define (bg-refd-vars vars)
  (do ((v vars (cdr v))
       (w '() (if (or (bg-var-properties-read-refs (car v))
		      (bg-var-properties-write-refs (car v)))
		  (cons (car v) w)
		  w)))
      ((null? v) (nreverse w))))

(define bg-func-name (bg-make-identifier-family "fn-"))

(define (bg-bind-analyze-clambda! cnode closure name)
  (bg-bind-analyze! (bg-clambda-body cnode) #f #f)
  (bg-alter-cnode! cnode
		   'clovars
		   (if (eq? closure 'noclose)
		       (bg-cnode-clovars (bg-clambda-body cnode))
		       (bg-cnode-refs cnode)))
  (bg-alter-clambda! cnode
		     'closure closure
		     'tvars (if (eq? closure 'noclose)
				(bg-refd-vars (bg-clambda-vars cnode))
				#f)
		     'name (or name (bg-func-name))))

(define (bg-bind-analyze-cif! cnode)
  (bg-bind-analyze! (bg-cif-pred cnode) #f #f)
  (bg-bind-analyze! (bg-cif-con cnode) #f #f)
  (bg-bind-analyze! (bg-cif-alt cnode) #f #f)
  (bg-alter-cnode! cnode
		   'clovars
		   (union (bg-cnode-clovars (bg-cif-pred cnode))
			  (union (bg-cnode-clovars (bg-cif-con cnode))
				 (bg-cnode-clovars (bg-cif-alt cnode))))))

; 
; This is the rabbit version.  I think it is wrong because it presumes that the
; value the set! is the value bound to the var (r4rs says it's unspecified so
; optimizing this presumption seems goofy).  Also, the optimization never seems
; to be used since the conversion to cps strings out the body of the set! (see
; bg-convert-set!. 
;
;(define (bg-bind-analyze-cset! cnode)
;  (let ((cn (bg-cset!-cont cnode))
;	(val (bg-cset!-body cnode)))
;    (bg-bind-analyze! cn 'noclose #f)
;    (cond ((and (bg-continuation? cn)
;		(bg-clambda? val))
;	   (bg-internal-error "far out man..." #f)
;	   (let ((var (bg-continuation-var cn)))
;	     (bg-alter-var-properties! var 'known-function val)
;	     (bg-bind-analyze!
;	      val
;	      (and (not (bg-var-properties-non-fn-referenced? var))
;		   (if (memq var (bg-cnode-clovars (bg-continuation-body cn)))
;		       'ezclose
;		       (begin
;			 (bg-alter-continuation! cn 'tvars #f)
;			 'noclose)))
;	      #f)))
;
;	  (else (bg-bind-analyze! val #f #f)))
;    (bg-alter-cnode! cnode
;		     'clovars (union (bg-cnode-clovars cn)
;				     (bg-cnode-clovars val)))))

;
; this is a simple form that does the same thing as the rabbit one did:
;
;(define (bg-bind-analyze-cset! cnode)
;  (let ((cn (bg-cset!-cont cnode))
;	(val (bg-cset!-body cnode)))
;    (bg-bind-analyze! cn 'noclose #f)
;    (bg-bind-analyze! val #f #f)
;    (bg-alter-cnode! cnode
;		     'clovars (union (bg-cnode-clovars cn)
;				     (bg-cnode-clovars val)))))

; Bugs blows off LABELS (letrec) and models it using LET and SET!.
; As a result, we want to handle the case where the body of a set! is a lambda
; specially.  Specificly, if this is the only non-fn occurence of the var being
; set, and if the only references to this variable are in the continuation
; (meaning that the initial value is never used), we will be able to set up a
; known-function property for the var etc.
;

(define (bg-bind-analyze-cset! cnode)
  (let ((cn (bg-cset!-cont cnode))
	(val (bg-cset!-body cnode))
	(var (bg-cset!-var cnode)))
    (bg-bind-analyze! cn 'noclose #f)
    (if (and (bg-clambda? val)
	     (= 1 (length (bg-var-properties-write-refs var)))
	     (bg-var-properties-set-before-strict? var))
	(begin
	  (bg-set-var-known-function! var val)
	  (bg-binding-analyze! val
			       (if (bg-var-properties-non-fn-referenced? var)
				   'ezclose
				   'noclose)
			       #f))
	(bg-bind-analyze! val #f #f))
    (bg-alter-cnode! cnode
		     'clovars (union (bg-cnode-clovars cn)
				     (bg-cnode-clovars val)))))


(define (bg-bind-analyze-return! cnode)
  (let ((cn (bg-return-cont cnode))
	(val (bg-return-val cnode)))
    (bg-bind-analyze! cn 'noclose #f)
    (cond ((and (bg-continuation? cn)
		(bg-clambda? val))
	   (let ((var (bg-continuation-var cn)))
	     (bg-set-var-known-function! var val)
	     (bg-bind-analyze!
	      val
	      (and (not (bg-var-properties-non-fn-referenced? var))
		   (if (memq var (bg-cnode-clovars (bg-continuation-body cn)))
		       'ezclose
		       (begin
			 (bg-alter-continuation! cn 'tvars #f)
			 'noclose)))
	      #f)))

	  (else (bg-bind-analyze! val #f #f)))

    (bg-alter-cnode! cnode
		     'clovars (union (bg-cnode-clovars cn)
				     (bg-cnode-clovars val)))))

(define (bg-bind-analyze-ccombination! cnode)
  (let ((args (bg-ccombination-args cnode)))
    (bg-bind-analyze! (car args) 'noclose #f)
    (let ((fn (car args)))
      (cond ((and (bg-trivial? fn)
		  (bg-var-ref? (bg-trivial-node fn))
		  (bg-trivial-fn?
		   (bg-var-ref-properties (bg-trivial-node fn))))
	     (bg-bind-analyze! (cadr args) 'noclose #f)
	     (bg-bind-ccombination-analyze! cnode
					 (cddr args)
					 #f
					 (bg-cnode-clovars (cadr args))))

	    ((bg-clambda? fn)
	     (bg-bind-ccombination-analyze! cnode
					 (cdr args)
					 (bg-clambda-vars fn)
					 (bg-cnode-clovars (car args)))
	     (map (lambda (v)
		    (if (let ((kfn (bg-var-properties-known-function v)))
			  (and kfn
			       (eq? (cond
				     ((bg-clambda? kfn)
				      (bg-clambda-closure kfn))
				     ((bg-continuation? kfn)
				      (bg-continuation-fn? kfn)))
				    'noclose)))
			(bg-alter-clambda!
			 fn
			 'tvars (delq! v (bg-clambda-tvars fn)))))
		  (bg-clambda-tvars fn)))

	    (else
	     (bg-bind-ccombination-analyze! cnode
					    (cdr args)
					    #f
					    (bg-cnode-clovars (car args))))))))

;;; vars may be nil - we depend on (cdr nil)=nil.

(define (bg-bind-ccombination-analyze! cnode args vars fcv)
  (do ((a args (cdr a))
       (v vars (and vars (cdr v)))
       (cv (or fcv '()) (union cv (let ((t (bg-cnode-clovars (car a))))
				    (or t '())))))
      ((null? a)
       (bg-alter-cnode! cnode 'clovars cv))
    (cond ((and vars
		(or (bg-clambda? (car a))
		    (bg-continuation? (car a)))
		(null? (bg-var-properties-write-refs (car v))))
	   (bg-set-var-known-function! (car v) (car a))
	   (bg-bind-analyze! (car a)
			     (and
			      (not (bg-var-properties-non-fn-referenced?
				    (car v)))
			      (if (memq (car v) fcv)
				  'ezclose
				  'noclose))
			     #f))
	  (else (bg-bind-analyze! (car a) #f #f)))))

(define **number-of-arg-regs** 10)


;;;
;;; This is the first point (pleasantly late in the computation) that bugs
;;; deviates significantly from rabbit.  Uptil now the diffences have had to do
;;; with: a different environment than the original MacLISP implementation;
;;; elimination of CATCH and LABELS, and (fixme: not true yet....) a slightly
;;; trickier flow analysis to compile macro expanded letrecs.
;;;

;;;
;;; Closure analysis designs the data structures that hold variables which are
;;; closed over.   RABBIT used simple lists to represent environments.
;;; BUGS tries to be more clever by using a list of vectors.
;;;
;;; Like rabbit, we compute for every lambda the environment before and 
;;; after closure creation.  This determines the set of variables which
;;; must be moved from registers to consed environments.
;;; 
;;; Rabbit maintained those sets as lists, and used the structure of the lists 
;;; to fix the structure of the consed env.  Bugs does some additional work
;;; by designing a vector to hold all of the closed variables of each lambda.
;;; 

;;; issues: 
;;;
;;; a combination and its continuation may both create closures over the same
;;; variables which are nevertheless not in the environment
;;;
;;;

;
; An env-desc type is created for each lambda with variables that 
; are closed over.  During close-analyze!, a description is intantiated and
; added to the environment as we descend through each lambda.  When it is
; discovered that a variable is needed in a closure, it is added to the
; appropriate env desc.  At run time, storage for all of the closed variables
; in a desc is allocated at once as a vector.
;
; future:  in the future, some parent env. could be combined with 
; children.  
; 
(define bg-frame-type
  (make-record-type "bg-frame"
		    '(binding		; lambda whose vars are here
		      parent		; another env desc
		      vars		; list of vars in frame
		      children		; list of children
		      lowest?		; ever closed over directly?
		      depth		; number of frames below globals
		     )))

(define bg-frame-constructor
  (record-constructor bg-frame-type))
(define (bg-make-frame binding parent)
  (let ((it (bg-frame-constructor binding
				  parent
				  '()	;vars
				  '()	;children
				  #f	;lowest?
				  (or (and parent (1+ (bg-frame-depth parent)))
				      0))))	;depth
    (if parent
	(bg-alter-frame! parent 'children
			 (cons it (bg-frame-children parent))))
    it))
(define bg-frame-binding (record-accessor bg-frame-type 'binding))
(define bg-frame-parent (record-accessor bg-frame-type 'parent))
(define bg-frame-vars (record-accessor bg-frame-type 'vars))
(define bg-frame-children (record-accessor bg-frame-type 'children))
(define bg-frame-depth (record-accessor bg-frame-type 'depth))
(define bg-frame-lowest? (record-accessor bg-frame-type 'lowest?))
(define bg-alter-frame! (record-generic-modifier bg-frame-type))

;; This ensures that var is in env
(define (bg-close! var env)
  (printf "closeing %s (%s)\\n"
	  (symbol->string (bg-var-properties-alpha-name var))
	  (symbol->string (bg-var-properties-name var)))
  (do ((e env (bg-frame-parent e)))
      ((or (not e)
	   (memq var (bg-clambda-vars (bg-frame-binding e))))
       (if e
	   (if (not (memq var (bg-frame-vars e)))
	       (begin
		 (bg-alter-var-properties! var
					   'offset (length (bg-frame-vars e))
					   'frame e)
		 (bg-alter-frame! e 'vars (cons var (bg-frame-vars e)))))
	   (bg-internal-error
	    "bg-close!: trying to enclose mystery variable"
	    var))
       bg-nothing)))

;; This marks the lowest frame being closed over with the fact that it is such.
(define (bg-will-close! env)
  (do ((e env (bg-frame-parent e)))
      ((or (not e) (not (null? (bg-frame-vars e))))
       (if e (bg-alter-frame! e 'lowest? #t)))))

;;; closure analysis for cps version

;;; for each clambda, continuation, and clabels we fill in:
;;;	consenv		the consed environment of the clambda,
;;;			continuation, or clabels (before any
;;;			closerefs have been consed on)
;;; for each clambda and continuation we fill in:
;;;	closerefs	a list of variables referenced by the clambda
;;;			or continuation which are not in the consed
;;;			environment at the point of the clambda or
;;;			continuation and so must be consed onto the
;;;			environment at closure time; however, these
;;;			need not be consed on if the clambda or
;;;			continuation is in function position of
;;;			a father which is a ccombination or return
;;; for the clambda's in the fndefs of a clabels, these may be
;;; slightly artificial for the sake of optimization (see below).
;;; for each clambda we fill in:
;;;	asetvars	a list of the variables bound in the clambda
;;;			which are ever aset and so must be consed
;;;			onto the environment immediately if any
;;;			closures occur in the body
;;; for each clabels we fill in:
;;;	fnenv		variables to be consed onto the current consenv
;;;			before closing the labels functions
;;; cenv is the consed environment (a list of variables)


; This filters a list of variables, removing those bound to a known function
; noclose and referenced only in the function position.
(define (bg-filter-closerefs refs)
  (do ((x refs (cdr x))
       (y '()
	  (if (let ((kfn (bg-var-properties-known-function (car x))))
		(and kfn
		     (eq? (bg-clambda-closure kfn) 'noclose)))
	      y
	      (cons (car x) y))))
      ((null? x) (nreverse y))))

; cenv is a bg-frame record, not a list of vars as in rabbit
(define (bg-close-analyze! cnode cenv)
  (cond
   ((bg-clambda? cnode)
    (let ((cr (if (eq? (bg-clambda-closure cnode) 'noclose)
		  '()
		  (bg-filter-closerefs (bg-cnode-refs cnode))))
	  (sv (do ((v (bg-clambda-vars cnode) (cdr v))
		   (a '() (if (and
			       (not (null?
				     (bg-var-properties-write-refs (car v))))
			       (memq (car v)
				     (bg-cnode-clovars
				      (bg-clambda-body cnode))))
			      (cons (car v) a)
			      a)))
		  ((null? v) a))))
      (bg-alter-clambda! cnode
			 'consenv cenv
			 'closerefs cr
			 'set!-vars sv)
      (let ((frame-here (bg-make-frame cnode cenv)))
	(map (lambda (v) (bg-close! v frame-here)) cr)
	(map (lambda (v) (bg-close! v frame-here)) sv)
	(bg-alter-clambda! cnode 'frame frame-here)
	(bg-close-analyze! (bg-clambda-body cnode) frame-here)
	; if the frame here contains no closed over vars, we can delete it.
	(if (null? (bg-frame-vars frame-here))
	    (let ((kids (bg-frame-children frame-here)))
	      (if cenv
		  (begin
		    (bg-alter-frame!
		     cenv 'children (append kids (bg-frame-children cenv)))
		    (map (lambda (k)
			   (bg-alter-frame! k 'parent cenv)
			   (bg-alter-frame!
			    k 'depth (1+ (bg-frame-depth cenv))))
			 kids)))
	      (bg-alter-clambda! cnode 'frame #f))))
      (if (eq? 'noclose (bg-clambda-body cnode))
	  (bg-alter-clambda! 'comp-env cenv))
      (if (not (eq? 'noclose (bg-clambda-closure cnode)))
	  (bg-will-close! cenv))))
   ((bg-cif? cnode)
    (bg-close-analyze! (bg-cif-pred cnode) cenv)
    (bg-close-analyze! (bg-cif-con cnode) cenv)
    (bg-close-analyze! (bg-cif-alt cnode) cenv))
   ((bg-cset!-? cnode)
    (bg-close-analyze! (bg-cset!-cont cnode) cenv)
    (bg-close-analyze! (bg-cset!-body cnode) cenv))
   ((bg-ccombination? cnode)
    (map (lambda (a) (bg-close-analyze! a cenv))
	 (bg-ccombination-args cnode))))
  bg-nothing)
