; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File generate.scm / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Common Lisp back end

; Translation of a single expression

; Compare this with SCHEMIFY, which is a different back end.

;+++
; To do:
;  - Pass continuations around so that RETURN's can be propagated
;    inside of PROG statements (for readability)

(define @lambda-encountered? (make-fluid #f))

; GENERATE

(define (generate-top node g-env)
  (case (node-type node)
    ((begin)
     (prognify (append (deprognify (generate-top (begin-first node) g-env))
		       (deprognify (generate-top (begin-second node) g-env)))))
    ((define)
     (generate-define node g-env))
    (else
     (let-fluid @lambda-encountered? #f
       (lambda ()
	 (noting-references-to-globals
	   (lambda ()
	     ;; Don't beta-reduce this LET; order is important!
	      (let ((code (generate node g-env)))
		(emit-top-level (locally-specialize (deprognify code)))))))))))

; DEFINE

(define (generate-define def env)
  (let-fluid @where (define-lhs def)
    (lambda ()
      (let-fluid @lambda-encountered? #f
	(lambda ()
	  (noting-references-to-globals
	    (lambda ()
	      (let* ((var (define-lhs def))
		     (name (variable-name var))
		     (CL-sym (get-global-reference var))
		     (rhs (define-rhs def))
		     (code (generate rhs env)))
		(cond ((mutable-global? var)
		       (note-reference-to-global! var CL-sym)
		       `(lisp:progn
			 ,(emit-top-level
			   (locally-specialize
			    `((lisp:setq ,CL-sym ,code))))
			 (schi:set-forwarding-function (lisp:quote ,CL-sym)
						       (lisp:quote ,name))))
		      ((sharp-quote-lambda? code)
		       (let* ((bvl+body (cdr (cadr code)))
			      (body (locally-specialize (cdr bvl+body))))
			 `(lisp:progn
			   (lisp:defun ,CL-sym ,(car bvl+body)
			     ,@(if (and (pair? body)
					(null? (cdr body))
					(car-is? (car body) 'lisp:locally))
				   (cdr (car body))
				   body))
			   (schi:set-value-from-function (lisp:quote ,CL-sym)
							 (lisp:quote ,name)))))
		      (else
		       (note-reference-to-global! var CL-sym)
		       `(lisp:progn
			 ,(emit-top-level
			   (locally-specialize
			    `((lisp:setq ,CL-sym ,code))))
			 ,@(emit-function-proclamation CL-sym)
			 (schi:set-function-from-value (lisp:quote ,CL-sym)
						       (lisp:quote ,name)))))))))))))

; Generate code for a non-definition

(define (generate node env)
  (case (node-type node)
    ((variable) (generate-variable node env))
    ((constant) (generate-constant node env))
    ((call)     (generate-call	   node env))
    ((lambda)   (generate-lambda   node env))
    ((letrec)   (generate-letrec   node env))
    ((if)       (generate-if	   node env))
    ((begin)	(generate-begin	   node env))
    ((set!)	(generate-set!	   node env))
    (else (note "don't know how to generate" node))))

(define (generate-list node-list env)
  (map (lambda (node) (generate node env))
       node-list))

(define (generate-body node env)
  (deprognify (generate node env)))

; Constant

(define (generate-constant node env)
    env ;ignored
    (let ((val (constant-value node)))
      (if (or (number? val) (char? val) (string? val) (boolean? val))
	  val
	  `(lisp:quote ,val))))

; Variable

(define (generate-variable var env)
  env ;ignored
  (let ((sub (get-substitution var)))
    (if (pair? sub)
	(case (car sub)
	  ((val) (cadr sub))
	  ((fun) `(lisp:function ,(cadr sub)))
	  (else
	   (let ((CL-sym (get-global-reference var)))
	     (note-reference-to-global! var CL-sym)
	     CL-sym)))
	(begin (if (global-variable? var)
		   (note-reference-to-global! var sub))
	       sub))))

(define (get-substitution var)
  (or (variable-substitution var)
      (let ((sub
	     (cond ((local-variable? var)
		    (note "Unbound variable in object code" var)
		    (variable-name var))
		   (else
		    (get-global-reference var)))))
	(set-substitution! var sub)
	sub)))

(define *declare-globals-special?* #t)

(define (note-reference-to-global! var CL-sym)
  CL-sym  ;ignore for now, might come in handy later
  (if (and (not (qualified-symbol? (variable-name var)))
	   (or *declare-globals-special?*
	       (eq? (variable-status var) 'defined)   ;+++ ?
	       (pair? (variable-substitution var))
	       ;; Kludge for ALMOST-NO-INTEGRATIONS
	       (signature-ref-aux revised^3-scheme-sig (variable-name var))
	       ))
      (let ((g (fluid @CL-variable-references)))
	(if (and (not (eq? g 'dont-accumulate))
		 (not (memq var g)))
	    (set-fluid! @CL-variable-references (cons var g))))))

; Combinations

(define (generate-call node env)
  (let ((proc (call-proc node))
	(args (call-args node)))
    (case (node-type proc)
      ((variable)
       (cond ((and (global-variable? proc)
		   (not (mutable-global? proc)))
	      (generate-call-to-global proc args env))
	     ((and (pair? (variable-substitution proc))
		   (eq? (car (variable-substitution proc)) '--generate-call--))
	      ;; GO
	      ((cadr (variable-substitution proc))
	       (generate-list args env)))
	     (else
	      (generate-unknown-call proc args env))))
      ((lambda)
       (if (and (not (n-ary? proc))
		(= (length args) (length (lambda-vars proc))))
	   (generate-let proc args env)
	   (generate-unknown-call proc args env)))
      (else
       (generate-unknown-call proc args env)))))

(define (generate-unknown-call proc args env)
  (funcallify (generate proc env)
	      (generate-list args env)))

(define (generate-call-to-global global args env)
  (let* ((sub (get-substitution global))
	 (args-code (generate-list args env)))
    (if (not (pair? sub))
	(generate-call-to-unknown-global global args-code)
	(case (car sub)
	  ((subst)
	   (if (= (length args-code) (length (cadr sub)))
	       (substitute-and-peep (map cons (cadr sub) args-code)
				    (prognify (cddr sub)))
	       (generate-call-to-unknown-global global args-code)))
	  ((lambda)
	   (if (= (length args-code) (length (cadr sub)))
	       `(lisp:let ,(map list (cadr sub) args-code) ,@(cddr sub))
	       (generate-call-to-unknown-global global args-code)))
	  ((fun)
	   `(,(cadr sub) ,@args-code))
	  ((val)
	   (funcallify (cadr sub) args-code))
	  ((case-aux)
	   ;; key key-lists else-thunk . thunks
	   `(lisp:case ,(car args-code)
		,@(map (lambda (key-list thunk)
			 `(,key-list
			   ,@(deprognify (funcallify thunk '()))))
		       (cadr (cadr args-code))
		       (cdddr args-code))
		(lisp:otherwise
		  ,@(deprognify (funcallify (caddr args-code) '())))))
	  ((struct)
	   (note "call to a structure" global) 
	   (generate-call-to-unknown-global global args-code))
	  (else (error "losing CASE" sub))))))

(define (generate-call-to-unknown-global global args-code)
  ;; Go through scheme symbol's function cell
  (let ((CL-sym (get-global-reference global)))
    (if (and (not (qualified-symbol? CL-sym))
	     (lisp:macro-function CL-sym))
	;; Prevent infinite compilation loops!
	`(lisp:funcall ,CL-sym ,@args-code)
	`(,CL-sym ,@args-code))))

; LAMBDA

(define (generate-lambda node env)
  (set-fluid! @lambda-encountered? #t)
  `(lisp:function (lisp:lambda ,@(generate-lambda-aux node env))))

; Returns (bvl . body)
(define (generate-lambda-aux node env)
  (let* ((bvl (lambda-vars node))
	 (vars (proper-listify bvl))
	 (new-names (cl-externalize-locals vars env))
	 (new-env (bind-variables vars new-names env))
	 (body-code (generate-body (lambda-body node) new-env)))
    (if (n-ary? node)
	(let ((bvl (insert-&rest new-names)))
	  `(,bvl
	    ,@(if (memq :lispm lisp:*features*)
		  (let ((rest-var (car (last-pair bvl))))
		    `((lisp:setq ,rest-var (lisp:copy-list ,rest-var))))
		  `())
	    ,@body-code))
	`(,new-names ,@body-code))))

; ,@(emit-sharp-plus ':lispm
;		     `(lisp:setq ,rest-var (lisp:copy-list ,rest-var)))

(define (generate-let proc args env)
  (let ((vars (lambda-vars proc)))
    (if (function-bindable? vars args)
	(let* ((new-names (cl-externalize-locals vars env))
	       (new-env (bind-functions vars new-names env)))
	  `(lisp:flet ,(map (lambda (new-name proc)
			       `(,new-name ,@(generate-lambda-aux proc env)))
			    new-names
			    args)
	     ,@(generate-body (lambda-body proc) new-env)))
	(let ((bvl+body (generate-lambda-aux proc env)))
	  `(lisp:let ,(map list (car bvl+body) (generate-list args env))
	     ,@(cdr bvl+body))))))

; The lexical environment keeps track of what names are in use so that
; we can know when it's safe not to rename.

(define (bind-variables vars new-names env)
  (for-each (lambda (var new-name)
	      (set-substitution! var new-name))
	    vars
	    new-names)
  (gbind vars env))

(define (bind-functions vars new-names env)
  (for-each (lambda (var new-name)
	      (set-substitution! var `(fun ,new-name)))
	    vars
	    new-names)
  (gbind vars env))

(define (gbind vars env)
  (append vars env))

; IF

(define (generate-if node env)
  (let ((test (generate (if-test node) env))
	(con  (generate (if-con node) env))
	(alt  (generate (if-alt node) env)))
    ;;+++ Reconstruct COND's
    (if (and (eq? alt 'schi:unspecified)
	     (fluid @translating-to-file?))
	`(lisp:if ,test ,con)
	`(lisp:if ,test ,con ,alt))))

; BEGIN

(define (generate-begin node env)
  (prognify (append (deprognify (generate (begin-first node) env))
		    (deprognify (generate (begin-second node) env)))))

; SET!

(define (generate-set! node env)
  (let ((var (set!-lhs node))
	(rhs-code (generate (set!-rhs node) env)))
    (cond ((global-variable? var)
	   (if (pair? (variable-substitution var))  ;ugh
	       (note "SET! of an integrated variable" node))
	   (let ((CL-sym (get-global-reference var)))
	     (note-reference-to-global! var CL-sym)
	     (emit-global-set! var CL-sym rhs-code)))
	  (else
	   `(lisp:progn (lisp:setq ,(variable-substitution var) ,rhs-code)
			schi:unspecified)))))

; LETREC

(define (generate-letrec node env)
  (case (get-letrec-strategy node)
    ((general) (generate-general-letrec node env))
    ((labels)  (generate-labels-letrec node env))
    ((prog)    (generate-prog-letrec node env))
    (else (error "unknown strategy" (get-letrec-strategy node)))))

(define (generate-general-letrec node env)
  (let* ((vars (letrec-vars node))
	 (vals (letrec-vals node))
	 (new-names (cl-externalize-locals vars env))
	 (new-env (bind-variables vars new-names env)))
    `(lisp:let ,(map (lambda (new-name)
		       `(,new-name schi:unassigned))
		     new-names)
       ,@(map (lambda (var val) `(lisp:setq ,var ,(generate val new-env)))
	      new-names
	      vals)
       ,@(deprognify (generate (letrec-body node) new-env)))))

(define (generate-labels-letrec node env)
  (let* ((vars (letrec-vars node))
	 (vals (letrec-vals node))
	 (new-names (cl-externalize-locals vars env))
	 (new-env (bind-functions vars new-names env)))
    `(lisp:labels ,(map (lambda (new-name proc)
			   `(,new-name ,@(generate-lambda-aux proc new-env)))
			new-names
			vals)
       ,@(generate-body (letrec-body node) new-env))))

; Sorry, I guess this is pretty hairy.  So it goes.
; It would certainly be cleaner if there were a separate pass that
; mutated the code tree to transform argument passing into assignment.
; Unfortunately, in the absence of parent pointers, tree mutation is
; pretty painful.  Fix later.

(define (generate-prog-letrec node env)
  (let* ((vars (letrec-vars node))
	 (procs (letrec-vals node))
	 (new-names (cl-externalize-locals vars env))
	 (new-env (bind-variables vars new-names env))
	 (temp-lists
	  (map (lambda (proc)
		 (map (lambda (var)
			;;+++ THIS ISN'T RIGHT.  Will sometimes lose
			;; on macros that expand into lambda-expressions.
			;; Really want to do a tree-walk over the
			;; scope of var to see if any closures refer
			;; to it.
			(if (variable-closed-over? var)
			    (make-name-from-uid (variable-name var)
						(generate-uid))
			    #f))
		      (lambda-vars proc)))
	       procs))
	 (proc-new-nameses (map (lambda (proc)
				 (cl-externalize-locals (lambda-vars proc)
							new-env))
			       procs))
	 (proc-envs (map (lambda (proc new-names)
			   (bind-variables (lambda-vars proc) new-names new-env))
			 procs
			 proc-new-nameses)))
    (for-each (lambda (var new-name proc-new-names temp-list)
		(set-substitution!
		 var
		 (list '--generate-call--
		       (lambda (args)
			 ;; Return a CL expression to do the call.
			 ;; Args are already translated.
			 (if (null? args)
			     `(lisp:go ,new-name)
			     `(lisp:progn
			       ;; If we had free-variable information, we could
			       ;; optimize this PSETQ into a SETQ, sometimes.
			       (,(if (null? (cdr args)) 'lisp:setq 'lisp:psetq)
				,@(apply append
					 (map (lambda (new-name temp actual)
						`(,(or temp new-name)
						  ,actual))
					      proc-new-names
					      temp-list
					      args)))
			       (lisp:go ,new-name)))))))
	      vars new-names proc-new-nameses temp-lists)
    `(lisp:prog ,(apply append (map (lambda (temp-list new-names)
				       (map (lambda (temp new-name)
					      (or temp new-name))
					    temp-list
					    new-names))
				     temp-lists
				     proc-new-nameses))
	,@(let ((body (generate-body (letrec-body node) new-env)))
	    ;; Peephole for a very common case
	    (if (and (call? (letrec-body node))
		     (eq? (call-proc (letrec-body node))
			  (car vars)))
		body
		`((lisp:return ,@body))))
	,@(apply append
		 (map (lambda (new-name proc temp-list proc-new-names proc-env)
			`(,new-name
			  (lisp:return
			   ,(letify (filter cadr
					    (map list proc-new-names temp-list))
				    (generate (lambda-body proc) proc-env)))))
		      new-names procs temp-lists proc-new-nameses proc-envs)))))
