/* -*-C-*-

Copyright (c) 1987 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. */

/* $Header: ttyio.c,v 9.22 87/05/14 13:50:34 GMT cph Rel $ */

/* Primitives to perform I/O to and from the console. */

#include "scheme.h"
#include "primitive.h"
#include "character.h"
#include "stringprim.h"

extern char OS_tty_read_char ();
extern char OS_tty_read_char_immediate ();
extern Boolean OS_read_char_ready_p ();

extern void OS_tty_write_char ();
extern Boolean OS_tty_write_chars ();
extern void OS_Flush_Output_Buffer ();

extern FILE *OS_file_open ();
extern Boolean OS_file_close ();
extern Boolean OS_file_write_chars ();

extern void OS_tty_beep ();
extern Boolean OS_Clear_Screen ();
extern Boolean OS_tty_move_cursor ();
extern Boolean OS_tty_get_cursor ();

extern int NColumns ();
extern int NLines ();

/* (TTY-READ-CHAR-READY? delay)
   Return #T iff a character is ready to be read from the console.
   If no such character is ready, wait at most DELAY microseconds
   for a character to be typed.  Return #T immediately if one is
   typed, otherwise return #F after DELAY. */

Built_In_Primitive (Prim_Tty_Read_Char_Ready_P, 1, "TTY-READ-CHAR-READY?", 0xCD)
{
  long delay;
  Primitive_1_Arg ();

  delay = (arg_nonnegative_integer (1));
  return ((OS_read_char_ready_p (delay)) ? TRUTH : NIL);
}

/* (TTY-READ-CHAR)
   Read a character from the console, allowing the user to edit the
   input.  This is used for reading entire expressions. */

Built_In_Primitive (Prim_Tty_Read_Char, 0, "TTY-READ-CHAR", 0xCE)
{
  char chr;
  Primitive_0_Args ();

  chr = (OS_tty_read_char ());
  if (Photo_Open)
    OS_file_write_chars (Photo_File_Handle, &chr, 1);
  return (c_char_to_scheme_char (chr));
}

/* (TTY-READ-CHAR-IMMEDIATE)
   Read a character from the console, without editing. */

Built_In_Primitive (Prim_Tty_Read_Char_Immediate, 0,
		    "TTY-READ-CHAR-IMMEDIATE", 0xCF)
{
  char chr;
  Primitive_0_Args ();

  chr = (OS_tty_read_char_immediate ());
  if (Photo_Open)
    OS_file_write_chars (Photo_File_Handle, &chr, 1);
  return  (c_char_to_scheme_char (chr));
}

/* (TTY-READ-FINISH)
   After having read an entire expression using TTY-READ-CHAR, this
   primitive is called to allow the input editor to clean up.
   Since there is no input editor in this implementation, this is
   a noop. */

Built_In_Primitive (Prim_Tty_Read_Finish, 0, "TTY-READ-FINISH", 0xD0)
{
  Primitive_0_Args ();

  return (NIL);
}

Built_In_Primitive (Prim_Tty_Write_Char, 1, "TTY-WRITE-CHAR", 0xFD)
{
  char chr;
  long x, y;
  Primitive_1_Args ();

  CHECK_ARG (1, CHARACTER_P);
  chr = (scheme_char_to_c_char (Arg1));
  if (chr == NOT_ASCII)
    error_bad_range_arg (1);
  OS_tty_write_char (chr);
  if (Photo_Open)
    OS_file_write_chars (Photo_File_Handle, &chr, 1);
  return (NIL);
}

Built_In_Primitive (Prim_Tty_Write_String, 1, "TTY-WRITE-STRING", 0xFE)
{
  char *string, chr;
  long x, y, length;
  Primitive_1_Args ();

  CHECK_ARG (1, STRING_P);
  string = (Scheme_String_To_C_String (Arg1));
  length = (string_length (Arg1));
  if (! (OS_tty_write_chars (string, length)))
    error_external_return ();
  if (Photo_Open)
    OS_file_write_chars (Photo_File_Handle, string, length);
  return (NIL);
}

Built_In_Primitive (Prim_tty_flush_output, 0, "TTY-FLUSH-OUTPUT", 0x1A2)
{
  Primitive_0_Args ();

  OS_Flush_Output_Buffer ();
  return (NIL);
}

Built_In_Primitive (Prim_Tty_Beep, 0, "TTY-BEEP", 0xFF)
{
  Primitive_0_Args ();

  OS_tty_beep ();
  return (NIL);
}

Built_In_Primitive (Prim_Tty_Clear, 0, "TTY-CLEAR", 0x100)
{
  Primitive_0_Args ();

  return ((OS_Clear_Screen ()) ? TRUTH : NIL);
}

Built_In_Primitive (Prim_Photo_Open, 1, "PHOTO-OPEN", 0x151)
{
  Primitive_1_Args ();

  CHECK_ARG (1, STRING_P);
  if (Photo_Open)
    return (NIL);
  Photo_File_Handle =
    (OS_file_open ((Scheme_String_To_C_String (Arg1)), true));
  Photo_Open = (Photo_File_Handle != NULL);
  if (! Photo_Open)
    error_bad_range_arg (1);
  return (TRUTH);
}

Built_In_Primitive (Prim_Photo_Close, 0, "PHOTO-CLOSE", 0x152)
{
  Primitive_0_Args ();

  if (Photo_Open)
    {
      Boolean result;

      result = (OS_file_close (Photo_File_Handle));
      Photo_Open = false;
      if (! result)
	error_external_return ();
      return (TRUTH);
    }
  else
    return (NIL);
}

Built_In_Primitive (Prim_Tty_Move_Cursor, 2, "TTY-MOVE-CURSOR", 0x10C)
{
  Primitive_2_Args ();

  return (OS_tty_move_cursor ((arg_nonnegative_integer (1)),
			      (arg_nonnegative_integer (2)))
	  ? TRUTH : NIL);
}

Built_In_Primitive (Prim_Tty_Get_Cursor, 0, "TTY-GET-CURSOR", 0x26)
{
  int temp;
  long x, y;
  Pointer *result;
  Primitive_0_Args ();

  if (! (OS_tty_get_cursor (&x, &y)))
    return (NIL);
  Primitive_GC_If_Needed (2);
  result = Free;
  *Free++ = (Make_Unsigned_Fixnum (x));
  *Free++ = (Make_Unsigned_Fixnum (y));
  return (Make_Pointer (TC_LIST, result));
}

Built_In_Primitive (Prim_Screen_X_Size, 0, "SCREEN-X-SIZE", 0x166)
{
  Primitive_0_Args ();

  return (Make_Unsigned_Fixnum (NColumns ()));
}

Built_In_Primitive (Prim_Screen_Y_Size, 0, "SCREEN-Y-SIZE", 0x167)
{
  Primitive_0_Args ();

  return (Make_Unsigned_Fixnum (NLines ()));
}

/* Old primitive (fossil?) */

Built_In_Primitive (Prim_Clear_To_End_Of_Line, 0,
		    "CLEAR-TO-END-OF-LINE", 0x134)
{
  extern Boolean OS_Clear_To_End_Of_Line ();
  Primitive_0_Args ();

  return ((OS_Clear_To_End_Of_Line ()) ? TRUTH : NIL);
}
