;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations make-empty-bit-string))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Test sets.

(define set1 (type "Set1" primitive-set-of '(1 2 3 4 5 6 7 8 9) lisp-objects))
#|SET1|#
(define set2 (type "Set2" primitive-set-of '(1 3 5 7 9)         set1))
#|SET2|#
(define set3 (type "Set3" primitive-set-of '(1 3 5 7)           set2))
#|SET3|#
(define setA (type "SetA" primitive-set-of '(1 3)               set3))
#|SETA|#
(define set4 (type "Set4" primitive-set-of '(1 2 3 4 5 6 7 8)   set1))
#|SET4|#
(define set5 (type "Set5" primitive-set-of '(1 2 3 5 7)         set4))
#|SET5|#
(define setB (type "SetB" primitive-set-of '(1 2 3)             set5))
#|SETB|#
;;; The network constructed here looks like this:
;;;       /--------set1-------\
;;;     set2                 set4
;;;       !                    !
;;;     set3                 set5
;;;       !                    !
;;;     setA                 setB

;;;; Test intersections and unions.
;;; These test cases define intersections and unions between
;;;  both set2 and set4 AND setA and setB; the subsequent union 
;;;  and intersection of set3 and set5 must lie between the
;;;  respective unions and intersections above and below.

;;; Test case 1: Intersections
(define (test-intersections)
  (let ((high-type (type-intersection set2 set4)))
    (let ((low-type (type-intersection setA setB)))
      (let ((sandwich (type-intersection set3 set5)))
	(if (and (subsumed-by? low-type sandwich)
		 (subsumed-by? sandwich high-type)
		 (subsumed-by? low-type high-type))
	    #T (ERROR "Intersection test failed."))))))

;;; Test case 2: Unions 
(define (test-unions)
  (let ((high-type (type-union set2 set4)))
    (let ((low-type (type-union setA setB)))
      (let ((sandwich (type-union set3 set5)))
	(if (and (subsumed-by? low-type sandwich)
		 (subsumed-by? sandwich high-type)
		 (subsumed-by? low-type high-type))
	    #T (ERROR "Union test failed."))))))


;;;; Testing image and tuple constraints.
;;; These test cases involve defining image constraints
;;;  and making sure that appropriately subset/superset
;;;  relations are defined for them.  They define two
;;;  constraints on lists, using the set hierarchy 
;;;  defined above.  The first is lists whose CAR's
;;;  are in set1, the second is lists whose CAR's are
;;;  are in set3, and the third is lists whose CAR's
;;;  are in set2.  If all goes well, these three should
;;;  be appropriately sandwiched in the lattice.

;;; Test case 3: Image constraints
(define (test-image-constraints)
  (let ((lists1 (image-constraint car set1)))
    (let ((lists3 (image-constraint car set3)))
      (let ((lists2 (image-constraint car set2)))
	(if (and (subsumed-by? lists2 lists1)
		 (subsumed-by? lists3 lists2))
	    #T (ERROR "Image constraint test failed."))))))

;;; Test case 4: Tuple constraints 
;;;   (combining image constraints and intersections)
(define (test-tuple-constraints)
  (let ((pairs-of-numbers  (cross-product numbers  numbers))
	(pairs-of-integers (cross-product integers integers)))
    (if (and (satisfies? '(12.01 3.01) pairs-of-numbers)
	     (satisfies? '(12 3) pairs-of-numbers)
	     (satisfies? '(12 3) pairs-of-integers)
	     (not (satisfies? '(12.01 3.01) pairs-of-integers))
	     (subsumed-by? pairs-of-integers pairs-of-numbers))
	#T (ERROR "Tuple constraint test failed."))))


;;;; Testing everything.

(define (test-suite)
  (message $NL "Testing intersections... "
	   ($delay test-intersections))
  (message $NL "Testing unions... "
	   ($delay test-unions))
  (message $NL "Testing image constraints... "
	   ($delay test-image-constraints))
  (message $NL "Testing tuple constraints... "
	   ($delay test-tuple-constraints))
  (message $NL "...All tests completed!"))


;;;; Testing bit strings.

(define (five-random-integers n)
  (list (random n)(random n) (random n) (random n) (random n)
	(random n) (random n) (random n) (random n) (random n)))

(define (random-bit-string size)
  (let ((str (make-empty-bit-string size)))
    (for-range (lambda (x) (if (zero? (random 2))
			       (set! str (bit-string-modify str x #T))))
	       0 size)
    str))

(define (bit-string->bit-list str)
  (define (hack-to-bits str bits cleavage)
    (if (< cleavage 0) bits
	(hack-to-bits str (if (check-bit str cleavage)
			      (cons cleavage bits) bits)
		      (-1+ cleavage))))
  (hack-to-bits str () (bit-string-length str)))

(define (compare-bit-string-to-bit-list string bits)
  (let ((match? #T))
    (for-range (lambda (i)
		 (if (check-bit string i)
		     (if (not (memq i bits))
			 (sequence (printout $NL "Bit mis-set:" i)
				   (set! match? #F)))
		     (if (memq i bits)
			 (sequence (printout $NL "Bit not set:" i)
				   (set! match? #F)))))
	       0 (bit-string-length string))
    match?))

(define (test-bit-string-modify bits-set bits-to-set bitstring)
  (if (compare-bit-string-to-bit-list bitstring bits-set)
      (if (null? bits-to-set) bitstring
	  (test-bit-string-modify
	   (cons (car bits-to-set) bits-set) (cdr bits-to-set)
	   (bit-string-modify bitstring (car bits-to-set) #T)))))
(define (test-bit-string-modify! bits-set bits-to-set bitstring)
  (if (compare-bit-string-to-bit-list bitstring bits-set)
      (if (null? bits-to-set) bitstring
	  (test-bit-string-modify!
	   (cons (car bits-to-set) bits-set) (cdr bits-to-set)
	   (bit-string-modify! bitstring (car bits-to-set) #T)))))

(define (tester size)
  (let* ((string (random-bit-string size))
	 (bits (bit-string->bit-list string)))
    (printout $NL "BITS= " ($comma-list bits))
    (printout $NL "Testing BIT-STRING-MODIFY")
    (test-bit-string-modify  () bits (make-empty-bit-string 0))
    (printout $NL "Testing BIT-STRING-MODIFY!")
    (test-bit-string-modify! () bits (make-empty-bit-string 0))))

(define (test-bit-strings count)
  (define (do-test inc)
    (let ((a (random-bit-string 50))
	  (b (random-bit-string 50)))
      (let ((a-bits (bit-string->bit-list a))
	    (b-bits (bit-string->bit-list b)))
	(let ((or-bits a-bits) (and-bits ()))
	  (printout $NL "Test #" inc)
	  (printout $NL "A bits= " ($comma-list a-bits))
	  (printout $NL "B bits= " ($comma-list b-bits))
	  (printout $NL "Testing BIT-STRING-MODIFY on A")
	  (test-bit-string-modify () a-bits (make-empty-bit-string 0))
	  (printout $Nl "Testing BIT-STRING-MODIFY! on A")
	  (test-bit-string-modify! () b-bits (make-empty-bit-string 0))
	  (printout $NL "Testing AND.")
	  (for-each (lambda (a-bit)
		      (if (memq a-bit b-bits)
			  (set! and-bits (cons a-bit and-bits))))
		    a-bits)
	  (compare-bit-string-to-bit-list (bit-string-and a b) and-bits)
	  (printout $NL "Testing OR.")
	  (for-each (lambda (b-bit)
		      (if (not (memq b-bit or-bits))
			  (set! or-bits (cons b-bit or-bits))))
		    b-bits)
	  (compare-bit-string-to-bit-list (bit-string-or a b) or-bits)
	  ))))
  (for-range do-test 0 count))


;;;; Procedures for getting statistics.

(define (collect-statistics)
  (define (make-record) (vector () () () ()))
  (let ((stats (make-vector type-count)))
    (define (number-of-arcs-down from-type)
      (if (undefined? (arc-down-cache from-type)) 
	  (let ((speczns (specializations from-type)))
	    ((modifier arc-down-cache) from-type
	     (apply + (length speczns) (map number-of-arcs-down speczns)))))
      (arc-down-cache from-type))
    (for-range 0 type-count (lambda (ignore) (make-record)))))

(define all-specializations-cache (td-property ':all-specializations))
(define (all-specializations type)
  (define (default-all-specializations value)
    (if (defined? value) value
	(apply union (specializations type)
	       (map all-specializations (specializations type)))))
  ((mutator all-specializations-cache) type default-all-specializations)
  (all-specializations-cache type))

(define (number-of-specializations type)
  (length (all-specializations type)))

(define arc-down-cache (td-property ':arc-down-cache))
(define arc-up-cache (td-property ':arc-up-cache))

(define (number-of-arcs-up from-type)
  (if (undefined? (arc-up-cache from-type)) 
      (let ((genzns (generalizations from-type)))
	((modifier arc-up-cache)
	 from-type (apply + (length genzns) (map number-of-arcs-up genzns)))))
  (arc-up-cache from-type))
(define (number-of-arcs-down from-type)
  (if (undefined? (arc-down-cache from-type)) 
      (let ((speczns (specializations from-type)))
	((modifier arc-down-cache)
	 from-type (apply + (length speczns) (map number-of-arcs-down speczns)))))
  (arc-down-cache from-type))

(define all-generalizations-cache (td-property ':all-generalizations))
(define (all-generalizations type)
  (define (default-all-generalizations value)
    (if (defined? value) value
	(apply union (generalizations type)
	       (map all-generalizations (generalizations type)))))
  ((mutator all-generalizations-cache) type default-all-generalizations)
  (all-generalizations-cache type))

(define (number-of-generalizations type)
  (length (all-generalizations type)))

(define (collect-statistics type)
  (number-of-generalizations type)
  (number-of-specializations type)
  (number-of-arcs-up type)
  (number-of-arcs-down type))

(define (compute-average fcn list)
  (/ (apply + (map fcn list)) (length list)))

(define (check-minimization type)
  (if (not (= (length (genzns type))
	      (length (minimal-type-set (all-generalizations type)))))
      (error "Specialization Bug on" type))
  (if (not (= (length (speczns type))
	      (length (maximal-type-set (all-specializations type)))))
      (error "Specialization Bug on" type)))
