;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/utility/tuple")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Restricting procedures.
;;; Restricting the domain of a procedure both defines a new procedure
;;; which does explicit type checking and adds a finer domain/range
;;; distinction.

(define restrict)
(define (find-unrestricted-function from-function)
  (if (defined? ((inverse restrict) from-function))
      (let ((restrict-form (first ((inverse restrict) from-function))))
	(find-unrestricted-function (first restrict-form)))
      from-function))
(define properties-to-copy-for-restriction (list memoizer-cache inverse))

(define (construct-restriction function domain range)
  (let ((real-function (find-unrestricted-function function))
	(given-domain (function-domain function))
	(given-range (function-range function)))
    (define (restricted-function x)
      (if (in? x domain) (real-function x)
	  (error "Not in domain!" (list domain x))))
    (define (restricted-mutator x f)
      (define (check-mutation v)
	(if (in? v range) v (error "Not in range" (list range v))))
      (if (in? x domain)
	  ((mutator real-function) x (lambda (v) (check-mutation (f v))))
	  (error "Not in domain!" (list domain x))))
    (if (and (defined? given-domain) (not (<<? domain given-domain)))
	(error "Can't restrict to domain" (list function given-domain domain)))
    (if (and (defined? given-range) (not (<<? range given-range)))
	(error "Can't restrict to range" (list function given-range range)))
    (declare-function! restricted-function domain range
		       ($procedure real-function) "-RESTRICTED")
    (copy-properties function restricted-function properties-to-copy-for-restriction)
    (if (defined? (mutator function))
	((modifier mutator) restricted-function restricted-mutator))
    restricted-function))

(set! restrict (canonical-cache construct-restriction))

(define restr restrict)


;;;; Composing procedures.
;;; Composing procedures is the obvious thing.  It might be that
;;; something special should be done for the undefined token.

(define (make-composition f g)
  (define (f-of-g x) (let ((v (g x))) (if (defined? v) (f v) v)))
  (let ((f-domain (function-domain f)) (f-range (function-range f))
	(g-domain (function-domain g)) (g-range (function-range g)))
    (if (not (<<? g-range f-domain))
	(error "Domain/Range does not overlap!" (list f g)))
    (declare-function! f-of-g g-domain f-range ($procedure F) "-OF-" ($procedure G))
    f-of-g))

(define compose (canonical-cache make-composition))


;;;; Mapping procedures.
;;; Similarly straightforward:

(define (make-mapper f)
  (define (mapper list) (map f list))
  (declare-function! mapper (list-of (function-domain f)) (list-of (function-range f))
		     "MAP-" ($procedure f))
  mapper)

(define mapper (canonical-cache make-mapper))


;;;; LIST-OF-RESULTS-OF
;;; Returns a list generating function which can (presumable be
;;; further composed); this an composition gives you the power of
;;; recursive COMBINATIONS.

(define (make-list-of-results-function from-functions)
  (define (generate-list-from x) (map (lambda (f) (f x)) from-functions))
  (declare-function! generate-list-from
		     (apply <AND> (map function-domain from-functions))
		     (apply cross-product (map function-range from-functions))))

(define make-list-of-results (simple-cache make-list-of-results-function))

(define (list-of-results-of . functions) (make-list-of-results functions))


