;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;; This implements a latticed type system in SCHEME.

;;;  A type is a Scheme predicate placed in a subsumption lattice of predicates.  
;;; The subsumption lattice is maintained by keeping a mapping between
;;; predicates and ``type descriptions'' which describe the corresponding
;;; predicates in the lattice.

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Recursive type definitions.

;;; A recursive type is defined by a base case and a set of ``parts''.
;;;  Each part is a mapping, and an object is of a recursive type if 
;;;  it is either in the base case or every one of its parts is of the 
;;;  recursive type.

;;; Recursive types are defined outside of the standard combinator
;;; structure; they are implemented as simple types, but certain special 
;;; properties are assigned to them based on their definition. 

;;; A recursive type is defined by an anchor, a test, and a set
;;;  of links.  The anchor and test are both already defined types;
;;;  the set of links is a set of mappings acceptable to IMAGE-CONSTRAINT.
;;; An object is in the recursive type if either it satisfies the
;;;  anchor type or it satisfies the test type and all the links
;;;  --- when applied --- produce objects in the anchor type.

(define all-inductive-definitions ())
(define (inductive-definition? x) (memq x all-inductive-definitions))

(define induction-anchor    (td-property 'induction-anchor))
(define induction-test      (td-property 'induction-test))
(define induction-links     (td-property 'induction-links))

(define (make-inductive-definition anchor test links)
  (let ((inductive-type #F))
    (let ((anchor-predicate (td-predicate anchor))
	  (test-predicate (td-predicate test))
	  (link-types (map function-domain links)))
      ;;; This is simply the predicate defined.
      (define (inductive-test x)
	(if (defined-true? (satisfies? x anchor)) #T
	    (if (defined-true? (satisfies? x test))
		(if (every? link-types (lambda (t) (satisfies? x t)))
		    (every? links (lambda (fn) (satisfies? (fn x) inductive-type)))
		    #F)
		(satisfies? x test))))
      ;;; Another definition is a generalization of this one if
      ;;;  its anchor, test and chain predicates are all generalizations
      ;;;  of this types corresponding components.
      (define (<<-def? anchor-1 test-1 links-1 anchor-2 test-2 links-2)
	(and (subsumed-by? anchor-1 anchor-2) (subsumed-by? test-1 test-2)
	     (subset-of? links-2 links-1)))
      (define (generalization? type)
	(<<-def? anchor test links
		 (induction-anchor type) (induction-test type) (induction-links type)))
      (define (specialization? type)
	(<<-def? (induction-anchor type) (induction-test type) (induction-links type)
		 anchor test links))
      (define (create-chain-generalization chain)
 	(add-td-generalization!
 	  inductive-type (<OR> anchor (image-constraint chain inductive-type))))
      (set! inductive-type
	    (simple-type inductive-test
			 (<OR> anchor (apply <AND> test link-types))))
      (set! all-inductive-definitions
	    (cons inductive-type all-inductive-definitions))
      ((modifier induction-anchor) inductive-type anchor)
      ((modifier induction-test) inductive-type test)
      ((modifier induction-links) inductive-type links)
      (add-td-generalization! anchor inductive-type)
      (for-each (lambda (x) (add-td-generalization! x inductive-type))
		(collect-unwindings test links anchor))
      (for-each create-chain-generalization links)
      (add-td-generalization!
	(apply <AND> test
	       (map (lambda (l) (image-constraint l anchor))
		    links))
	inductive-type)
      (add-td-generalization!
	(apply <AND> test
	       (map (lambda (l) (image-constraint l inductive-type))
		    links))
	inductive-type)
      (for-each (lambda (x) (add-td-generalization! inductive-type x))
		(minimal-type-set
		  (extract generalization? all-inductive-definitions)))
      (for-each (lambda (x) (add-td-generalization! x inductive-type))
		(maximal-type-set
		  (extract specialization? all-inductive-definitions)))
      inductive-type)))

(define (subset-of? l1 l2)
  (cond ((null? l1) #T)
	((null? l2) #F)
	(ELSE
	 (let ((head (car l2)))
	   (if (memq head l1)
	       (subset-of? (delq head l1) (cdr l2))
	       #F)))))

;;; The unwindings of an inductive definition are the types which are
;;; finite subtypes of the potentially infinite supertype.
(define (collect-unwindings test links anchor)
  (let ((recursions ()))
    (define (recursion? r)
      (cond ((eq? r anchor) #T)
	    ((<<? r test)
	     (let ((constraints (map (lambda (l) (mapping-constraint l r)) links)))
	       (and (every? constraints defined?) (every? constraints recursion?))))
	    (ELSE #F)))
    (define (test-potential-recursion r)
      (let ((constraints (map (lambda (l) (mapping-constraint l r)) links)))
	(if (and (every? constraints defined?) (every? constraints recursion?))
	    (set! recursions (cons r recursions)))))
    (maptree test-potential-recursion (list test) td-specializations)
    recursions))

;;; ----------------------------------------------------------------
;;;;   INDUCTIVE-DEFINITION 

;;; Lots of hair for providing canonicalized inputs:
(define link-set-standardizer (make-standardizer))
(define (canonicalize-inductive-definition anchor test links)
  (list anchor test (link-set-standardizer links)))

(define canonicalized-inductive-definition
  (canonical-cache make-inductive-definition))
(define (inductive-definition . args)
  (apply canonicalized-inductive-definition
	 (apply canonicalize-inductive-definition args)))
(define inductive-type inductive-definition)

(define (list-of a-type)
  (type (list "List of " a-type) inductive-type
	empty-lists (image-constraint car a-type) (list cdr)))

