#| -*-Scheme-*-

Written 11/16/91 by Max Hailperin <max@nic.gac.edu>
Provides Schematik-style graphics. Schematik-style graphics are availble
iff the environment variable USE_SCHEMATIK_STYLE_GRAPHICS is set.

This code is partially derived from x11graph.scm, which is

Copyright (c) 1989, 1990 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Schematik-style Graphics Interface
;;; package: (runtime schematik-style-graphics)

(declare (usual-integrations))

(define-primitives
  (schgraph-open 2)
  (schgraph-close 1)
  (schgraph-flush 1)
  (schgraph-reset-clip-rectangle 1)
  (schgraph-set-clip-rectangle 5)
  (schgraph-draw-line 5)
  (schgraph-draw-point 3)
  (schgraph-draw-text 4)
  (schgraph-set-drawing-mode 2)
  (schgraph-set-line-style 2)
  (schgraph-draw-postscript 2)
  (schgraph-print 1)
  (schgraph-adjust-coordinates 9)
  (schgraph-clear 1)
  (schgraph-set-foreground-hsb 4)
  (schgraph-set-foreground-rgb 4)
  (schgraph-set-foreground-gray 2)
  (schgraph-set-background-hsb 4)
  (schgraph-set-background-rgb 4)
  (schgraph-set-background-gray 2)
  (real-timer-clear 0)
  (real-timer-set 2)
  (get-environment-variable 1))
  
(define normal/timer-interrupt)

(define (initialize-package!)
  (set! normal/timer-interrupt timer-interrupt)
  (set! schematik-style-graphics-device-type
        (make-graphics-device-type
         `((available? ,operation/available?)
           (clear ,operation/clear)
           (close ,operation/close)
           (coordinate-limits ,operation/coordinate-limits)
           (device-coordinate-limits ,operation/device-coordinate-limits)
           (drag-cursor ,operation/drag-cursor)
           (draw-line ,operation/draw-line)
           (draw-point ,operation/draw-point)
           (draw-postscript ,operation/draw-postscript)
           (draw-text ,operation/draw-text)
           (flush ,operation/flush)
           (move-cursor ,operation/move-cursor)
           (open ,operation/open)
           (print ,operation/print)
           (reset-clip-rectangle ,operation/reset-clip-rectangle)
           (set-background-gray ,operation/set-background-gray)
           (set-background-hsb ,operation/set-background-hsb)
           (set-background-rgb ,operation/set-background-rgb)
           (set-clip-rectangle ,operation/set-clip-rectangle)
           (set-coordinate-limits ,operation/set-coordinate-limits)
           (set-drawing-mode ,operation/set-drawing-mode)
           (set-foreground-gray ,operation/set-foreground-gray)
           (set-foreground-hsb ,operation/set-foreground-hsb)
           (set-foreground-rgb ,operation/set-foreground-rgb)
           (set-line-style ,operation/set-line-style))))
  (set! dirty-schematik-style-devices '())
  unspecific)

(define schematik-style-graphics-device-type)

(define dirty-schematik-style-devices)

(define-structure (schematik-style-device
                   (conc-name schematik-style-device/)
                   (constructor
                    make-schematik-style-device
                    (window devxr devyt)))
  (window false read-only true)
  (dirty? #f)
  (curx 0)
  (cury 0)
  (xl -1)
  (yb -1)
  (xr 1)
  (yt 1)
  (devxr false read-only true)
  (devyt false read-only true))

(define pixels-per-inch 92)
(define points-per-inch 72)

(define (operation/open unit width height)
  (let ((scale (case unit
                 ((pixel pixels) 1)
                 ((point points) (/ pixels-per-inch points-per-inch))
                 (else (error "Unknown unit" unit)))))
    (let ((width (round->exact (* scale width)))
          (height (round->exact (* scale height))))
      (let ((window (schgraph-open width height)))
        (make-schematik-style-device window width height)))))

(define (operation/available?)
  (not (not (get-environment-variable
             "USE_SCHEMATIK_STYLE_GRAPHICS"))))

(define (operation/close device)
  (if (schematik-style-device/dirty? device)
      (flush-dirty-devices))
  (schgraph-close (schematik-style-device/window device))
  unspecific)

(let-syntax
    ((define-schgraph
       (macro (name . args)
         `(define (,(symbol-append 'operation/ name) device ,@args)
            (,(symbol-append 'schgraph- name)
             (schematik-style-device/window device) ,@args)))))
  (define-schgraph clear)
  (define-schgraph draw-line old-x old-y new-x new-y)
  (define-schgraph draw-point x y)
  (define-schgraph draw-postscript string)
  (define-schgraph draw-text x y string)
  (define-schgraph print)
  (define-schgraph reset-clip-rectangle)
  (define-schgraph set-background-gray gray)
  (define-schgraph set-background-hsb hue saturation brightness)
  (define-schgraph set-background-rgb red green blue)
  (define-schgraph set-clip-rectangle xl yb xr yt)
  (define-schgraph set-drawing-mode mode)
  (define-schgraph set-foreground-gray gray)
  (define-schgraph set-foreground-hsb hue saturation brightness)
  (define-schgraph set-foreground-rgb red green blue))

(define (operation/set-line-style device style)
  (schgraph-adjust-coordinates (schematik-style-device/window device)
                               (schematik-style-device/xl device)
                               (schematik-style-device/yb device)
                               (schematik-style-device/xr device)
                               (schematik-style-device/yt device)
                               0
                               0
                               (schematik-style-device/devxr device)
                               (schematik-style-device/devyt device))
  (schgraph-set-line-style (schematik-style-device/window device) style)
  (schgraph-adjust-coordinates (schematik-style-device/window device)
                               0
                               0
                               (schematik-style-device/devxr device)
                               (schematik-style-device/devyt device)
                               (schematik-style-device/xl device)
                               (schematik-style-device/yb device)
                               (schematik-style-device/xr device)
                               (schematik-style-device/yt device)))

(define (operation/flush dev)
  (if (not (schematik-style-device/dirty? dev))
      (let ((dirty-devs dirty-schematik-style-devices))
        (set-schematik-style-device/dirty?! dev #t)
        (set! dirty-schematik-style-devices (cons dev dirty-devs))
        (if (null? dirty-devs)
            (start-flushing-timer))))
  unspecific)

(define (operation/coordinate-limits device)
  (values (schematik-style-device/xl device)
          (schematik-style-device/yb device)
          (schematik-style-device/xr device)
          (schematik-style-device/yt device)))

(define (operation/device-coordinate-limits device)
  (values 0 0
          (schematik-style-device/devxr device)
          (schematik-style-device/devyt device)))

(define (operation/drag-cursor device x y)
  (schgraph-draw-line (schematik-style-device/window device)
                      (schematik-style-device/curx device)
                      (schematik-style-device/cury device)
                      x y)
  (set-schematik-style-device/curx! device x)
  (set-schematik-style-device/cury! device y)
  unspecific)

(define (operation/move-cursor device x y)
  (set-schematik-style-device/curx! device x)
  (set-schematik-style-device/cury! device y)
  unspecific)

(define (operation/set-coordinate-limits device xl yb xr yt)
  (schgraph-adjust-coordinates
   (schematik-style-device/window device)
   (schematik-style-device/xl device)
   (schematik-style-device/yb device)
   (schematik-style-device/xr device)
   (schematik-style-device/yt device)
   xl yb xr yt)
  (set-schematik-style-device/xl! device xl)
  (set-schematik-style-device/yb! device yb)
  (set-schematik-style-device/xr! device xr)
  (set-schematik-style-device/yt! device yt)
  unspecific)

(define (flush-dirty-devices)
  (real-timer-clear)
  (set! timer-interrupt normal/timer-interrupt)
  (actually-flush-dirty-devices))

(define (actually-flush-dirty-devices)
  (define (loop)
    (if (not (null? dirty-schematik-style-devices))
        (let ((dev (car dirty-schematik-style-devices)))
          (set-schematik-style-device/dirty?! dev #f)
          (set! dirty-schematik-style-devices 
                (cdr dirty-schematik-style-devices))
          (schgraph-flush (schematik-style-device/window dev))
          (loop))))
  ;; The below is in danger of creating an infinite loop if an error
  ;; occurs in the early parts of the loop and keeps getting escaped out
  ;; of only to retry it.  However, my hope is that the only place an
  ;; escape would occur would be from an error handler in the schgraph-flush
  ;; so by then that device has already been removed from the list, and hence
  ;; will not be retried.  The point is to continue flushing the remaining
  ;; devices even if an error occurs in flushing one and the user chooses to
  ;; abort.
  (dynamic-wind (lambda () #f)
                loop
                (lambda ()
                  (if (not (null? dirty-schematik-style-devices))
                      (actually-flush-dirty-devices)))))

;;; We do our best to co-exist with other timer users.
(define (start-flushing-timer)
  (if (or (eq? timer-interrupt normal/timer-interrupt)
          (eq? timer-interrupt actually-flush-dirty-devices))
      (begin (set! timer-interrupt actually-flush-dirty-devices)
             (real-timer-set 100 0))
      (actually-flush-dirty-devices)))
