/* eval, apply, etc. */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"
#include "utils.h"
#include "refine.h"
#include "codegen.h"
#include "parser.h"
#include <pwd.h>

#ifndef lint
static char SccsId[] = "%W% %G%";
#endif

	/* utility routines */

/* Evaluate expression in scope defined by frame and return the result, which
 * is placed in the current gc. */
Obj objEval(expr, frame)
     Obj expr, frame;
{
    CodeGen gen;
    Obj obj, cvec;
    Callback_t gencb;
    optInfo_t opt;

    /* for a symbol or packaged symbol just look up its value */
    if (CLASS(expr) == Symbol) {
	if ((obj = objLookup(expr, frame)) == (Obj)NULL)
	    errorPrint(BadSymbol, "%O in %O", expr, frame);
	return obj;
    }
    else if (CLASS(expr) == Package) {
	if ((obj = objLookup(DATA(expr, sym, packageInst),
			      objCdr(DATA(expr, package, packageInst))))
	    == (Obj)NULL)
	    errorPrint(BadSymbol, "%O in package %O",
		       DATA(expr, sym, packageInst),
		       objCar(DATA(expr, package, packageInst)));
	return obj;
    }

    obj = macroExpand(expr);
#ifdef DEBUG
    if (debugMacro) objPrintf("*** macro-expand\n%O\n***\n", obj);
#endif /* DEBUG */
    obj = scopeBind(obj, frame);

    /* compile expr into a codevector */
    gen = codeNew(&gencb);
    DEF_OPT(opt, NULL);
    compExpr(obj, gen, FALSE, &opt);
    codeInst(code_halt, gen);

    /* build a codevector */
    cvec = newCodevec(gcTemp, gen, (Obj)NULL, expr);

#ifdef DEBUG
    if (debugCode) codevecPrint(cvec, 0);
#endif /* DEBUG */

    /* execute the codevector */
    obj = codeExec(cvec, frame);

    gcInsert(obj);
    return obj;
}

/* quote each item in list */
static Obj objQuoteAll(list)
     Obj list;
{
    if (list == NilSymb) return list;
    else return newPair(gcNew, objOp1(QuoteSymb, objCar(list)),
			objQuoteAll(objCdr(list)));
}

/* Apply proc (user or prim) to a single arg and return the result
 * (which is placed in the current gc) */
Obj objApply1(proc, arg)
     Obj proc, arg;
{
    CodeGen gen;
    Obj obj, cvec;
    Callback_t gencb;

    if (CLASS(proc) == Symbol)
	proc = objGet(proc, GlobalEnv);
    gen = codeNew(&gencb);

    codePush(arg, gen);
    codePush(proc, gen);
    codeInst(code_call, gen);
    codeOp(1, gen);
    codeInst(code_halt, gen);

    /* build a codevector */
    cvec = newCodevec(gcTemp, gen, (Obj)NULL, (Obj)NULL);

#ifdef DEBUG
    if (debugCode) codevecPrint(cvec, 0);
#endif /* DEBUG */

    /* execute the codevector */
    obj = codeExec(cvec, GlobalEnv);

    gcInsert(obj);
    return obj;
}

/* Apply proc (user or prim) to args (list) and return the result, which is
 * placed in the current gc. */
Obj objApply(proc, args)
     Obj proc, args;
{
    CodeGen gen;
    Obj obj, cvec;
    Callback_t gencb;

    if (CLASS(proc) == Symbol)
	proc = objGet(proc, GlobalEnv);
    gen = codeNew(&gencb);

    codePush(proc, gen);
    codePush(args, gen);
    codeInst(code_apply, gen);
    codeInst(code_halt, gen);

    /* build a codevector */
    cvec = newCodevec(gcTemp, gen, (Obj)NULL, (Obj)NULL);

#ifdef DEBUG
    if (debugCode) codevecPrint(cvec, 0);
#endif /* DEBUG */

    /* execute the codevector */
    obj = codeExec(cvec, GlobalEnv);

    gcInsert(obj);
    return obj;
}

/* standard exit */
void FATAL(mesg)
     char *mesg;
{
    (void)fprintf(stderr, "Fatal error: %s\n", mesg);
#ifdef DEBUG
    abort();
#else
    errorExit(1);
#endif /* DEBUG */
}

/* Start a read-eval-print loop. Expressions are read from fin and
 * evaluated in the current package.  Results are sent to fout
 * if not NULL.  If prompt is TRUE then prompts are sent to stdout.
 * An error terminates the loop. */
void repLoop(fin, fout, prompt)
     Obj fin;
     FILE *fout;
     Boolean prompt;
{
    jmp_buf errmark;
    Obj res;
    FILE *fp;

    errorSetLevel(errmark);

    fp = prompt ? stdout : (FILE *)NULL;
    if (setjmp(errmark) == 0) {
	for (;;) {
	    gcBegin();

	    intEnable();
	    if ((res = parseObj(fin, fp)) == (Obj)NULL)
		errorPrint(NoInput, (char *)NULL);
	    intDisable();

	    if (fout) objfPrintf(fout, "%O\n", objEval(res, CurrentPackage));
	    else (void)objEval(res, CurrentPackage);

	    gcEnd();
	}
    }
    else {
	/* error */
	errorClearLevel();
    }
}

/* Expand leading ~ or ~username in fn.  Returns a static array. */
char *expandFilename(fn)
     char *fn;
{
    static char fbuf[ 500 ];
    extern char *getenv();
    extern struct passwd *getpwnam();

    char *ptr, c;

    if ((c = *fn) == '~') {
	c = *++fn;
	if (c == '/' || c == '\0') {
	    if (ptr = getenv("HOME"))
		(void)strcpy(fbuf, ptr);
	    else {
		(void)fprintf(stderr, "getenv:  HOME not set, using \"%s\"\n",
			      fn);
		return fn;
	    }
	}
	else {
	    struct passwd *pwd;

	    ptr = fbuf;
	    while ((c = *fn) && c != '/') {
		*(ptr++) = c;
		fn++;
	    }
	    *ptr = '\0';
	    if ((pwd = getpwnam(fbuf)) == (struct passwd *)NULL) {
		(void)fprintf(stderr,
			      "getpwnam:  no entry for %s, using \"%s\"\n",
			      fbuf, fn);
		return fn;
	    }
	    (void)strcpy(fbuf, pwd->pw_dir);
	}
	(void)strcat(fbuf, fn);
	fn = fbuf;
    }

    return fn;
}

/* Load fn into current package.  If script is set, then the first line
 * is stripped.
 * The result of evaluating each expression is echoed if verbose is TRUE.
 * If fn does not exist and toplevel is TRUE then exit, otherwise call
 * errorPrint. */
void loadFile(fn, script, verbose, toplevel)
     char *fn;
     Boolean script, verbose, toplevel;
{
    FILE *fp;
    Obj file, save = CurrentPackage;

    if ((fp = fopen(expandFilename(fn), "r")) == (FILE *)NULL) {
	if (toplevel) {
	    perror(fn);
	    exit(1);
	}
	else errorPrint(BadFile, "%s", fn);
    }
    if (script) {
	int c;

	c = fgetc(fp);
	if (c == '#') {
	    while (c != EOF && c != '\n')
		c = fgetc(fp);
	}
	else ungetc(c, fp);
    }

    file = newFile(gcNew, fp, FALSE);
    repLoop(file, verbose ? stdout : (FILE *)NULL, FALSE);
    objDestroy(file);
    CurrentPackage = save;
}

/* Convert list to vector.  If list isn't really a list, then return NULL. */
Obj listToVector(list)
     Obj list;
{
    int len, i;
    Obj vec, tmp = list;

    for (len = 0; CLASS(tmp) == Pair; len++) {
	intCheck();
	tmp = objCdr(tmp);
    }

    if (tmp != NilSymb) return (Obj)NULL;

    vec = newVector(gcTemp, len);
    for (i = 0; i < len; i++) {
	objVectorSet(vec, i, objCar(list));
	list = objCdr(list);
    }
    return vec;
}

/* Convert vector to list */
Obj vectorToList(vec)
     Obj vec;
{
    int i;
    Obj list;

    list = NilSymb;
    i = objVectorSize(vec);
    while (--i >= 0)
	list = newPair(gcTemp, objVectorRef(vec, i), list);
    return list;
}

/* append list b to list a destructively */
Obj objAppendList(a, b)
     Obj a, b;
{
    Obj temp, tail;
    
    if (a == NilSymb) return b;
    for (temp = a; CLASS(temp) == Pair; temp = tail) {
	tail = objCdr(temp);
	if (tail == NilSymb) {
	    objSetCdr(temp, b);
	    return a;
	}
    }
    errorPrint(BadSyntax, "in %O", a);
    /*NOTREACHED*/	    
}

/* append the list of b to a destructively */
Obj objAppendObj(a, b)
     Obj a, b;
{
    Obj temp;

    objSetCar(temp = gcTemp(Pair), b);
    objSetCdr(temp, NilSymb);

    if (a == NilSymb) return temp;
    return objAppendList(a, temp);
}

/* Return (op q). */
Obj objOp1(op, q)
     Obj op, q;
{
    return newPair(gcTemp, op,
		   newPair(gcNew, q, NilSymb));
}

/* Return (op a b). */
Obj objOp2(op, a, b)
     Obj op, a, b;
{
    return newPair(gcTemp, op,
		   newPair(gcNew, a,
			   newPair(gcNew, b, NilSymb)));
}

/* Get argn from the list (arg0 arg1 ...) or NULL if n is too large. */
Obj objGetArg(exp, num)
     Obj exp;
     int num;
{
    while (--num >= 0) {
	if (CLASS(exp) == Pair) exp = objCdr(exp);
	else return (Obj)NULL;
    }
    if (CLASS(exp) == Pair) return objCar(exp);

    return (Obj)NULL;
}

/* if obj is not type then generate an error */
void _typeStrict(obj, type)
     Obj obj;
     Class type;
{
    TYPE_STRICT(obj, type);
}

/* if obj is not type or a subclass of type then generate an error */
void _typeCheck(obj, type)
     Obj obj;
     Class type;
{
    TYPE_CHECK(obj, type);
}

/* wrapper for objPrint that prints strings and characters with any escapes */
void objDisplay(obj, port)
     Obj obj;
     FILE *port;
{
    if (CLASS(obj) == String)
	fputs(objString(obj), port);
    else if (CLASS(obj) == Character)
	putc(objInteger(obj), port);
    else objPrint(obj, port);
}

struct tmp_file_s {
    char fname[ 32 ];
    FILE *fptr;
};

static void openTempFile(tp)
     struct tmp_file_s *tp;
{
    (void)strcpy(tp->fname, "/usr/tmp/flXXXXXX");
    mktemp(tp->fname);
    if ((tp->fptr = fopen(tp->fname, "w+")) == (FILE *)NULL) {
	perror(tp->fname);
	exit(1);
    }
}

static void closeTempFile(tp)
     struct tmp_file_s *tp;
{
    fclose(tp->fptr);
    unlink(tp->fname);
}

/* make a string that is a representation of obj
 *
 * prnt is a function of two arguments (obj, file) that sends the
 * representation of obj to file.  The output to file is returned
 * as a C string.
 * If obj has cycles then NULL is returned. */
char *objAsString(obj, prnt)
     Obj obj;
     F_VOID prnt;
{
    char *str;
    long len;
    struct tmp_file_s temp;
    Callback_t cb;

    openTempFile(&temp);
    cb.arg = (Ptr)&temp;
    cb.func = closeTempFile;
    errorPushCB(&cb);

    (*prnt)(obj, temp.fptr);
    len = ftell(temp.fptr);
    if (fseek(temp.fptr, 0L, 0) < 0) {
	perror("fseek");
	exit(1);
    }
    str = NEWVEC(char, len + 1);
    fread(str, sizeof (*str), (int)len, temp.fptr);
    str[ len ] = '\0';

    errorPopCB();
    closeTempFile(&temp);
    return str;
}

/* look for cycles in obj setting the MARK bit of each pair, vector, or box
 *
 * Assumes that only pairs, vectors, or boxes can have cycles. */
static Boolean findCycle(obj)
     Obj obj;
{
    Class type;

    for (;;) {
	type = CLASS(obj);
	if (type == Pair) {
	    if (checkCond(obj, MARK)) return TRUE;
	    setCond(obj, MARK);
	    if (objHasCycle(objCar(obj))) return TRUE;
	    obj = objCdr(obj);
	}
	else if (type == Vector) {
	    int len;
	    Obj *vec;

	    if (checkCond(obj, MARK)) return TRUE;
	    len = DATA(obj, size, vectorInst);
	    vec = DATA(obj, vector, vectorInst);
	    while (--len >= 0)
		if (objHasCycle(*vec++)) return TRUE;
	    return FALSE;
	}
	else if (type == Box) {
	    if (checkCond(obj, MARK)) return TRUE;
	    setCond(obj, MARK);
	    obj = DATA(obj, ref, boxInst);
	}
	else return FALSE;
    }
    /*NOTREACHED*/
}

/* clear all the MARK bits of pairs, vectors, or boxes in obj
 *
 * Assumes that only pairs, vectors, or boxes can have cycles. */
void clearMarks(obj)
     Obj obj;
{
    Class type;

    for (;;) {
	type = CLASS(obj);
	if (type == Pair) {
	    if (!checkCond(obj, MARK)) return ;
	    clearCond(obj, MARK);
	    clearMarks(objCar(obj));
	    obj = objCdr(obj);
	}
	else if (type == Vector) {
	    int len;
	    Obj *vec, tmp;

	    if (!checkCond(obj, MARK)) return ;
	    clearCond(obj, MARK);
	    len = DATA(obj, size, vectorInst);
	    vec = DATA(obj, vector, vectorInst);
	    while (--len >= 0) {
		if (tmp = *vec++) clearMarks(tmp);
	    }
	    return ;
	}
	else if (type == Box) {
	    if (!checkCond(obj, MARK)) return ;
	    clearCond(obj, MARK);
	    obj = DATA(obj, ref, boxInst);
	}
	else return ;
    }
    /*NOTREACHED*/
}

/* return TRUE if obj is cyclic
 *
 * Assumes that the only pairs or vectors can have cycles. */
Boolean objHasCycle(obj)
     Obj obj;
{
    Boolean cyclep;

    cyclep = findCycle(obj);
    clearMarks(obj);
    return cyclep;
}
