;*---------------------------------------------------------------------*/
;*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime1.7/Llib/hash.scm ...         */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Sep  1 08:51:06 1994                          */
;*    Last change :  Thu Dec 29 15:08:31 1994 (serrano)                */
;*    -------------------------------------------------------------    */
;*    Un module de table de hash.                                      */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __hash
   (foreign (long get_hash_number                    (string)
		  "get_hash_number")
	    (long get_hash_power_number              (string long)
		  "get_hash_power_number")
	    (long get_hash_number_from_int           (long)
		  "get_hash_number_from_int")
	    (long get_hash_power_number_from_int     (long long)
		  "get_hash_power_number_from_int")
	    (long get_hash_number_from_pointer       (obj)
		  "get_hash_number_from_int")
	    (long get_hash_power_number_from_pointer (obj long)
		  "get_hash_power_number_from_int"))
   (export  (make-hash-table         <size> <hashnb> <getkey> <eq> . <size>)
	    (hash-table?             <obj>)
	    (hash-table-nb-entry     <table>)
	    (hash-table->vector      <table>)
	    (get-hash                <key> <table>)
	    (put-hash!               <obj> <table>)
	    (rem-obj-hash!           <obj> <table>)
	    (rem-key-hash!           <obj> <table>)
	    (for-each-hash           <fun> <table>)
	    (inline string->0..255   <string>)
	    (inline string->0..2^x-1 <string> <power>)
	    (inline int->0..255      <int>)
	    (inline int->0..2^x-1    <int> <power>)
	    (inline obj->0..255      <obj>)
	    (inline obj->0..2^x-1    <obj> <power>)))

;*---------------------------------------------------------------------*/
;*    hash-table                                                       */
;*---------------------------------------------------------------------*/
(define-struct hashtbl
   max-size           ;; integer   : la taille maximum de la table
   size               ;; integer   : la taille courante de la table
   get-hash-number    ;; integer   : la fonction de hashage
   get-key            ;; obj->key  : la fonction cherchant la cle des objets
   nb-entry           ;; integer   : le nombre d'entrees de la table
   predicate          ;; oxo->bool : le predicat d'equivalence
   table)             ;; vecteur   : la table elle meme.

;*---------------------------------------------------------------------*/
;*    make-hash-table ...                                              */
;*    -------------------------------------------------------------    */
;*    max-size       : int           : la taille max de la table       */
;*    get-hash-number: key -> int    : la fonction de hashage          */
;*    get-key        : obj -> key    : la cle des objets               */
;*    eq             : obj x obj -> b: le test d'egalite.              */
;*    init-size      : int           : la taille initiale de la table  */
;*---------------------------------------------------------------------*/
(define (make-hash-table max-size get-hash-number get-key eq . init-size)
   (let ((size     (if (null? init-size)
		       max-size
		       (car init-size)))
	 (2^power? (lambda (size)
		      ;; je suis sur qu'on peut ecrire cette fonction
		      ;; vraiment mieux mais comme ses perfs n'ont aucune
		      ;; importance, je laisse ca comme ca pour le moment.
		      (let loop ((n 1))
			 (let ((num (bit-lsh 1 n)))
			    (cond
			       ((=fx num size)
				#t)
			       ((<fx num size)
				(loop (+fx n 1)))
			       (else
				#f)))))))
      (cond
	 ((not (2^power? size))
	  (error "make-hash-table"
		 "Illegal init-size (not a 2 power)"
		 size))
	 ((not (2^power? max-size))
	  (error "make-hash-table"
		 "Illegal max-size (not a 2 power)"
		 max-size))
	 ((>fx size max-size)
	  (error "make-hash-table"
		 "init-size greater than max-size !"
		 size))
	 (else
	  (let ((table (make-vector size '())))
	     (hashtbl max-size size get-hash-number get-key 0 eq table))))))

;*---------------------------------------------------------------------*/
;*    hash-table? ...                                                  */
;*---------------------------------------------------------------------*/
(define (hash-table? obj)
   (hashtbl? obj))

;*---------------------------------------------------------------------*/
;*    hash-table-nb-entry ...                                          */
;*---------------------------------------------------------------------*/
(define (hash-table-nb-entry table)
   (hashtbl-nb-entry table))

;*---------------------------------------------------------------------*/
;*    hash-table->vector ...                                           */
;*---------------------------------------------------------------------*/
(define (hash-table->vector table)
   (hashtbl-table table))

;*---------------------------------------------------------------------*/
;*    get-hash-number ...                                              */
;*---------------------------------------------------------------------*/
(define (get-hash-number table key)
   (let* ((hash-function (hashtbl-get-hash-number table))
	  (max-size      (hashtbl-max-size table))
	  (size          (hashtbl-size table))
	  (num           (hash-function key))
	  (res           (if (<fx size max-size)
			     (modulo num size)
			     num)))
      (if (>=fx res max-size)
	  (error "get-hash-number" "Illegal get-hash-function" table)
	  res)))
   
;*---------------------------------------------------------------------*/
;*    get-hash ...                                                     */
;*---------------------------------------------------------------------*/
(define (get-hash key table)
   (let* ((hash-num  (get-hash-number table key))
	  (hash-eq?  (hashtbl-predicate table))
	  (bucket    (vector-ref (hashtbl-table table) hash-num))
	  (get-key   (hashtbl-get-key table)))
      (let loop ((bucket bucket))
	 (cond
	    ((null? bucket)
	     #f)
	    ((hash-eq? (get-key (car bucket)) key)
	     (car bucket))
	    (else
	     (loop (cdr bucket)))))))

;*---------------------------------------------------------------------*/
;*    put-hash! ...                                                    */
;*---------------------------------------------------------------------*/
(define (put-hash! obj table)
   (if (and (<fx (hashtbl-size table) (hashtbl-max-size table))
	    (>fx (hash-table-nb-entry table)
		 (/fx (hashtbl-size table) 2)))
       ;; quand le nombre d'entree est egal a la taille de la table
       ;; divisee par deux, a augemte la table
       (hash-table-grows! table))
   (let* ((key        ((hashtbl-get-key table) obj))
	  (hash-eq?   (hashtbl-predicate table))
	  (hash-num   (get-hash-number table key))
	  (vec        (hashtbl-table table))
	  (bucket     (vector-ref vec hash-num))
	  (get-key    (hashtbl-get-key table)))
      (if (null? bucket)
	  (begin
	     (hashtbl-nb-entry-set! table (+fx 1 (hashtbl-nb-entry table)))
	     (vector-set! vec hash-num (list obj))
	     obj)
	  (let loop ((bucket bucket))
	     (cond
		((hash-eq? (get-key (car bucket)) key)
		 (car bucket))
		((null? (cdr bucket))
		 (hashtbl-nb-entry-set! table (+fx 1 (hashtbl-nb-entry table)))
		 (set-cdr! bucket (list obj))
		 obj)
		(else
		 (loop (cdr bucket))))))))

;*---------------------------------------------------------------------*/
;*    rem-obj-hash! ...                                                */
;*---------------------------------------------------------------------*/
(define (rem-obj-hash! obj table)
   (let* ((key        ((hashtbl-get-key table) obj))
	  (hash-eq?   (hashtbl-predicate table))
	  (hash-num   (get-hash-number table key))
	  (vec        (hashtbl-table table))
	  (bucket     (vector-ref vec hash-num))
	  (get-key    (hashtbl-get-key table)))
      (cond
	 ((null? bucket)
	  #f)
	 ((eq? (car bucket) obj)
	  (hashtbl-nb-entry-set! table (-fx (hashtbl-nb-entry table) 1))
	  (vector-set! vec hash-num (cdr bucket))
	  #t)
	 (else
	  (let loop ((bucket bucket))
	     (cond
		((eq? (cadr bucket) obj)
		 (hashtbl-nb-entry-set! table (-fx (hashtbl-nb-entry table) 1))
		 (set-cdr! bucket (cddr bucket))
		 #t)
		((null? (cdr bucket))
		 #f)
		(else
		 (loop (cdr bucket)))))))))

;*---------------------------------------------------------------------*/
;*    rem-key-hash! ...                                                */
;*---------------------------------------------------------------------*/
(define (rem-key-hash! key table)
   (let* ((hash-eq?   (hashtbl-predicate table))
	  (hash-num   (get-hash-number table key))
	  (vec        (hashtbl-table table))
	  (bucket     (vector-ref vec hash-num))
	  (get-key    (hashtbl-get-key table)))
      (cond
	 ((null? bucket)
	  #f)
	 ((hash-eq? (get-key (car bucket)) key)
	  (hashtbl-nb-entry-set! table (-fx (hashtbl-nb-entry table) 1))
	  (vector-set! vec hash-num (cdr bucket))
	  #t)
	 (else
	  (let loop ((bucket bucket))
	     (cond
		((hash-eq? (get-key (cadr bucket)) key)
		 (hashtbl-nb-entry-set! table (-fx (hashtbl-nb-entry table) 1))
		 (set-cdr! bucket (cddr bucket))
		 #t)
		((null? (cdr bucket))
		 #f)
		(else
		 (loop (cdr bucket)))))))))

;*---------------------------------------------------------------------*/
;*    for-each-hash ...                                                */
;*---------------------------------------------------------------------*/
(define (for-each-hash fun table)
   (let ((vec (hashtbl-table table)))
      (let loop ((i (-fx (hashtbl-size table) 1)))
	 (if (=fx i -1)
	     (unspecified)
	     (begin
		(for-each fun (vector-ref vec i))
		(loop (-fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    hash-table-grows! ...                                            */
;*---------------------------------------------------------------------*/
(define (hash-table-grows! table)
   (let* ((max-size   (hashtbl-max-size table))
	  (size       (hashtbl-size table))
	  (new-size   (*fx size 2))
	  (new-table  (make-vector new-size '()))
	  (old-table  (hash-table->vector table)))
      ;; on reajuste la table de hash
      (hashtbl-size-set!  table new-size)
      (hashtbl-table-set! table new-table)
      ;; on rehash tous les elements du vector dans la nouvelle table.
      (let loop ((i 0))
	 (if (=fx i size)
	     'done
	     (let ((bucket (vector-ref old-table i)))
		(for-each (lambda (obj)
			     (put-hash! obj table))
			  bucket)
		(loop (+fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    string->0..255 ...                                               */
;*---------------------------------------------------------------------*/
(define-inline (string->0..255 string)
   (get_hash_number string))

;*---------------------------------------------------------------------*/
;*    string->0..2^x-1 ...                                             */
;*---------------------------------------------------------------------*/
(define-inline (string->0..2^x-1 string power)
   (get_hash_power_number string power))

;*---------------------------------------------------------------------*/
;*    int->0..255 ...                                                  */
;*---------------------------------------------------------------------*/
(define-inline (int->0..255 int)
   (get_hash_number_from_int int))

;*---------------------------------------------------------------------*/
;*    int->0..2^x-1 ...                                                */
;*---------------------------------------------------------------------*/
(define-inline (int->0..2^x-1 int power)
   (get_hash_power_number_from_int int power))

;*---------------------------------------------------------------------*/
;*    obj->0..255 ...                                                  */
;*---------------------------------------------------------------------*/
(define-inline (obj->0..255 obj)
   (cond
      ((string? obj)
       (string->0..255 obj))
      ((symbol? obj)
       (string->0..255 (symbol->string obj)))
      ((integer? obj)
       (int->0..255 obj))
      ((char? obj)
       (char->integer obj))
      (else
       (get_hash_number_from_pointer obj))))

;*---------------------------------------------------------------------*/
;*    obj->0..2^x-1 ...                                                */
;*---------------------------------------------------------------------*/
(define-inline (obj->0..2^x-1 obj power)
   (cond
      ((string? obj)
       (string->0..2^x-1 obj power))
      ((symbol? obj)
       (string->0..2^x-1 (symbol->string obj) power))
      ((integer? obj)
       (int->0..2^x-1 obj power))
      ((char? obj)
       (char->integer obj))
      (else
       (get_hash_power_number_from_pointer obj power))))



