;;; Register machine simulator for multiple processes.
;;; R.S.Nikhil, MIT, November 27, 1989

;;; ****************************************************************
;;; Hack to get vector functions in Chipmunk Scheme

(enable-language-features)
  (define vector        vector)
  (define vector-set!   vector-set!)
  (define vector-ref    vector-ref)
  (define vector-length vector-length)
  (define make-vector   make-vector)
  (define list->vector  list->vector)
(disable-language-features)

;;; ****************************************************************
;;; PROCESSORS (abbreviation in identifiers: ``P'')
;;; A processor is a collection of 10 registers; respresented as
;;; a vector of length 10.

;;; An alist associating register names with numbers

(define register-list '(
  (pc       . 0)
  (exp      . 1)
  (env      . 2)
  (val      . 3)
  (fun      . 4)
  (unev     . 5)
  (argl     . 6)
  (continue . 7)
  (stack    . 8)
  (pid      . 9)
))

;;; Symbolic names for register numbers

(define p-pc       (cdr (assq 'pc       register-list)))  ; program counter
(define p-exp      (cdr (assq 'exp      register-list)))  ; same as in A&S
(define p-env      (cdr (assq 'env      register-list)))  ; same as in A&S
(define p-val      (cdr (assq 'val      register-list)))  ; same as in A&S
(define p-fun      (cdr (assq 'fun      register-list)))  ; same as in A&S
(define p-unev     (cdr (assq 'unev     register-list)))  ; same as in A&S
(define p-argl     (cdr (assq 'argl     register-list)))  ; same as in A&S
(define p-continue (cdr (assq 'continue register-list)))  ; same as in A&S
(define p-stack    (cdr (assq 'stack    register-list)))  ; same as in A&S
(define p-pid      (cdr (assq 'pid      register-list)))  ; processor identifier

(define (make-processor) (make-vector 10))

(define (p-display p)  (map-vec abbrev-display p))

;;; ****************************************************************
;;; Instruction execution in processors.
;;; When executing an instruction in a processor, the processor may stop,
;;; be suspended (on a locked cell), fork a new processor, etc.  Thus,
;;; it takes a processor list and conses on one or more processors on to that
;;; list.  For most instructions, just the current processor is consed onto the list.

(define (execute-instruction mp p new-p-list)
  (let ((pc (vector-ref p p-pc)))
    (let ((instruction (vector-ref (vector-ref mp mp-code-vec) pc)))
      ((instruction p) mp p pc new-p-list))))

;; Implementation of ASSIGN op code

(define (assign r v)
  (lambda (mp p pc new-p-list)
    (vector-set! p r v)
    (vector-set! p p-pc (1+ pc))
    (cons p new-p-list)))

;; Implementation of SAVE op code

(define (save r)
  (lambda (mp p pc new-p-list)
    (vector-set! p p-stack (cons (vector-ref p r)
                                 (vector-ref p p-stack)))
    (vector-set! p p-pc (1+ pc))
    (cons p new-p-list)))

;; Implementation of RESTORE op code

(define (restore r)
  (lambda (mp p pc new-p-list)
    (let ((stack (vector-ref p p-stack)))
      (vector-set! p r       (car stack))
      (vector-set! p p-stack (cdr stack))
      (vector-set! p p-pc (1+ pc))
      (cons p new-p-list))))

;; Implementation of GOTO op code

(define (goto L)
  (lambda (mp p pc new-p-list)
    (vector-set! p p-pc L)
    (cons p new-p-list)))

;; Implementation of BRANCH op code

(define (branch x L)
  (lambda (mp p pc new-p-list)
    (if x
        (vector-set! p p-pc L)
        (vector-set! p p-pc (1+ pc)))
    (cons p new-p-list)))

;; Implementation of STOP op code

(define (stop)
  (lambda (mp p pc new-p-list)
    new-p-list))

;; Implementation of PERFORM op code

(define (perform *discard*)
  (lambda (mp p pc new-p-list)
    (vector-set! p p-pc (1+ pc))
    (cons p new-p-list)))

;; Implementation of FORK op code

(define (fork starting-pc new-stack-val)
  (lambda (mp p pc new-p-list)
    (let  ((new-p (make-processor-in-mp mp)))
      (vector-set! new-p p-stack new-stack-val)
      (vector-set! new-p p-pc    starting-pc)
      (vector-set! p p-pc (1+ pc))
      (cons p (cons new-p new-p-list)))))

;; Implementation of JOIN op code

(define (join cell)
  (lambda (mp p pc new-p-list)
    (let ((n (- (car cell) 1)))
      (set-car! cell n)
      (cond
       ((> n 0) new-p-list)                    ; die
       (else    (vector-set! p p-pc (1+ pc))   ; go on
	        (cons p new-p-list))))))

;; Implementation of SET-I-CELL! op code

(define (set-I-cell! I-cell v)
  (lambda (mp p pc new-p-list)
    (let ((waiting-processors (I-cell-waiting-list I-cell)))
      (set-I-cell-value! I-cell v)
      (set-I-cell-flag! I-cell)
      (vector-set! p p-pc (1+ pc))
      (cons p (append waiting-processors new-p-list)))))

;; Implementation of GET-I-CELL op code

(define (get-I-cell r x)
  (lambda (mp p pc new-p-list)
    (cond
     ((not (is-I-cell? x)) (vector-set! p r x)
			   (vector-set! p p-pc (1+ pc))
			   (cons p new-p-list))
     ((I-cell-flag-set? x) (vector-set! p r (I-cell-value x))
			   (vector-set! p p-pc (1+ pc))
			   (cons p new-p-list))
     (else (add-to-I-cell-waiting-list x p)
	   new-p-list))))

;; Implementation of RECORD-VALUE op code
;; (used to record final value of program)

(define (record-value v)
  (lambda (mp p pc new-p-list)
    (vector-set! mp mp-final-val v)
    (princ "(Result available at step ")
    (princ (vector-ref mp mp-step-count))
    (princ ")")
    (vector-set! p p-pc (1+ pc))
    (cons p new-p-list)))

;;; ****************************************************************
;;; MULTIPROCESSORS (abbreviation in identifiers: ``MP'')

;;; A multiprocessor consists, mainly, of
;;;    a code sequence (a vector of instructions)
;;;    a list of (logical) processors
;;; The basic simulator action is:

;;;    processor-list <- (initial ``main'' processor)
;;;    WHILE non-empty processor-list
;;;        FOREACH processor IN processor-list
;;;            execute one instruction

;;; The inner (FOREACH) loop is called a ``step'' thru all the processors.
;;; Note: the processor-list grows and shrinks because instructions may stop,
;;; suspend, and fork processors.

;;; Additional components of the multiprocessor state are used to collect
;;; execution statistics.
;;; ``Max parallelism''     = max length of processor list
;;; ``Parallelism Profile'' = graph of length of processor list vs. step number
;;;                           Stored as a list of lengths (numbers) in *reverse* order.

(define (make-multiprocessor controller-code)
  (vector (compile-code register-list controller-code) ; 0 code vector
          (length register-list)                       ; 1 number of registers in each processor
          '()                                          ; 2 main processor
          '()                                          ; 3 processor-list
          '*undefined*                                 ; 4 final value
          
          0                                            ; 5 step count
          0                                            ; 6 processor count
          0                                            ; 7 maximum parallelism
          0                                            ; 8 total instruction count
          '()                                          ; 9 parallelism profile
          #f                                           ; 10 trace flag (for debugging only)
          ))

;;; Symbolic names for multiprocessor state components

(define mp-code-vec        0)
(define mp-p-size          1)
(define mp-main-p          2)
(define mp-p-list          3)
(define mp-final-val       4)
(define mp-step-count      5)
(define mp-p-count         6)
(define mp-max-parallelism 7)
(define mp-tot-instrs      8)
(define mp-par-prof        9)
(define mp-trace-flag      10)

;;; create a new processor in a multiprocessor

(define (make-processor-in-mp mp)
  (let ((p (make-processor))                    ; the new processor
        (pnum (vector-ref mp mp-p-count)))      ; the new processor's id
    (vector-set! p p-pid pnum)                  ; set processor's id
    (vector-set! mp mp-p-count (+ pnum 1))      ; increment processor count
    p))                                         ; return the processor

;;; Display results of multiprocessor (value and statistics)

(define (mp-results mp)
  (newline)
  (princ "Value: ") (user-print (vector-ref mp mp-final-val))
  (newline) (princ "Steps ...................................... ")
            (princ (vector-ref mp mp-step-count))
  (newline) (princ "Total instructions ......................... ")
            (princ (vector-ref mp mp-tot-instrs))
  (newline) (princ "Processes forked ........................... ")
            (princ (vector-ref mp mp-p-count))
  (newline) (princ "Max parallelism ............................ ")
            (princ (vector-ref mp mp-max-parallelism))
  (newline) (princ "Average parallelism (instructions/step) .... ")
            (princ  (/ (vector-ref mp mp-tot-instrs)
                      (vector-ref mp mp-step-count)))
  (newline)
  'done)

;;; Run the program in a multiprocessor to completion.
;;; Then, display results

(define (run-mp mp)
;  (define t1 0)
;  (define t2 0)
  (define p-list (vector-ref mp mp-p-list))
  (define new-p-list '())
  (define tot-instrs 0)
  (define max-parallelism 0)
  (define step-count 0)
  (define par-prof '())
  (define j 1)
  (define n 0)
  
;  (set! t1 (time))
  (newline)
  (let do-one-step ()    ; while p-list non-empty, do steps
    (cond
     (p-list (set! step-count (1+ step-count))
	     (vector-set! mp mp-step-count step-count)
	     (set! n (length p-list))
	     (set! new-p-list '())
	     (let do-instruction-in-processor ()    ; for each processor, do an instruction
	       (cond
		(p-list (set! new-p-list (execute-instruction mp (car p-list) new-p-list))
			(set! p-list (cdr p-list))
			(do-instruction-in-processor))))
	     (set! p-list new-p-list)
	     (set! tot-instrs (+ tot-instrs n))
	     (set! max-parallelism (max max-parallelism n))
	     (set! par-prof (cons n par-prof))
	     (if (= j 100)
		 (sequence (princ ".") (set! j 1))
		 (set! j (1+ j)))
	     (do-one-step))))
;  (set! t2 (time))

  (vector-set! mp mp-tot-instrs tot-instrs)
  (vector-set! mp mp-par-prof par-prof)
  (vector-set! mp mp-max-parallelism max-parallelism)
  
  (mp-results mp)
;  (princ "Elapsed time ")  (princ (/ (- t2 t1) 60))  (princ " s")  (newline)
)

;;; Load an expression to be executed into a multiprocessor.

(define (load-mp mp expr)
  (let ((p0 (make-processor)))
    (vector-set! mp mp-step-count      0)
    (vector-set! mp mp-p-count         1)
    (vector-set! mp mp-max-parallelism 0)
    (vector-set! mp mp-tot-instrs      0)
    (vector-set! mp mp-par-prof        '())
    (vector-set! mp mp-main-p          p0)
    (vector-set! mp mp-p-list          (list p0))
    (vector-set! mp mp-final-val      '*undefined*)
    (vector-set! p0 p-pc 0)
    (vector-set! p0 p-exp expr)
    (vector-set! p0 p-env *the-global-environment*)
    'initialized))

;;; Extract the parallelism profile from the multiprocessor.

(define (get-pp mp)
    (reverse (vector-ref mp mp-par-prof)))

;;; ****************************************************************
;;; Multiprocessor user interface.

;;; Pick one of these parallelism-profile plotters:
;;; Warning: only one of these lines should be UNcommented!

(define (plot-pp mp) (ascii-plot-pp mp))
; (define (plot-pp mp) (mac-plot-pp mp))
; (define (plot-pp mp) (chipmunk-plot-pp mp))

;;; Load a multiprocessor with an expression and run it.

(define (load-and-go mp expr)
  (load-mp mp expr)
  (run-mp mp)
  (plot-pp mp))

;;; ****************************************************************
;;; Compilation of controller code
;;; Given
;;;    An alist of register names/register numbers, and
;;;    A  source-instruction list (list of labels and instructions),
;;; Returns
;;;    A vector of object-instructions
;;;      each obj instr is the value of (LAMBDA (P) SRC-INSTR')
;;;      where SRC-INSTR' is a transformation of a source instr, as follows:
;;;         labels           ==> addresses (vector indices)
;;;         (FETCH register) ==> (VECTOR-REF P register-number)
;;;         register         ==> register-number

(define (compile-code reg-regnum-alist src-instrs)
  (let ((x (collect-labels src-instrs)))
    (let ((label-pc-alist (car x))
	  (src-instrs-rev  (cdr x)))
      (let ((obj-instrs (reverse
			 (mapcar
			  (lambda (x)
			    (convert
			     reg-regnum-alist
			     label-pc-alist 
			     x))
			  src-instrs-rev))))
	(list->vector obj-instrs)))))

;;; -------------------------------------------------------------------
;;; Given an instruction list,
;;; Returns (LABEL-PC-ALIST . INSTRUCTIONS-REVERSED-NO-LABELS)
;;;    An alist of labels and their pc-values
;;;    The list of instructions in reverse order, with labels removed.

(define (collect-labels src-instrs)
  (define (loop src-instrs pc label-pc-alist src-instrs-rev)
    (cond
     ((null? src-instrs)          (cons label-pc-alist src-instrs-rev))
     ((symbol? (car src-instrs))  (loop (cdr src-instrs)
                                        pc
                                        (cons (cons (car src-instrs) pc) label-pc-alist)
                                        src-instrs-rev))
     (else                        (loop (cdr src-instrs)
                                        (+ pc 1)
                                        label-pc-alist
                                        (cons (car src-instrs) src-instrs-rev)))))
  (loop src-instrs 0 '() '()))

;;; -------------------------------------------------------------------
;;; Given a machine-code INSTRUCTION, REG-REGNUM-ALIST and LABEL-PC-ALIST,
;;;    converts the source instruction into an object-code instruction.

(define (convert reg-regnum-alist label-pc-alist instr)
  
  (define (traverse instr)
    (cond
     ((symbol? instr) (convert-symbol reg-regnum-alist label-pc-alist instr))
     ((pair? instr) (if (eq? (car instr) 'fetch)
                        (list 'vector-ref 'p (convert-symbol reg-regnum-alist
							     label-pc-alist
							     (cadr instr)))
                        (cons (traverse (car instr)) (traverse (cdr instr)))))
     (else instr)))
  
  (eval  
   (list 'lambda
         '(p)
         (cons (car instr)
               (traverse (cdr instr))))
   user-initial-environment)
  )

;;; Given a symbol X, the REGISTER-REGISTERNUMBER-ALIST and LABEL-PC-ALIST,
;;;    returns J if (X.J) is in the register alist
;;;    returns J if (X.J) is in the label alist
;;;    returns X otherwise

(define (convert-symbol reg-regnum-alist label-pc-alist x)
  (let ((xj (assq x reg-regnum-alist)))
    (if (pair? xj)
        (cdr xj)
        (let ((xj (assq x label-pc-alist)))
          (if (pair? xj)
              (cdr xj)
              x)))))

;;; ****************************************************************
;;; General help functions

;;; Vector version of MAP

(define (map-vec f v)
  (define n (vector-length v))
  (define (loop w j)
    (if (= j n)
        w
        (sequence (vector-set! w j (f (vector-ref v j)))
		  (loop w (+ j 1)))))
  (loop (make-vector n) 0))

;;; Transforms structure so that lists longer than 3 elements are truncated with ellipses
      
(define (abbrev-display l)
    (if (not (pair? l))
        l
        (abbrev-list l 3)))

(define (abbrev-list l n)
  (cond
    ((not (pair? l)) l)
    ((= n 0)  '( ... ) )
    (else (cons (abbrev-display (car l))
                (abbrev-list (cdr l) (- n 1))))))
