;;; Plotting parallelism profiles.
;;; Nikhil, November 1989

;;; Two versions are defined here:
;;;    ascii-plot-pp    --- for ascii terminals
;;;    mac-plot-pp      --- for MacScheme
;;; To be done:
;;;    chipmunk-plot-pp --- for HP Chipmunks

;;; ****************************************************************
;;; ASCII terminal version:

;;; Show the parallelism profile stored in a multiprocessor (i.e., from its last run)
;;; averaging the parallelism profiles in STEPSIZE steps together.

(define (ascii-plot-pp mp)
  (define stepsize (max 1 (round (/ (vector-ref mp mp-step-count)
				    20))))    ;;; Display max 20 lines on terminal
  (define loop (lambda (step tot pp)
    (if (null? pp)
        'done
        (let  ((n (car pp)))
          (if (zero? (remainder step stepsize))
              (sequence
                (newline)
                (princ step)
                (princ " : ")
                (ascii-print-bar (round (/ (+ tot n) stepsize)))
                (loop (+ step 1) 0 (cdr pp)))
              (loop (+ step 1) (+ tot n) (cdr pp)))))))
  (princ "Parallelism profile : ")
  (loop 1 0 (get-pp mp)))

(define (ascii-print-bar n)
   (if (zero? n)
       'done
       (sequence (princ "*")
		 (ascii-print-bar (- n 1)))))

;;; ****************************************************************
;;; MacScheme version:

(define (pix-scale max-pix max-val)
  (if (<= max-pix max-val)
      (let ((foos/pix (ceiling (/ max-val max-pix))))
	(let ((pix/foo  (/ 1 foos/pix))
	      (pix-tick (round (/ max-pix 10))))
	  (let ((foo-tick (round (* pix-tick foos/pix))))
	    (list foos/pix pix/foo foo-tick pix-tick))))
      (let ((pix/foo  (floor (/ max-pix max-val))))
	(let ((foos/pix (/ 1 pix/foo))
	      (pix-tick (round (/ max-pix 10))))
	  (let ((foo-tick (round (* pix-tick foos/pix))))
	    (list foos/pix pix/foo foo-tick pix-tick))))))

(define (do-x-ticks step-tick x-tick)
  (define (loop x step)
    (if (>= x 440)
        'done
        (sequence
         (draw-line x 244 x 249)
         (move-to x 259)
         (draw-string (number->string step))
         (loop (+ x x-tick) (+ step step-tick)))))
  (loop 40 0))
    
(define (do-y-ticks proc-tick y-tick)
  (define (loop y proc)
    (if (< y 5)
        'done
        (sequence
         (draw-line 36 y 40 y)
         (move-to 5  (+ y 5))
         (draw-string (number->string proc))
         (loop (- y y-tick) (+ proc proc-tick)))))
  (loop 244 0))

;;; Plots the parallelism profile assuming more than one step/pixel
    
(define (mac-plot-pp-aux1 pp pix/proc steps/pix)
  (define (loop pp step x j cum)
    (if (null? pp)
        'done
        (let  ((n     (car pp)))
          (if (< j steps/pix)
              (loop (cdr pp) (1+ step) x (+ j 1) (+ cum n))
              (sequence
               (move-to x 244)
               (line-to x (- 245
                             (round (* pix/proc (/ (+ cum n) steps/pix)))))
               (loop (cdr pp) (1+ step) (+ x 1) 1 0))))))
  (loop pp 1 40 1 0))

;;; Plots the parallelism profile assuming fewer than one step/pixel
  
(define (mac-plot-pp-aux2 pp pix/proc pix/step)
  (define (loop pp step x)
    (if (null? pp)
        'done
        (let ((n     (car pp)))
	  (let ((y     (* n pix/proc))
		(next-x (+ x pix/step)))
	    (paint-rect x (- 245 y) next-x 244)
	    (loop (cdr pp) (1+ step) next-x)))))
  (loop pp 1 40))
  
(define (mac-plot-pp mp)
  (let* ((max-steps (max 10 (vector-ref mp mp-step-count)))
         (x-parms (pix-scale 400 max-steps))
         (steps/pix (car x-parms))
         (pix/step  (cadr x-parms))
         (step-tick (caddr x-parms))
         (x-tick     (cadddr x-parms))
         
         (max-procs (max 10 (vector-ref mp mp-max-parallelism)))
         (y-parms (pix-scale 240 max-procs))
         (procs/pix (car y-parms))
         (pix/proc  (cadr y-parms))
         (proc-tick (caddr y-parms))
         (y-tick    (cadddr y-parms)))
    (clear-graphics)
    (draw-picture
      (move-to 40 244)
      (pen-size 2 2)
      (line-to 40 5)
      (line-to 439 5)
      (line-to 439 244)
      (line-to 40 244)
      (pen-size 1 1)
      (move-to 150 20)
      (draw-string "Processes vs. Step")
      (do-x-ticks step-tick x-tick)
      (do-y-ticks proc-tick y-tick)
      (if (> steps/pix 1)
          (mac-plot-pp-aux1 (get-pp mp) pix/proc steps/pix)
          (mac-plot-pp-aux2 (get-pp mp) pix/proc pix/step)))
    'done))

;;; ****************************************************************
