;;; Controller code for explicit-control evaluator with FUTURE construct
;;; for parallel evaluation.
;;; Based on the regular explicit-control evaluator from Abelson and Sussman.
;;; R.S.Nikhil, MIT, November 27, 1989

;;; ****************************************************************
;;; ****************************************************************
;;; Processors have nine registers:
;;;   EXP, ENV, VAL, FUN, UNEV, ARGL, CONTINUE,
;;;   PROGRAM-COUNTER, STACK
;;; ****************************************************************
;;; New instructions:

;;;    (fork <starting-pc> <stack-contents>)
;;; Forks a new processor with program-counter and stack initialized
;;; with given values.

;;;    (join cell)
;;; Decrements the counter in the car of the cell.
;;; If it is still non-zero, die, else continue

;;; ****************************************************************
;;; ****************************************************************
;;; Here is the controller code.

(define future-controller-code '(

;;; ****************************************************************
;;; Top-level evaluation of an expression
;;; Assume: stack: (exp env)
;;; Effect: stop after storing eval(exp,env) in VAL register, and recording
;;; the result.

TOP-LEVEL-EVAL
  (assign continue RECORD-RESULT)
  (goto EVAL-DISPATCH)

RECORD-RESULT
  (record-value (fetch val))
  (stop)

;;; ****************************************************************
;;; The regular EVAL function.
;;; Assume: EXP:e, ENV:E, CONTINUE:L
;;; Effect: at L; stack unchanged; VAL:  [[ e ]]E

EVAL-DISPATCH
  (branch (self-evaluating? (fetch exp)) EV-SELF-EVAL)
  (branch (symbol? (fetch exp))          EV-SYMBOL)
  (branch (define? (fetch exp))          EV-DEFINE)
  (branch (quoted? (fetch exp))          EV-QUOTE)
  (branch (lambda? (fetch exp))          EV-LAMBDA)
  (branch (let? (fetch exp))             EV-LET)
  (branch (if? (fetch exp))              EV-IF)
  (branch (cond? (fetch exp))            EV-COND)
  (branch (application? (fetch exp))     EV-APPLICATION)

  (perform (user-print 'unknown-expression-type-error))
  (perform (user-print (fetch exp)))
  (stop)

;;; ****************************************************************
;;; Numbers, strings, etc.
EV-SELF-EVAL
  (assign val (fetch exp))
  (goto (fetch continue))

;;; ****************************************************************
EV-SYMBOL
  (assign val (lookup (fetch exp) (fetch env)))
  (goto (fetch continue))

;;; ****************************************************************
;;; Definitions: (DEFINE x e) or (DEFINE (x ..args..) ..es..)

EV-DEFINE
  ; first convert (DEFINE (x ...) ..es..) to (DEFINE x e) form, if nec'y
  (assign exp (desugar-define (fetch exp)))

  (save continue)

  (assign exp (cdr (fetch exp)))
  (assign fun (car (fetch exp)))       ; x from (DEFINE x e)
  (save fun)
  (save env)
  (assign exp (cdr (fetch exp)))
  (assign exp (car (fetch exp)))       ; e from (DEFINE x e)
  (assign continue DEFINE-EXPR-DONE)
  (goto EVAL-DISPATCH)
DEFINE-EXPR-DONE
  (restore env)
  (restore fun)                        ; x from (DEFINE x e)
  (perform (define-var-value! (fetch fun) (fetch val) (fetch env)))
  (assign val (fetch fun))

  (restore continue)
  (goto (fetch continue))

;;; ****************************************************************
EV-QUOTE
  (assign val (cdr (fetch exp)))
  (assign val (car (fetch val)))
  (goto (fetch continue))

;;; ****************************************************************
EV-LAMBDA
  (assign val (make-procedure (fetch exp) (fetch env)))
  (goto (fetch continue))

;;; ****************************************************************
EV-LET
  (assign exp (desugar-let (fetch exp)))
  (goto EV-APPLICATION)

;;; ****************************************************************
;;; ****************************************************************
;;; Assume:  EXP: (FUTURE e);  ENV: E;  CONTINUE: label
;;; Effect:  Allocate new I-cell,
;;;          Fork process to evaluate e in E, store result in I-cell
;;;          VAL: I-cell; continue at label
EV-FUTURE







; ... missing code for (FUTURE e) forms ...
; Note: an I-cell can be allocated using the
; ``primitive'' function (make-I-cell)









;;; ****************************************************************
;;; Assume:  EXP: (TOUCH e);  ENV: E;  CONTINUE: label
;;; Effect:  evaluate e in E to get an I-cell.
;;;          VAL: value in I-cell; continue at label

EV-TOUCH







; ... missing code for (TOUCH e) forms ...









;;; ****************************************************************
;;; Forked expression-evaluation begins here.
;;; Assume: stack: exp,env,I-cell, ...
;;; Effect: eval(exp,env), store into I-cell, stop.

FORKED-EVAL
  (restore exp)
  (restore env)
  (assign continue SET-VALUE)
  (goto EVAL-DISPATCH)
SET-VALUE
  (restore exp)                           ; the I-cell
  (set-I-cell! (fetch exp) (fetch val))
  (stop)

;;; ****************************************************************
;;; EXP: (e1 ... eN);  ENV:E;  CONTINUE: L
EV-APPLICATION
  (save continue)

  (assign unev (cdr (fetch exp)))  ; (e2 ... eN)
  (assign exp (car (fetch exp)))   ; e1
  (save env)
  (save unev)
  (assign continue EVAL-ARGS)
  (goto EVAL-DISPATCH)
EVAL-ARGS
  (restore unev)
  (restore env)
  (assign fun (fetch val))
  (save fun)
  (assign argl '())
  (goto EVAL-ARG-LOOP)

EVAL-ARG-LOOP
  (save argl)
  (assign exp (car (fetch unev)))
  (branch (last-operand? (fetch unev)) EVAL-LAST-ARG)
  (save env)
  (save unev)
  (assign continue ACCUMULATE-ARG)
  (goto EVAL-DISPATCH)
ACCUMULATE-ARG
  (restore unev)
  (restore env)
  (restore argl)
  (assign argl (cons (fetch val) (fetch argl)))
  (assign unev (cdr (fetch unev)))
  (goto EVAL-ARG-LOOP)
EVAL-LAST-ARG
  (assign continue ACCUMULATE-LAST-ARG)
  (goto EVAL-DISPATCH)
ACCUMULATE-LAST-ARG
  (restore argl)
  (assign argl (cons (fetch val) (fetch argl)))
  (restore fun)
  (goto APPLY-DISPATCH)

;;; ****************************************************************
;;; The apply function
;;; Assumes: FUN: a procedure; ARGL: reversed list of args ; STACK: L,...
;;; Effect: PC:L; STACK: ...; VAL: value of application of proc to args

APPLY-DISPATCH
  (branch (primitive-procedure? (fetch fun))      PRIMITIVE-APPLY)
  (branch (compound-procedure? (fetch fun))       COMPOUND-APPLY)

UNKNOWN-PROCEDURE-TYPE-ERROR
  (perform (user-print 'unknown-procedure-type-error))
  (perform (user-print (fetch fun)))
  (stop)

;;; ****************************************************************

PRIMITIVE-APPLY
  (branch (prim-car?      (fetch fun)) APPLY-CAR)
  (branch (prim-cdr?      (fetch fun)) APPLY-CDR)
  (branch (prim-cons?     (fetch fun)) APPLY-CONS)

  (branch (prim-pair??    (fetch fun)) APPLY-PAIR?)
  (branch (prim-null??    (fetch fun)) APPLY-NULL?)
  (branch (prim-not?      (fetch fun)) APPLY-NOT)
  (branch (prim-truncate? (fetch fun)) APPLY-TRUNCATE)
  (branch (prim-floor?    (fetch fun)) APPLY-FLOOR)

  (branch (prim-eq??      (fetch fun)) APPLY-EQ?)
  (branch (prim-and?      (fetch fun)) APPLY-AND)
  (branch (prim-or?       (fetch fun)) APPLY-OR)
  (branch (prim-+?        (fetch fun)) APPLY-PLUS)
  (branch (prim--?        (fetch fun)) APPLY-MINUS)
  (branch (prim-*?        (fetch fun)) APPLY-TIMES)
  (branch (prim-/?        (fetch fun)) APPLY-DIVIDE)
  (branch (prim-<?        (fetch fun)) APPLY-LESS-THAN)
  (branch (prim->?        (fetch fun)) APPLY-GREATER-THAN)
  (branch (prim-=?        (fetch fun)) APPLY-=)
  (branch (prim-max?      (fetch fun)) APPLY-MAX)
  (branch (prim-min?      (fetch fun)) APPLY-MIN)

UNKNOWN-PRIMITIVE-ERROR
  (perform (user-print 'unknown-primitive-error))
  (perform (user-print (fetch fun)))
  (stop)

;;; ----------------
APPLY-CAR
  (assign val (car (fetch argl)))  ; the cons cell
  (assign val (car (fetch val)))   ; its car
  (restore continue)
  (goto (fetch continue))

APPLY-CDR
  (assign val (car (fetch argl)))  ; the cons cell
  (assign val (cdr (fetch val)))   ; its cdr
  (restore continue)
  (goto (fetch continue))

APPLY-CONS
  ; return the CONS of the two frame-cells in the frame
  ; i.e., don't wait for either car or cdr to be computed.
  (assign val (car (fetch argl)))    ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))     ; first arg
  (assign val (cons (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

;;; ----------------
;;; Monadic functions

APPLY-PAIR?
  (assign val (car (fetch argl)))
  (assign val (pair? (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-NULL?
  (assign val (car (fetch argl)))
  (assign val (null? (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-NOT
  (assign val (car (fetch argl)))
  (assign val (not (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-TRUNCATE
  (assign val (car (fetch argl)))
  (assign val (truncate (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-FLOOR
  (assign val (car (fetch argl)))
  (assign val (floor (fetch val)))
  (restore continue)
  (goto (fetch continue))

;;; ----------------
;;; Dyadic functions

APPLY-EQ?
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (eq? (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-AND
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (and (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-OR
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (or (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-PLUS
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (+ (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-MINUS
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (- (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-TIMES
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (* (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-DIVIDE
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (/ (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-LESS-THAN
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (< (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-GREATER-THAN
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (> (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-=
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (= (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-MAX
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (max (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-MIN
  (assign val (car (fetch argl)))         ; second arg
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))          ; first arg
  (assign val (min (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

;;; ****************************************************************
COMPOUND-APPLY
  (assign env (make-bindings (fetch fun) (fetch argl)))
;                                        Warning: argl is in rev order.
  (assign unev (procedure-body (fetch fun)))

;;; ****************************************************************
;;; Eval-sequence is the same as sequential case

EVAL-SEQUENCE
  (assign exp (car (fetch unev)))
  (assign val (cdr (fetch unev)))
  (branch (null? (fetch val)) LAST-EXP)
  (save unev)
  (save env)
  (assign continue EVAL-SEQUENCE-CONTINUE)
  (goto EVAL-DISPATCH)

EVAL-SEQUENCE-CONTINUE
  (restore env)
  (restore unev)
  (assign unev (cdr (fetch unev)))
  (goto EVAL-SEQUENCE)

LAST-EXP
  (restore continue)
  (goto EVAL-DISPATCH)

;;; ****************************************************************
;;; Evaluation of conditionals is the same as before

EV-IF
  (save continue)
  (assign continue EV-IF-DECIDE)
  (assign unev (cdr (fetch exp)))
  (assign exp (car (fetch unev)))
  (assign unev (cdr (fetch unev)))
  (save env)
  (save unev)
  (goto EVAL-DISPATCH)

EV-IF-DECIDE
  (restore unev)
  (restore env)
  (restore continue)
  (branch (null? (fetch val)) EV-ELSE-PART)
EV-THEN-PART
  (assign exp (car (fetch unev)))
  (goto EVAL-DISPATCH)
EV-ELSE-PART
  (assign exp (cdr (fetch unev)))
  (assign exp (car (fetch exp)))
  (goto EVAL-DISPATCH)

;;; ----------------
;;; (cond ((p1 ..e1s..) (p2 ..e2s..) ... (else ..eNs..)))
EV-COND
  (save continue)
  (assign continue EVCOND-DECIDE)
  (assign unev (cdr (fetch exp)))

EVCOND-PRED
  (branch (null? (fetch unev)) EVCOND-RETURN-NIL)
  (assign exp (car (fetch unev)))
  (branch (else-clause? (fetch exp)) EVCOND-ELSE-CLAUSE)
  (save env)
  (save unev)
  (assign exp (car (fetch exp)))    ; predicate
  (goto EVAL-DISPATCH)

EVCOND-DECIDE
  (restore unev)
  (restore env)
  (branch (fetch val) EVCOND-TRUE-PREDICATE)
  (assign unev (cdr (fetch unev)))
  (goto EVCOND-PRED)

EVCOND-TRUE-PREDICATE
  (assign exp (car (fetch unev)))
EVCOND-ELSE-CLAUSE
  (assign unev (cdr (fetch exp)))
  (goto EVAL-SEQUENCE)

EVCOND-RETURN-NIL
  (restore continue)
  (assign val '())
  (goto (fetch continue))

;;; ****************************************************************
;;; End of controller sequence

))

;;; ----------------------------------------------------------------
;;; A particular multiprocess simulator with futures called F-EVAL

(define f-eval
  (let ((mp (make-multiprocessor future-controller-code)))
    (lambda (e) (load-and-go mp e))))

;;; ----------------------------------------------------------------
;;; Having compiled the controller code into the multiprocessor,
;;; redefine it to '() to save space.  This is purely an efficiency hack.

(define future-controller-code '())

;;; ****************************************************************

;;; Example expression-type predicate to act as guide for FUTURE?
;;;  and TOUCH? predicates

(define (cond?   exp) (has-type? exp 'cond))
