;*              Copyright 1992 Digital Equipment Corporation
;*                         All Rights Reserved
;*
;* Permission to use, copy, and modify this software and its documentation is
;* hereby granted only under the following terms and conditions.  Both the
;* above copyright notice and this permission notice must appear in all copies
;* of the software, derivative works or modified versions, and any portions
;* thereof, and both notices must appear in supporting documentation.
;*
;* Users of this software agree to the terms and conditions set forth herein,
;* and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
;* right and license under any changes, enhancements or extensions made to the
;* core functions of the software, including but not limited to those affording
;* compatibility with other hardware or software environments, but excluding
;* applications which incorporate this software.  Users further agree to use
;* their best efforts to return to Digital any such changes, enhancements or
;* extensions that they make and inform Digital of noteworthy uses of this
;* software.  Correspondence should be provided to Digital at:
;* 
;*			Director, Cambridge Research Lab
;*			Digital Equipment Corp
;*			One Kendall Square, Bldg 700
;*			Cambridge MA 02139
;* 
;* This software may be distributed (but not offered for sale or transferred
;* for compensation) to third parties, provided such third parties agree to
;* abide by the terms and conditions of this notice.
;* 
;* THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
;* WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
;* MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
;* CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
;* DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;* PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
;* ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
;* SOFTWARE.

; $Id: bobxlt.scm,v 1.23 1993/01/12 02:39:23 jmiller Exp $

(define (bob->dylan string error-handler)
  (call-with-current-continuation
   (lambda (exit-from-bob->dylan)
     (let ((tree
	    (bob string (lambda args
			  (exit-from-bob->dylan
			   (apply error-handler args))))))
       (make-begin
	`(,@bob-preamble
	  ,(expr->dylan tree)))))))

(define bob-preamble
  '((DEFINE-GENERIC-FUNCTION /BOB/AS (OBJ CLASS))
    (ADD-METHOD /BOB/AS (METHOD (OBJ CLASS) (AS CLASS OBJ)))
    (ADD-METHOD /BOB/AS
		(METHOD((OBJ <STRING>) (CLASS <NUMBER>)) (AS CLASS OBJ)))
    (ADD-METHOD /BOB/AS (METHOD (OBJ (CLASS <NUMBER>)) obj class #F))
    (DEFINE-GENERIC-FUNCTION /BOB/RIGHT-SHIFT (OBJ1 OBJ2))
    (ADD-METHOD /BOB/RIGHT-SHIFT (METHOD (OBJ1 OBJ2) (ASH OBJ1 (- OBJ2))))
    (DEFINE-GENERIC-FUNCTION /BOB/NOT-EQUAL (OBJ1 OBJ2))
    (ADD-METHOD /BOB/NOT-EQUAL (METHOD (OBJ1 OBJ2) (NOT (= OBJ1 OBJ2))))))

(define (make-begin exprs)
  (define (simplify-if-begin expr)
    (if (and (pair? expr) (eq? (car expr) 'begin))
	(cdr expr)
	(list expr)))
  (let ((simplified (apply append (map simplify-if-begin exprs))))
    (cond ((null? simplified) '#F)
	  ((null? (cdr simplified)) (car simplified))
	  (else `(BEGIN ,@simplified)))))

(define (literal? expr)
  (and (pair? expr)
       (eq? (car expr) 'literal)))

(define (literal->value expr)
  (cadr expr))

(define (keyword-node->keyword e)
  (cadr e))

(define (symbol-node->symbol e)
  (cadr e))

(define (extract-literals e)
  (case (car e)
    ((LITERAL) (literal->value e))
    ((KEYWORD) (keyword-node->keyword e))
    ((SYMBOL) (symbol-node->symbol e))
    ((CONSTANT-LIST CONSTANT-VECTOR)
     (map extract-literals (cdr e)))
    (else (error "unknown literal" e))))

(define (expr->dylan expr)
  (if (symbol? expr)
      expr				; variable
      (case (car expr)
	((SETTER) expr)			; "setter variable"
	((LITERAL) (literal->value expr))
	((SYMBOL) `(QUOTE ,(symbol-node->symbol expr)))
	((KEYWORD) (keyword-node->keyword expr))
	((CONSTANT-LIST)
	 `(QUOTE ,(extract-literals expr)))
	((CONSTANT-VECTOR)
	 `(QUOTE ,(list->vector (extract-literals expr))))
	((UNOP)
	 `(,(unop->name (cadr expr))
	   ,(expr->dylan (caddr expr))))
	((DECLARE) (declare->dylan expr))
	((GENERIC-FUNCTION) (generic-function->dylan expr))
	((METHOD) (method->dylan expr))
	((CASE)
	 `(SELECT ,(expr->dylan (list-ref expr 1))
		  ,(expr->dylan (list-ref expr 2))
		  ,@(map (lambda (clause)
			   `(,(map expr->dylan (list-ref clause 1))
			     ,(expr->dylan (list-ref clause 2))))
			 (list-ref expr 3))
		  ,@(if (list-ref expr 4)
			`((ELSE: ,(expr->dylan (list-ref expr 4))))
			`())))
	((FOR) (for->dylan expr))
	((FOREACH) (foreach->dylan expr))
	((IF) `(IF ,(expr->dylan (list-ref expr 1))
		   ,(expr->dylan (list-ref expr 2))
		   ,@(map expr->dylan (list-tail expr 3))))
	((ADD-SLOT) (add-slot->dylan expr))
	((ADD-METHOD) (add-method->dylan expr))
	((CLASS) (class->dylan expr))
	((VALUES) `(VALUES ,@(list-ref expr 1)))
	((BINOP) `(,(binop->name (list-ref expr 1))
		   ,(expr->dylan (list-ref expr 2))
		   ,(expr->dylan (list-ref expr 3))))
	((ASSIGN) `(SET! ,(list-ref expr 1) ,(expr->dylan (list-ref expr 2))))
	((SETTER-CALL)
	 `(SET! (,(list-ref expr 1) ,@(args->dylan (list-ref expr 2)))
		,(expr->dylan (list-ref expr 3))))
	((CALL) `(,(expr->dylan (list-ref expr 1))
		  ,@(args->dylan (list-ref expr 2))))
	((ELEMENT)
	 `(ELEMENT ,(expr->dylan (list-ref expr 1))
		   ,@(map expr->dylan (list-ref expr 2))))
	((PROTECT) `(UNWIND-PROTECT ,(expr->dylan (list-ref expr 1))
				    ,(expr->dylan (list-ref expr 2))))
	((RERAISE) `(__BOB__RERAISE))
	((UNIT) (unit->dylan expr))
	(else (error "Unknown parse tree item" expr)))))

(define unop-table
  '((BITWISE-NEGATE . LOGNOT)
    (LOGICAL-NEGATE . NOT)
    (MINUS . -)))

(define binop-table
  `((ADD . +)
    (AS . /BOB/AS)
    (BITWISE-AND . LOGAND)
    (BITWISE-OR . LOGIOR)
    (DIVIDE . /)
    (GREATER-THAN . >)
    (GREATER-THAN-OR-EQUAL . >=)
    (LEFT-SHIFT . ASH)
    (LESS-THAN . <)
    (LESS-THAN-OR-EQUAL . <=)
    (LOGICAL-AND . AND)
    (LOGICAL-EQUAL . =)
    (LOGICAL-OR . OR)
    (MINUS . -)
    (MULTIPLY . *)
    (REMAINDER . REMAINDER)
    (RIGHT-SHIFT . /BOB/RIGHT-SHIFT)
    (UNEQUAL . /BOB/NOT-EQUAL)
    (XOR . LOGXOR)))

(define (op->name name op-table)
  (lambda (op)
    (let ((result (assq op op-table)))
      (if result
	  (cdr result)
	  (error (string-append name ": Unknown operator") op)))))

(define unop->name (op->name "UNOP->NAME" unop-table))
(define binop->name (op->name "BINOP->NAME" binop-table))

(define (unit->dylan expr)
  (let ((declarations (list-ref expr 1))
	(expressions (list-ref expr 2)))
    (define (declared-vars declaration)
      (let ((variables (list-ref declaration 1))
	    (rest (list-ref declaration 2)))
	`(,@variables
	  ,@(if rest `((,rest <object>)) `()))))
    (define (var->name var) (list-ref var 1))
    (define (var->type var) (list-ref var 2))
    (define (new-var old)
      (if (eq? '<object> (var->type old))
	  (rename (var->name old))
	  (list (rename (var->name old)) (expr->dylan (var->type old)))))
    (define (new-name var) (rename (var->name var)))
    (define (rename old)
      (if (symbol? old)
	  (string->symbol
	   (string-append "/BOB/" (symbol->string old)))
	  `(SETTER ,(rename (list-ref old 1)))))
    (make-begin
     `(,@(map (lambda (declaration)
		(let ((vars (declared-vars declaration)))
		  `(BIND ((,@(map new-var vars)
			  ,(expr->dylan (list-ref declaration 3))))
			 ,@(map (lambda (var)
				  `(DEFINE ,(var->name var) ,(new-name var)))
				vars))))
	      declarations)
       ,@(map expr->dylan expressions)))))

(define (method->dylan expr)
  (let ((requireds (map variable->dylan (list-ref expr 1)))
	(next (list-ref expr 2))
	(rest (list-ref expr 3))
	(keywords (list-ref expr 4))
	(body (list-ref expr 5)))
    (define (expand-keyword keyword-tree)
      (let ((keyword (list-ref keyword-tree 1))
	    (known-as (list-ref keyword-tree 2))
	    (default (list-ref keyword-tree 3)))
	(cond ((and known-as default)
	       `(,(cadr keyword) ,known-as ,(expr->dylan default)))
	      (known-as `(,(cadr keyword) ,known-as))
	      (default `(,(keyword->symbol keyword)
			 ,(expr->dylan default)))
	      (else (keyword->symbol keyword)))))
    `(METHOD (,@requireds ,@(if next `(!next ,next) '())
			  ,@(if rest `(!rest ,rest) '())
			  ,@(cond ((eq? keywords #T) `(!key))
				  ((null? keywords) `())
				  (else `(!key
					  ,@(map expand-keyword keywords)))))
	     ,(declare->dylan body))))

(define (generic-function->dylan expr)
  (let ((requireds (list-ref expr 1))
	(rest (list-ref expr 2))
	(keywords (list-ref expr 3))
	(methods (list-ref expr 4)))
    (let ((fn `(MAKE <GENERIC-FUNCTION>
		     REQUIRED: ,(length requireds)
		     REST?: ',rest
		     KEY?: (LIST ,@keywords))))
      (if (null? methods)
	  fn
	  `(BIND ((/BOB/GENERIC-FUNCTION ,fn))
	     ,@(map (lambda (e)
		      `(ADD-METHOD /BOB/GENERIC-FUNCTION ,(expr->dylan e)))
		    methods)
	     /BOB/GENERIC-FUNCTION)))))       

(define (args->dylan arg-list)
  (define (arg->dylan arg)
    (cond ((not (pair? arg)) (list (expr->dylan arg)))
	  ((eq? (car arg) 'keyword-arg)
	   (list (list-ref arg 1) (expr->dylan (list-ref arg 2))))
	  ((eq? (car arg) 'test-keyword-arg)
	   (list (expr->dylan (list-ref arg 1))
		 (expr->dylan (list-ref arg 2))))
	  (else (list (expr->dylan arg)))))
  (apply append (map arg->dylan arg-list)))

(define (variable->dylan var)
  (let ((name (list-ref var 1))
	(type (list-ref var 2)))
    (if (eq? type '<object>)
	name
	(list name (expr->dylan type)))))

(define (process-new-decl-set decls body)
  (define (kind decl) (list-ref decl 0))
  (define (names decl) (map variable->dylan (list-ref decl 1)))
  (define (rest decl) (list-ref decl 2))
  (define (expr decl) (list-ref decl 3))
  (define (is-exit-binding? declaration)
    (eq? (kind declaration) 'BIND-EXIT))
  (define (is-method-binding? declaration)
    (and (eq? (kind declaration) 'BIND)
	 (let ((names (names declaration))
	       (rest (rest declaration))
	       (expr (expr declaration)))
	   (and
	    (not (null? names))
	    (null? (cdr names))
	    (not rest)
	    (pair? expr)
	    (eq? (car expr) 'METHOD)))))
  (define (is-regular-binding? declaration)
    (not (or (is-exit-binding? declaration)
	     (is-method-binding? declaration))))
  (define (make-process more? entry)
    (lambda (decls continue)
      (let loop ((done '())
		 (decls decls))
	(if (or (null? decls)
		(not (more? (car decls))))
	    (continue (reverse done) decls)
	    (loop (cons (entry (car decls)) done)
		  (cdr decls))))))
  (if (null? decls)
      body
      (let ((this (car decls))
	    (process-methods
	     (make-process is-method-binding?
			   (lambda (this)
			     `(,@(names this)
			       ,@(cdr (method->dylan (expr this)))))))
	    (process-regular
	     (make-process is-regular-binding?
			   (lambda (this)
			     `(,@(names this)
			       ,@(if (rest this) `(!rest ,(rest this)) '())
			       ,(expr->dylan (expr this)))))))
	(cond ((is-exit-binding? this)
	       `((BIND-EXIT (,(list-ref this 1))
			   ,@(process-new-decl-set (cdr decls) body))))
	      ((is-method-binding? this)
	       (process-methods decls
		 (lambda (processed left)
		   `((BIND-METHODS ,processed
				  ,@(process-new-decl-set left body))))))
	      (else
	       (process-regular decls
		 (lambda (processed left)
		   `((BIND ,processed
			   ,@(process-new-decl-set left body))))))))))

(define (declare->dylan expr)
  (let ((decls (list-ref expr 1))
	(body (map expr->dylan (list-ref expr 2)))
	(exceptions (list-ref expr 3)))
    (let ((main-body
	   (make-begin (process-new-decl-set decls body))))
      (if (null? exceptions)
	  main-body
	  `(HANDLER-CASE
	    ,main-body
	    ,@(map (lambda (catcher)
		     (let* ((name-and-class (list-ref catcher 1))
			    (param-name (list-ref name-and-class 0))
			    (class (list-ref name-and-class 1))
			    (condition (list-ref catcher 2))
			    (handler (list-ref catcher 3))
			    (descr (list-ref catcher 4)))
		       `((,(expr->dylan class)
			  ,@(if condition
				`(TEST: ,(expr->dylan condition))
				`())
			  ,@(if descr
				`(DESCRIPTION: ,(expr->dylan descr))
				`())
			  ,@(if param-name
				`(CONDITION: ,param-name)
				'()))
			 ,(expr->dylan handler))))
		   exceptions))))))

(define (for-handler iterator body bindings)
  (let* ((test-type (list-ref body 0))
	 (yielding (list-ref body (if (eq? test-type 'FOREVER) 1 2)))
	 (yields (list-ref yielding 1))
	 (action (list-ref yielding 2)))
    (define (standard-test test)
      `,(if (eq? (list-ref test 0) 'UNTIL)
	    (expr->dylan (list-ref test 1))
	    `(NOT ,(expr->dylan (list-ref test 1)))))
    (case test-type
      ((FOREVER)
       (let ((iteration `(,iterator ,bindings (#F) ,(expr->dylan action))))
	 (if (not (and (literal? yields)
		       (eq? #F (literal->value yields))))
	     (make-begin (list iteration (expr->dylan yields)))
	     iteration)))
      ((TEST-FIRST)
       `(,iterator ,bindings
		   (,(standard-test (list-ref body 1)) ,(expr->dylan yields))
		   ,(expr->dylan action)))
      ((TEST-LAST)
       `(BIND ((-EXIT-TEST- #F))
	  (,iterator ,bindings
		     (-EXIT-TEST- ,(expr->dylan yields))
		     ,(expr->dylan action)
		     (SET! -EXIT-TEST- ,(standard-test
					 (list-ref body 1)))))))))

(define (for->dylan expr)
  (for-handler 'FOR (list-ref expr 2)
    (let ((for-names (list-ref expr 1)))
      (let ((names (map (lambda (e) (list-ref e 1)) for-names))
	    (inits (map (lambda (e) (list-ref e 2)) for-names))
	    (bys   (map (lambda (e) (list-ref e 3)) for-names)))
	(map (lambda (name init by)
	       (if by
		   `(,name ,(expr->dylan init) ,(expr->dylan by))
		   `(,name ,(expr->dylan init))))
	     names inits bys)))))

(define (foreach->dylan expr)
  (for-handler 'FOR-EACH (list-ref expr 2)
    (let ((for-names (list-ref expr 1)))
      (let ((names (map (lambda (e) (list-ref e 1)) for-names))
	    (colls (map (lambda (e) (list-ref e 2)) for-names)))
	(map (lambda (name coll) `(,name ,(expr->dylan coll)))
	     names colls)))))

(define (generate-add-slot parent slot)
  (define (variable? expr)
    (or (symbol? expr)
	(and (pair? expr)
	     (eq? (car expr) 'SETTER))))
  (define (expand offset keyword)
    (if (list-ref slot offset)
	(list keyword (expr->dylan (list-ref slot offset)))
	'()))
  (let ((add-slot `(ADD-SLOT ,parent
			     ,@(expand 1 'GETTER:)
			     ,@(expand 2 'SETTER:)
			     ,@(expand 3 'INIT-FUNCTION:)
			     ,@(expand 4 'INIT-VALUE:)
			     ,@(expand 5 'INIT-KEYWORD:)
			     ,@(expand 6 'REQUIRED-INIT-KEYWORD:)
			     ,@(expand 7 'ALLOCATION:)))
	(getter (list-ref slot 1))
	(setter (list-ref slot 2)))
    (let ((get-def (if (variable? getter)
		       `((DEFINE-GENERIC-FUNCTION ,getter (obj)))
		       '()))
	  (set-def (if (variable? setter)
		       `((DEFINE-GENERIC-FUNCTION ,setter (obj value)))
		       `())))
      `(,@get-def ,@set-def ,add-slot))))

(define (add-slot->dylan expr)
  (generate-add-slot (expr->dylan (list-ref expr 1)) (list-ref expr 2)))

(define (add-method->dylan expr)
  (let ((method (list-ref expr 1))
	(gen-fn (list-ref expr 2)))
    `(ADD-METHOD ,(expr->dylan gen-fn) ,(expr->dylan method))))

(define (class->dylan expr)
  (let ((superclasses (map expr->dylan (list-ref expr 1)))
	(slots (list-ref expr 2)))
    (if (null? slots)
	`(MAKE <CLASS> SUPERCLASSES: (LIST ,@superclasses))
	`(BIND ((-THE-CLASS- (MAKE <CLASS> SUPERCLASSES:
				   (LIST ,@superclasses))))
	       ,@(apply append
			(map (lambda (slot)
			       (generate-add-slot '-THE-CLASS- slot))
			     slots))
	       -THE-CLASS-))))

