;;;             Copyright (C) 1989, by William M. Wells III
;;;                         All Rights Reserved
;;;     Permission is granted for unrestricted non-commercial use.

(declare (usual-integrations))

;;;
;;; On the representation of parsing tables:
;;;
;;; Action function is an array, indexed by the state number,
;;; of functions of grammar symbols represented as osets of
;;; 3 element lists containing a g-symbol index, the character
;;; s, r, or a for shift reduce or accept, and an integer encoding the
;;; next state, or production index as appropriate.
;;;
;;; Goto for non-terminals will be represented by a parallel array
;;; of osets of pairs whose cars are g-symbol indices, and whose
;;; cdrs are state indices.

(define action-array)
(define goto-array)


;;; An oset order function for parse table entries.

(define (integer-function-order-function a b)
  (integer-order-function (car a) (car b)))


;;; Build the description of the state machine hich is the lr-parser.
;;; The lr0-item-sets correspond to the states of the parser machine.

(define (build-parse-tables doing-lalr1)
  (set! action-array (make-vector lr0-item-set-count))
  (set! goto-array (make-vector lr0-item-set-count))
  (do ((i 0 (+ i 1)))
      ((= i lr0-item-set-count))
    (vector-set! action-array i
		 (make-oset 'order-fn integer-function-order-function))
    (vector-set! goto-array i
		 (make-oset 'order-fn integer-function-order-function)))
  (oset-for-each
   (lambda (item-set)
     (oset-for-each
      (lambda (goto-elt)	     
	;; Car of goto-elt is g-sym, cdr is item-set.
	(if (g-symbol-non-terminal? (car goto-elt))
	    (oset-insert! (cons (g-symbol-index (car goto-elt))
				(item-set-index (cdr goto-elt)))
			  (vector-ref goto-array
				      (item-set-index item-set)))
	    (parse-table-insert! (g-symbol-index (car goto-elt))
                                 #\s
                                 (item-set-index (cdr goto-elt))
                                 item-set)))
      (item-set-goto-map item-set))
     (oset-for-each
      (lambda (closure-item)
	;; Could these be kernel items?
	(if (dot-at-right-end? closure-item)
	    (if (eq? augmented-start-g-symbol
		     (lhs (item-production closure-item)))
		(parse-table-insert! (g-symbol-index the-end-g-symbol)
                                     #\a 0 item-set) ; accept, bogus 0
		(oset-for-each
		 (lambda (gs)
		   (parse-table-insert! (g-symbol-index gs)
					#\r
					(car (item-production
					      closure-item))
					item-set))
		 ;; Here is the only difference between slr and lalr1
		 ;; (in the table construction phase).
		 (if doing-lalr1
		     (item-look-aheads closure-item)
		     (g-symbol-follow-set
		      (lhs (item-production closure-item))))))))
      (item-set-get-closure! item-set))
     )
   lr0-item-sets))


;;; An auxillary function for adding an entry to a parse table.
;;; A simple feature allows the system to be used with some 
;;; ambiguous grammars:  if the variable allow-conflicts it true,
;;; then when a conflict arises at table construction time, simply
;;; prefer the action which is already in the tables.  
;;; This feature works for the "dangling else" problem.

(define allow-conflicts #f)

(define (parse-table-insert! g-sym-index action-char index item-set)
  (let* ((to-insert (list g-sym-index action-char index))
         (res (oset-insert-2! to-insert
			      (vector-ref action-array
					  (item-set-index item-set)))))
    (if (not (car res))
	(begin
            (newline)
            (display "-------------------") (newline)
            (display "ACTION CONFLICT!!! -- state: ")
            (display (item-set-index item-set))
            (display "  old entry: ") (display (cdr res))
            (display "  new entry: ") (display to-insert) (newline)
            (if (not allow-conflicts) 
                (error "" '())
                (begin
                  (display 
                    "WARNING: continuing to build tables despite conflicts...")
                  (newline)
                  (display "Will prefer old entry: ") (display (cdr res))))))))

(define (get-print-name index)
  (g-symbol-print-name (vector-ref symbol-array index)))


(define (cruise-parse-tables)
  (display "start-state is ") (display lr0-start-state-index)
  (do ((i 0 (+ 1 i)))
      ((= i lr0-item-set-count))
    (newline)
    (display i) (newline)
    (display "actions: ")
    (oset-for-each
     (lambda (action-elt)
       (display (get-print-name (car action-elt)))
       (display " : ") (display (cadr action-elt)) (display (caddr action-elt))
       (display "   ")
       )
     (vector-ref action-array i))
    (newline) (display "gotos: ")
    (oset-for-each
     (lambda (goto-elt)
       (display (get-print-name (car goto-elt))) (display " : ")
       (display (cdr goto-elt)) (display "   ")
       )
     (vector-ref goto-array i))
    (newline)
    (display "--------------------------------------------------")
    ))


;;; (load "tables")

;;; PC scheme requires a control-Z at the end of each source file: 
