;;;             Copyright (C) 1989, by William M. Wells III
;;;                         All Rights Reserved
;;;     Permission is granted for unrestricted non-commercial use.

(declare (usual-integrations))

;;; This defines the representation for sets of items, and
;;; computes the canonical lr(0) collection of sets of items.
;;; It currently leaves the closures lying around on the sets
;;; of items, they could be flushed just after they are used.
;;; It gets hold of the grammar via the symbol 'augmented start
;;; and the application of g-symbol-own-productions to symbols.
;;; The grammar should have been previously internalized
;;; using load-grammar.

(define lr0-item-set-count)
(define lr0-item-sets)
(define lr0-start-state-index)

;;; A type for sets of items.
;;; The kernel will be a o-set of items, the closure might be
;;; an o-set, or might be null if we are trying to save space.
;;; goto-map will be a oset of pairs whose cars are grammar symbols
;;; and whose cdrs are item-sets.

(define-struct item-set
  index
  kernel
  (closure '())
  goto-map)

(define (goto-map-order-function a b)
  (g-symbol-order-function (car a) (car b)))

(define (new-item-set kernel)
  (make-item-set 'kernel kernel
		 'goto-map (make-oset
			    'order-fn goto-map-order-function)))


;;; Item sets can be identified by looking at their kernels, so:

(define (item-set-order-function a b)
  (oset-order-function (item-set-kernel a) (item-set-kernel b)))


;;; Result is an oset of item-sets which comprise the canonical
;;; lr(0) sets of items.

(define (make-lr0-collection )
  (let ((lr0-set (make-oset 'order-fn item-set-order-function))
	(start-prod (car (g-symbol-own-productions
			  augmented-start-g-symbol)))
	(initial-kernel (make-oset 'order-fn item-order-function))
	(initial-state 'bogus))
    (oset-insert! (new-item start-prod) initial-kernel)
    (set! initial-state (new-item-set initial-kernel))
    (lr0-insert-item-set! initial-state lr0-set)
    (set! lr0-item-set-count 0)
    (oset-for-each (lambda (is)
		     (set-slot! (item-set-index is)
			   (post-inc lr0-item-set-count)))
		   lr0-set)
    (set! lr0-start-state-index (item-set-index initial-state))
    (display lr0-item-set-count) (display " item sets") (newline)
    (set! lr0-item-sets lr0-set)
    '()))

;;; item-set should be of that type.
;;; Collection should be an o-set of item-sets.
;;; Returns a pointer to the item set in the collection.

(define (lr0-insert-item-set! item-set collection)
  (let ((insertion-result (oset-insert-2! item-set collection)))
    (if (car insertion-result)		; item wasn't already there
	(begin
          (display ".")
          (for-each
	   (lambda (subset)		
	     ;; subset is an oset of items with same after dot
	     (let ((goto-set (goto subset)))
	       (if (not (oset-empty? goto-set))
		   (oset-insert!
		    (cons (symbol-after-dot
			   (car (oset-item-list subset)))
			  (lr0-insert-item-set! (new-item-set goto-set)
						collection))
		    (item-set-goto-map item-set)))))
	   (oset-select-subsets
	    (item-set-get-closure! item-set)
	    symbol-after-dot))))
    (cdr insertion-result)))


;;; Returns the oset of items which is the closure of the item
;;; set, calculating it if need be from the kernel.
;;; Caches the closure in the closure slot.

(define (item-set-get-closure! item-set)
  (if (null? (item-set-closure item-set))
      (set-slot! (item-set-closure item-set)
		 (closure (item-set-kernel item-set))))
  (item-set-closure item-set))


;;; This isn't used in the current implementation: Sep 13, 1989.

(define (item-set-flush-closure item-set)
  (set-slot! (item-set-closure item-set) '()))

;;; Subset is an oset of items which all have the same after dot symbol.
;;; Result is an oset of items.
;;; Gives back an empty set if the dots are all the way to the right
;;; in the input set.

(define (goto subset)
  (let ((result (make-oset 'order-fn item-order-function)))
    (oset-for-each
     (lambda (item)
       (let ((next (advance-dot item)))
	 (if next (oset-insert! next result))))
     subset)
    result))


(define (print-collection closures-too?)
  (display "start state index: ") (display lr0-start-state-index)
  (newline)
  (oset-for-each
   (lambda (item-set)
     (display "-----------------------------------------") (newline)
     (display (item-set-index item-set)) (newline)
     (item-set-print-kernel item-set closures-too?)
     (display "gotos: ")
     (oset-for-each
      (lambda (gmelt)
	(display (g-symbol-print-name (car gmelt)))
	(display " -> ")
	(display (item-set-index (cdr gmelt))) (display "  ")
	)
      (item-set-goto-map item-set))
     (newline)
     )
   lr0-item-sets))

		 
(define (item-set-print-kernel item-set closure-too?)
  (oset-for-each
   (lambda (item)
     (item-print item) (newline))
   (if closure-too?
       (item-set-get-closure! item-set)
       (item-set-kernel item-set))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; test:

(comment-out
 (load "lr0-sets")
 (load-grammar "ex1.grm")
 (make-lr0-collection)
 (print-collection #f)
 (print-collection #t)
 )


;;; PC scheme requires a control-Z at the end of each source file: 
