;;;             Copyright (C) 1989, by William M. Wells III
;;;                         All Rights Reserved
;;;     Permission is granted for unrestricted non-commercial use.

(declare (usual-integrations))

;;;
;;; Discover and propagate lalr(1) look-aheads among members of lr(0)
;;; collection.

;;; This algorithm for propagating lalr(1) lookaheads is a straightforward
;;; recursive version of the algorithm sketched in section 6.9 of the (older)
;;; dragon book @u(Displayiples of Compiler Design) by A.V. Aho and J.D Ullman.
;;; The major drawback of this algorithm is that it may be somewhat wasteful
;;; of space.  With modern address spaces who cares?
;;; Basically, it crawls around on the lr(0) item sets and as it goes,
;;; it discovers both lookheads which are "spontaneously" generated for
;;; an item set, and item sets to whom lookaheads propagate.  The doubly
;;; recursive way of implementing this is similar to the method used
;;; in calculating first sets in first.s

;;; (New) the names are getting a bit confusing here.  This function transforms
;;; the data structure lr0-item-sets from being the lr(0) collection to
;;; the lalr(1) collection.

(define (lalr1-do-lookaheads)
  ;; Introduce a "dummy" terminal symbol which is used as a hack in
  ;; lookahead calculations.
  (let ((dummy-g-symbol (new-g-symbol "dummy" -1)))
    ;; The dummy symbol is terminal and must be in its own first set.
    (oset-insert! dummy-g-symbol (g-symbol-first-set dummy-g-symbol))
    ;; Map over all the kernel items.
    (oset-for-each
     (lambda (item-set)
       (oset-for-each
	(lambda (kernel-item)
	  ;; Special case: the end symbol is a lookahead for the start
	  ;; production.
	  (if (equal? lr0-start-state-index (item-set-index item-set))
	      ;; There had better only be one item in this set!
	      (lalr1-add-lookahead the-end-g-symbol kernel-item))

	  ;; Here we use the hack in dragon 6.9 (fig 6.20) of using lr(1)
	  ;; closure with a dummy grammar symbol to discover propagated
	  ;; and spontaneous lookaheads for a lr(0) kernel item.  The
	  ;; funny-closure-items are in J' of the figure.

	  (oset-for-each
	   (lambda (funny-closure-item)
	     (if 
	      (not (oset-empty? (item-look-aheads funny-closure-item)))
	      (begin
		(let ((goto-item-proto (advance-dot funny-closure-item)))
		  (if goto-item-proto
		      (begin
			;; Here we go to some expense to locate the goto set
			;; for an item.
			;; These should be pre-computed and cached instead.
			(let ((goto-item
			       (oset-find
				goto-item-proto
				(item-set-kernel
				 (find-goto-set
				  item-set
				  (symbol-after-dot funny-closure-item)))
				"internal error - failed to find goto item")))
			  (oset-for-each
			   (lambda (lookahead)
			     (if (eq? lookahead dummy-g-symbol)
				 ;; Discovered lookahead propagation.
				 (lalr1-add-depender goto-item kernel-item)
				 ;; Discovered lookahead.
				 (lalr1-add-lookahead lookahead goto-item)))
			   (item-look-aheads funny-closure-item)))))))))
	   ;; The set of "funny" closure items. J'.
	   (single-item-closure-1 (copy-lr0-item kernel-item)
				  dummy-g-symbol)))
	(item-set-kernel item-set))
       (display "."))
     lr0-item-sets))

  ;; NEW STUFF HERE: 1-27-88
  (newline)
  (oset-for-each
   (lambda (item-set)
     (closure-1! (item-set-closure item-set))
     (display "."))
   lr0-item-sets
))



;;; This is used when we discover that lookaheads propagate from one
;;; lr(0) item set to another during the calculation of lalr(1) sets
;;; of items.  Add a link to the dependency digraph and propagate the
;;; lookaheads we already know about.

(define (lalr1-add-depender propagate-to propagate-from)
  (if
   (oset-insert! propagate-to (item-look-ahead-dependers propagate-from))
   (oset-for-each
    (lambda (gs)
      (lalr1-add-lookahead gs propagate-to))
    (item-look-aheads propagate-from))))


;;; This is used when we discover a lookhead for an lr(0) item set during
;;; the calculation of lalr(1) sets.  If the lookahead wasn't already there,
;;; add it, and also add it to the "dependers": those item sets to whom
;;; lookaheads propagate from the item in question.

(define (lalr1-add-lookahead symbol item)
  (if
   (oset-insert! symbol (item-look-aheads item))
   ;; Wasn't already there.
   (oset-for-each
    (lambda (depender)
      (lalr1-add-lookahead symbol depender))
    (item-look-ahead-dependers item))))


;;; This should be primitive, and not insert if not there.
;;; Third arg is optional error msg
;;; result is eq to member of the set.

(define (oset-find element set . rest)
  (let ((insertion-result (oset-insert-2! element set)))
    (assert (not (car insertion-result))
            (if (null? rest)
                "oset-find failed to find element"
                (car rest)))
    (cdr insertion-result)))


(define (find-goto-set item-set symbol)
  (cdr (oset-find
	(list symbol)
	(item-set-goto-map item-set)
	"find-goto-set failed to find the goto set")))


(define (copy-lr0-item i)
  (make-item 'production (item-production i)
             'after-dot (item-after-dot i)))



;;;    Do all needed to generate parse tables starting with a lisp syntax
;;;    grammar. (doesn't write out a table file)

(define (lalr1-tables-from-grammar file-name)
  (load-grammar file-name)
  (calculate-empty-string-derivers)
  (calculate-first-sets)
  (calculate-follow-sets)
  (make-lr0-collection)
  (lalr1-do-lookaheads)
  (build-parse-tables #t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; test
(comment-out
;;;    (lalr1-tables-from-grammar "ex6_2.grm")
 (load "lalr1")
 (lalr1-tables-from-grammar "ex4.grm")
 (display "symbols: ") (newline)
 (cruise-symbols-2)
 (display "productions: ") (newline)
 (print-productions)
 (display "lr0 item sets: ") (newline)
 (print-collection #f)
 (display "lalr(1) tables: ") (newline)
 (cruise-parse-tables)
 )

;;; PC scheme requires a control-Z at the end of each source file: 
