;;;             Copyright (C) 1989, by William M. Wells III
;;;                         All Rights Reserved
;;;     Permission is granted for unrestricted non-commercial use.

(declare (usual-integrations))

;;;
;;; Calculate the first sets of the grammar symbols.
;;; Basic design from John Bear :
;;;    University of Texas at Austin Tech Report GRG 220
;;;    "A Breadth-First Syntactic Component"
;;; I added empty string handling: Sandy Wells.


(define (calculate-first-sets )
  ;; The start set of a terminal symbol is the symbol itself.
  (for-each (lambda (gs)
	      (if (not (g-symbol-non-terminal? gs))
                  (oset-insert! gs (g-symbol-first-set gs))))
	    symbols)
  (for-each (lambda (prod)
	      (calculate-first-sets-aux (lhs prod) (rhs prod)))
	    productions))

(define (calculate-first-sets-aux prod-lhs prod-rhs-cdr)
  ;; See if we've empty stringed ourselves out of rhs.
  (if (null? prod-rhs-cdr)
      (first-set-insert! empty-string-g-symbol prod-lhs)
      (let ((rhs-cdr-first (car prod-rhs-cdr)))
        (cond
	 ;; check for terminal symbol or empty string
	 ((or (not (g-symbol-non-terminal? rhs-cdr-first))
	      (eq? rhs-cdr-first empty-string-g-symbol))
	  (first-set-insert! rhs-cdr-first prod-lhs))
	 ;; must be non terminal
	 (else (first-set-add-depender! prod-lhs rhs-cdr-first)
	       (if (g-symbol-derives-empty-string rhs-cdr-first)
		   (calculate-first-sets-aux prod-lhs (cdr prod-rhs-cdr))))))))


;;; Add a symbol to the first set of another symbol.
;;; If it isn't the empty string, and wasn't there already,
;;; add it to the first sets of the guys who's first sets contain this guys.
;;; (the dependers)

(define (first-set-insert! to-insert insertee)
  (if (and (oset-insert! to-insert (g-symbol-first-set insertee))
           (not (eq? empty-string-g-symbol to-insert)))
      (oset-for-each
       (lambda (depender)
	 (first-set-insert! to-insert depender))
       (g-symbol-first-set-dependers insertee))))

(define (first-set-add-depender! new-depender gs)
  (if (oset-insert! new-depender (g-symbol-first-set-dependers gs))
      (oset-for-each
       (lambda (sym)
	 (first-set-insert! sym new-depender))
       (g-symbol-first-set gs))))

(define (print-first-set x)
  (oset-for-each
   (lambda (ee)
     (display (g-symbol-print-name ee)) (display "  "))
   x))

(define (cruise-first-sets )
  (for-each
   (lambda (sym)
     (newline)
     (display (g-symbol-print-name sym))
     (display " : ")
     (print-first-set
      (g-symbol-first-set sym))
     (newline)
     (display "--------------------"))
   symbols))


;;; first-seq (sequence of symbols) returns {s | seq =*=> s...}

(define (first-seq seq)
  (if (null? seq) 
      (make-oset 'order-fn g-symbol-order-function)
      (let ((firsts (g-symbol-first-set (car seq))))
	(if (g-symbol-derives-empty-string (car seq))
	    (oset-union
	     (oset-delete empty-string-g-symbol firsts)
	     (first-seq (cdr seq)))
	    firsts))))
		      

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; test
(comment-out
 (load "first")
 (load-grammar "ex2.grm")
 (calculate-empty-string-derivers)
 (calculate-first-sets)
 (cruise-first-sets)
)



;;; PC scheme requires a control-Z at the end of each source file: 
