;;;             Copyright (C) 1989, by William M. Wells III
;;;                         All Rights Reserved
;;;     Permission is granted for unrestricted non-commercial use.

(declare (usual-integrations))

;;;
;;; Dump parsing tables and associated stuff into a file.
;;;
;;; The follwing stuff is dumped in parenized lists which a lisp read
;;; should be able to read:
;;;
;;; An ordered (by grammar symbol index) lexicon .
;;; A list of the indices of terminal grammar symbols.
;;; A list of production info, ordered by production index, of lists
;;;    containing the index of the lhs grammar symbol and the length
;;;    of the rhs of the production.
;;; A sparse list of lists representation of the action function
;;;    (eyball one and you'll get the idea...).
;;; A similar representation of the goto function.
;;; The index of the start state.
;;; The index of the end symbol.
;;; A list of the client lambda forms.

(define (dump-tables filename)
  (newline)
  (display "Dumping parse tables to ") (display filename)
  (call-with-output-file
      filename
    (lambda (port)
      ;; Dump out an ordered lexicon.
      (write (map
	      car
	      (reverse g-symbol-alist))
             port)
      (newline port) (newline port)

      ;; Dump a list of the indices of terminal grammar symbols
      ;; deal with some special cases... .
      (display (map
		(lambda (gs) (g-symbol-index gs))
		(delete!
                 '()
                 (delete!
		  empty-string-g-symbol
		  (delete!
		   augmented-start-g-symbol
		   (delete!
		    the-end-g-symbol
		    (map (lambda (gs)
			   (if (g-symbol-non-terminal? gs)
			       '()
			       gs))
			 (reverse symbols)))))))
	       port)
      (newline port) (newline port)

      ;; For the lr parser, dump a list of info on the productions.
      ;; The order of the list follows the productions indices in
      ;; the parse tables.  Each element is a list of the index of
      ;; the lhs grammar symbol and the length of the rhs of the production.
      (write
       (map
	(lambda (prod) (list (g-symbol-index (lhs prod))
			     (length (rhs prod))))
	(reverse productions))
       port)
      (newline port)(newline port)

      ;; Dump out a representation of the action function.
      (display "(" port)
      (map
       (lambda (ae)
	 (newline port)
	 (display (oset-item-list ae) port))
       (vector->list action-array))
      (display ")" port)
      (newline port) (newline port)

      ;; Dump out a representation of the goto function for non-terminals
      (display "(" port)
      (map
       (lambda (ge)
	 (newline port)
	 (display (map pair-to-list (oset-item-list ge)) port))
       (vector->list goto-array))
      (display ")" port)
      (newline port) (newline port)

      ;; Dump the index of the start state.
      (display lr0-start-state-index port)
      (newline port) (newline port)

      ;; Dump the index of the end symbol.
      (display (g-symbol-index the-end-g-symbol) port)
      (newline port) (newline port)

      ;; Dump out a list of the client lambdas
      (display "(" port) 
      (map (lambda (the-lambda) (newline port) (write the-lambda port))
           (reverse lambdas))
      (newline port)
      (display ")" port)

      (newline port)
      )))

(define (pair-to-list p)
  (list (car p) (cdr p)))



;;; Set up some convenient ways to process grammars.

(define (compile-slr-grammar grammar-file table-file)
  (slr-tables-from-grammar grammar-file)
  (dump-tables table-file))

(define (compile-lalr1-grammar grammar-file table-file)
  (lalr1-tables-from-grammar grammar-file)
  (dump-tables table-file))
;;;;;;;;;;;;;
;;; test:
(comment-out
;;; (load "dump")(load "slr")(load "lalr1")
 (compile-slr-grammar "ex1.grm" "ex1.tab")
 (compile-slr-grammar "ex2.grm" "ex2.tab")

;; fails : not slr
 (compile-slr-grammar "ex3.grm" "ex3.tab") 
 (compile-slr-grammar "ex4.grm" "ex4.tab")

;; fails : not slr
 (compile-slr-grammar "ex6-2.grm" "ex6-2.tab") 



 (compile-lalr1-grammar "ex1.grm" "ex1.tab")
 (compile-lalr1-grammar "ex2.grm" "ex2.tab")
 (compile-lalr1-grammar "ex3.grm" "ex3.tab")
 (compile-lalr1-grammar "ex4.grm" "ex4.tab")
 (compile-lalr1-grammar "ex6-2.grm" "ex6-2.tab")
 )


;;; PC scheme requires a control-Z at the end of each source file: 
