;\c	    Copyright (C) 1990 Pertti Kellomaki
;\c	 
;\c	 This file is part of Taurus, a parser generator producing Scheme
;\c	 
;\c	 Taurus is free software; you can redistribute it and/or modify
;\c	 it under the terms of the GNU General Public License as published by
;\c	 the Free Software Foundation; either version 1, or (at your option)
;\c	 any later version.
;\c	 
;\c	 Taurus is distributed in the hope that it will be useful,
;\c	 but WITHOUT ANY WARRANTY; without even the implied warranty of
;\c	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;\c	 GNU General Public License for more details.
;\c	 
;\c	 You should have received a copy of the GNU General Public License
;\c	 along with Taurus; see the file COPYING.  If not, write to
;\c	 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;
;\node Generating FIRST Sets, , , 
;\comment  node-name,  next,  previous,  up
;\chapter{Generating FIRST Sets}
;
;In places where parsing can take more than one route, the parser must
;examine the next token in order to decide, which route to take. For
;these decisions, the \dfn{FIRST} set (the set of terminal symbols that can start the
;given expression) for each expression must be computed. 
;
;The procedures doing this form the module
;
(module tstart)

;\node Computing FIRST Sets
;\comment  node-name,  next,  previous,  up
;\section{Computing FIRST Sets}
;
;The procedure \code{first-set} finds the FIRST set of the given
;expression. The reason that there are \code{first-set} and
;\code{first-set-expr}, is that the extra argument in
;\code{first-set-expr} is used for detecting left recursion.
;\c WORK ON THIS
;\c The FIRST set is only computed once, and it is stored in the
;\c expression, where it is fetched if needed.
;
;\findex{first-set}
(define (first-set expr grammar)
  (first-set-expr expr grammar '()))

  
;\findex{first-set-expr}
(define (first-set-expr expr grammar visited-nonterminals)
  (cond ((terminal? expr) `(,expr))
	((action? expr) `(,(make-empty)))
	((empty? expr) `(,(make-empty)))
	((nonterminal? expr)
	 (nonterminal-first-set expr
				grammar
				visited-nonterminals))
	((sequence? expr)
	 (sequence-first-set expr
			     grammar
			     visited-nonterminals))
	((alternative? expr)
	 (alternative-first-set expr
				grammar
				visited-nonterminals))
	((zero-iteration? expr)
	 (zero-iteration-first-set expr
				   grammar
				   visited-nonterminals))
	((nonzero-iteration? expr)
	 (nonzero-iteration-first-set expr
				      grammar
				      visited-nonterminals))
	(else (taurus-error "first-set-expr: bad expression"
			    expr))))

;\node FIRST Set of a Sequence
;\comment  node-name,  next,  previous,  up
;\subsection{FIRST Set of a Sequence}
;
;The FIRST set of a sequence is the same as the first set of its first
;element.  If the FIRST set of the first element includes the empty
;symbol, then the FIRST set of the rest of the sequence belongs to the
;set also, but \code{empty} does not. If the sequence ends before an
;element whose FIRST set does not include \code{empty}, \code{empty} is
;explicitely added to the set. This way the FIRST set of an empty
;sequence is \code{(empty)}.
;
;\findex{sequence-first-set}
(define (sequence-first-set expr
			    grammar
			    visited-nonterminals)
  
  (let loop ((elements (sequence-elements expr))
	     (set '()))
    (if (null? elements)
	(cons (make-empty) set)
	(let ((this-set (first-set-expr (car elements)
					grammar
					visited-nonterminals)))
	  (if (member (make-empty) this-set)
	      (loop (cdr elements)
		    (set-union (remove (make-empty) this-set)
			       set))
	      (set-union this-set set))))))

;\node FIRST Set of an Alternative
;\comment  node-name,  next,  previous,  up
;\subsection{FIRST Set of an Alternative}
;
;The FIRST set of an alternative expression is the union of the FIRST
;sets of its choices.
;
;\findex{alternative-first-set}
(define (alternative-first-set expr
			       grammar
			       visited-nonterminals)
  
  (let loop ((choices (alternative-choices expr))
	     (set '()))
    (if (null? choices)
	set
	(loop (cdr choices)
	      (set-union (first-set-expr (car choices)
					 grammar
					 visited-nonterminals)
		      set)))))

;\node FIRST Set of an Iteration
;\comment  node-name,  next,  previous,  up
;\subsection{FIRST Set of an Iteration}
;
;The FIRST set of an iteration is the same as the first set of the
;expression being iterated. If the expression can be iterated zero times,
;also \code{empty} belongs to the FIRST set.
;
;\findex{nonzero-iteration-first-set}
(define (nonzero-iteration-first-set expr
				     grammar
				     visited-nonterminals)
  
  (first-set-expr (iteration-iterand expr)
		  grammar
		  visited-nonterminals))

;\findex{zero-iteration-first-set}
(define (zero-iteration-first-set expr
				  grammar
				  visited-nonterminals)
  (let ((set (first-set-expr (iteration-iterand expr)
			     grammar
			     visited-nonterminals)))
    (if (member (make-empty) set)
	set
	(cons (make-empty) set))))

;\node FIRST Set of a Nonterminal
;\comment  node-name,  next,  previous,  up
;\subsection{FIRST Set of a Nonterminal}
;
;\code{nonterminal-first-set} returns the FIRST set of a nonterminal. It
;is first looked up in \code{*nonterminal-first-sets*}. If it is not
;found, it has to be computed.  If the FIRST set for this nonterminal is
;already being computed, the grammar is left recursive, and construction
;of the parser is not possible. 
;
;The computed FIRST set is stored in \code{*nonterminal-first-sets*}.
;\code{make-parser} calls \code{init-first-sets} before processing the grammar.
;
;\findex{*nonterminal-first-sets*}
(define *nonterminal-first-sets* '())
;\findex{init-first-sets}
(define (init-first-sets)
  (set! *nonterminal-first-sets* '()))


;\findex{nonterminal-first-set}
(define (nonterminal-first-set nonterminal
			       grammar
			       visited-nonterminals)
  (cond ((member nonterminal visited-nonterminals)
	 (taurus-error
	  "nonterminal-first-set: left recursion in"
	  (nonterminal-name nonterminal)))
	((let ((set (assoc (nonterminal-name nonterminal)
			   *nonterminal-first-sets*)))
	   (if set
	       (cdr set)
	       #f)))
	(else
	 (let ((new-first-set
		(first-set-expr (rule-expr (grammar-rule-for
					    nonterminal
					    grammar))
			   grammar
			   (cons nonterminal
				 visited-nonterminals))))
	   (set! *nonterminal-first-sets*
		 `((,(nonterminal-name nonterminal)
		    ,@new-first-set)
		   ,@*nonterminal-first-sets*))
	   new-first-set))))
