
;;;; Fixpoint module semantics following Bracha and Lindstrom.

;; module-instantiate : Module -> Record Val

;; Module     = Self -> PreMod
;; PreMod     = Record Entry
;; Self       = Delay PreMod
;; Entry      = Self * Exp * Val
;; Exp        = Env -> Val

;; A pre-module is a record of expressions that remember which
;; environment they're to be closed over.  Memoizing self is critical
;; for the semantics.

;;; Names

;; Names are compared using eq? and can be anything, but MIT Scheme
;; environments insist on symbols.

(define name-symbol identity-procedure)

;;; Entries

(define unknown-value '(*unknown*))

(define-structure entry
  self exp value)

(define (make-new-entry self exp)
  (make-entry self exp unknown-value))

(define (entry-premod entry)
  (force (entry-self entry)))

(define (entry-value-unknown? entry)
  (eq? unknown-value (entry-value entry)))

(define (entry-determine-value! entry a-list)
  (if (entry-value-unknown? entry)
      (set-entry-value!
       entry
       ((entry-exp entry)
	(cdr (assq (entry-premod entry) a-list))))))

;;; Premods

(define binding-entry binding-value)

(define (premod-expand premod)
  (map (lambda (binding)
	 (entry-premod (binding-entry binding)))
       (record->list premod)))

(define (premod->record premod)
  (record-map
   (lambda (binding)
     (make-binding (binding-name binding)
		   (entry-value (binding-entry binding))))
   premod))

(define (premod-make-env premod)
  (eval `(make-environment 
	   ,@(map (lambda (name) `(define ,(name-symbol name)))
		  (record-names premod)))
	system-global-environment))

(define (premod-loop premods envs f)
  (for-each 
   (lambda (premod env)
     (record-for-each 
      (lambda (binding) (f binding env)) 
      premod))
   premods envs))

;;; Instantiate

(define (module-instantiate module)

  (let* ((premod (fix module))
	 (premods (transitive-closure (list premod) premod-expand))
	 (envs (map premod-make-env premods))
	 (a-list (map cons premods envs)))

    (premod-loop premods envs
     (lambda (binding env)
       (entry-determine-value! (binding-entry binding) a-list)))

    (premod-loop premods envs
     (lambda (binding env)
       (environment-assign!
	env (binding-name binding)
	(entry-value (binding-entry binding)))))

    (premod->record premod)))

;;; Needed stuff

(define (transitive-closure nodes expand)
  (let loop ((tc '()) (nodes nodes))
    (if (null? nodes)
	tc
	(let ((node (car nodes)))
	  (if (memq node tc)
	      (loop tc (cdr nodes))
	      (loop (cons node tc) (append (expand node) nodes)))))))

(define (fix f)				; f : Delay S -> S
  (letrec ((g (f (delay g))))
    g))

;;; Module operators

(define (make-mop0 record-operator)
  record-operator)

(define module-defines?    (make-mop0 record-defines?))
(define module-names       (make-mop0 record-names))
(define empty-module       (make-mop0 empty-record))
(define module-empty?      (make-mop0 record-empty?))

(define (make-mop1 record-operator)
  (lambda (a module)
    (lambda (s) (record-operator a (module s)))))

(define module-delete     (make-mop1 record-delete))
(define module-extract    (make-mop1 record-extract))
(define module-copy-as    (make-mop1 record-copy-as))
(define module-extract-as (make-mop1 record-extract-as))

(define (make-mop2 record-operator)
  (lambda (m1 m2)
    (lambda (s) (record-operator (m1 s) (m2 s)))))

(define module-union                (make-mop2 record-union))
(define module-layer                (make-mop2 record-layer))
(define module-difference           (make-mop2 record-difference))
(define module-symmetric-difference (make-mop2 record-symmetric-difference))

;;; freeze, freeze-except

(define (module-freeze names module)
  (lambda (s)
    (letrec ((r (module 
		 (delay 
		   (let ((self (force s)))
		     (record-layer (record-extract names r) self))))))
      r)))

(define (module-freeze-except names module)
  (lambda (s)
    (letrec ((r (module 
		 (delay 
		   (let ((self (force s)))
		     (record-layer 
		      (record-extract names self)
		      (record-layer r self)))))))
      r)))

;;; hide, hide-except

(define (module-hide names module)
  (lambda (s)
    (record-delete names ((module-freeze names module) s))))

(define (module-hide-except names module)
  (lambda (s)
    (record-extract names ((module-freeze-except names module) s))))

;;; rename, rename-inputs, rename-outputs

(define module-rename-outputs 
  (make-mop1 record-rename))

(define module-rename-outputs*
  (make-mop1 record-rename*))

(define (module-rename bindings module)
  (module-rename-outputs*
   bindings
   (module-rename-inputs bindings module)))

(define (module-rename-inputs bindings module)
  (lambda (s)
    (module
     (delay 
       (let ((self (force s)))
	 (record-layer
	  (record-extract-as (invert-bindings bindings) self)
	  self))))))

