;;;; -*- Scheme -*-
;;;; $Header: /home/panda/pg/bevan/progs/scheme/reduction/RCS/sk.scm,v 1.1 91/05/26 12:51:08 bevan Exp $
;;;+file-summary
;;; SKI compiler.
;;;
;;; Takes expressions of the form :-
;;;
;;; expr ::= (const int)
;;;        | (var var)
;;;        | (App expr expr)
;;;        | (lam var expr)
;;;
;;; and converts them into SKI combinators of the form 
;;;
;;; SKI  ::= (S)
;;;        | (K)
;;;        | (I)
;;;        | ($ builtin)
;;;
;;; Also includes an evaluator which takes SKI combinators an reduces them
;;;
;;; See any good book on lambda calculus for what the above means.
;;; For example, I prefer :-
;;;   The Implementation of Functional Programming Languages
;;;   Simon L. Peyton Jones
;;;   Prentice Hall
;;;   ISBN 0-13-453325-9
;;; because, (as the title suggests) its about how functional languages can
;;; implemented as well as the theory behind them.
;;;
;;; The evaluators are a loose translation of ML/Miranda ones I wrote
;;; while I was taking a course on functional programming languages.
;;;
;;; Stephen J. Bevan <bevan@cs.man.ac.uk>
;;;-

(define sk-mk-var (lambda (var) (list 'var var)))
(define sk-var? (lambda (expr) (eq? (car expr) 'var)))
(define sk-var (lambda (var) (cadr var)))

(define sk-mk-const (lambda (int) (list 'const int)))
(define sk-const? (lambda (expr) (eq? (car expr) 'const)))
(define sk-const (lambda (const) (cadr const)))

(define sk-mk-app (lambda (expr1 expr2) (list 'app expr1 expr2)))
(define sk-app? (lambda (expr) (eq? (car expr) 'app)))
(define sk-app-lhs (lambda (expr) (cadr expr)))
(define sk-app-rhs (lambda (expr) (caddr expr)))

(define sk-mk-lambda (lambda (var expr) (list 'lam var expr)))
(define sk-lambda? (lambda (expr) (eq? (car expr) 'lam)))
(define sk-lambda-var (lambda (lam) (cadr lam)))
(define sk-lambda-expr (lambda (lam) (caddr lam)))

(define sk-mk-s (lambda () (list 'S)))
(define sk-s? (lambda (expr) (eq? (car expr) 'S)))

(define sk-mk-k (lambda () (list 'K)))
(define sk-k? (lambda (expr) (eq? (car expr) 'K)))

(define sk-mk-i (lambda () (list 'I)))
(define sk-i? (lambda (expr) (eq? (car expr) 'I)))

(define sk-mk-builtin (lambda (name) (list '$ name)))
(define sk-builtin? (lambda (expr) (eq? (car expr) '$)))
(define sk-builtin (lambda (builtin) (cadr builtin)))

;;;+fs
;;; Abstracts the variable `x' out of the expression `f'
;;;-
(define sk-abstract
  (lambda (x f)
    (cond
     ((sk-app? f)
      (sk-mk-app
       (sk-mk-app (sk-mk-s) (sk-abstract x (sk-app-lhs f)))
       (sk-abstract x (sk-app-rhs f))))
     ((sk-var? f)
      (if (eq? x (sk-var f))
	  (sk-mk-i)
	  (sk-mk-app (sk-mk-k) f)))
     (else (sk-mk-app (sk-mk-k) f)))))

;;;+fs
;;; Compile the given lambda expression into SKI combinators.
;;;-
(define sk-compile
  (lambda (expr)
    (cond
     ((sk-app? expr)
      (sk-mk-app (sk-compile (sk-app-lhs expr))
		 (sk-compile (sk-app-rhs expr))))
     ((sk-lambda? expr)
      (sk-abstract (sk-lambda-var expr)
		   (sk-compile (sk-lambda-expr expr))))
     (else expr))))

;;;+fs
;;; Pretty print the SKI expression
;;;-
(define sk-pp
  (lambda (expr)
    (cond
     ((sk-app? expr)
      (let ((lhs (sk-app-lhs expr))
	    (rhs (sk-app-rhs expr)))
	(cond ((and (sk-app? lhs)
		    (sk-s? (sk-app-lhs lhs)))
	       (list 'S
		     (sk-pp (sk-app-rhs lhs))
		     (sk-pp rhs)))
	      ((sk-k? lhs) (list 'K (sk-pp rhs)))
	      (else (list (sk-pp lhs) (sk-pp rhs)))
	      )))
     ((sk-s? expr) 'S)
     ((sk-k? expr) 'K)
     ((sk-i? expr) 'I)
     ((sk-const? expr) (sk-const expr))
     ((sk-var? expr) (sk-var expr))
     ((sk-builtin? expr) (sk-builtin expr)))))

;;;+fs
;;; Evaluate the SKI expression using SKI reduction rules
;;; I have my doubts about this function!
;;; I think I've implemented it wrong as it does not seem lazy enough in
;;; its evaluation.
;;;-
(define sk-eval
  (lambda (expr)
    (let ((lhs (sk-app-lhs expr))
	  (rhs (sk-app-rhs expr)))
      (cond
       ;; handle I
       ((sk-i? lhs) rhs)
       ;; K only has one argument so return the expr unevaluated.
       ((sk-k? lhs) expr)
       ;; S only has one arg so again return it unevaluated.
       ((sk-s? lhs) expr)
       ;; All builtins are binary so not enough args yet.
       ((sk-builtin? lhs) expr)
       ;; Try an app of an app.
       ((sk-app? lhs)
	(let ((lhs2 (sk-app-lhs lhs))
	      (rhs2 (sk-app-rhs lhs)))
	  (cond
	   ;; K has correct args so deal with it.
	   ((sk-k? lhs2) rhs2)
	   ;; S still does not have enought args.
	   ((sk-s? lhs2) expr)
	   ;; Try for an S with correct args.
	   ((sk-app? lhs2)
	    (let ((lhs3 (sk-app-lhs lhs2))
		  (rhs3 (sk-app-rhs lhs2)))
	      (sk-eval (sk-mk-app (sk-eval (sk-mk-app rhs3 rhs))
				  (sk-eval (sk-mk-app rhs2 rhs)))))))))
       (else (sk-eval (sk-mk-app (sk-eval lhs) rhs)))))))

;;;+file-examples
;;; The rest is some examples of how to use the above.

;;; Some primitive numbers.
(define zero (lam 'x (lam 'y (var 'y))))
(define one (lam 'x (lam 'y (app (var 'x) (var 'y)))))
(define two (lam 'x (lam 'y (app (var 'x) (app (var 'x) (var 'y))))))

;;; The successor function for numbers.
(define succ
  (lam 'x
    (lam 'y 
      (lam 'z (app (var 'y)
                   (app (app (var 'x) (var 'y))
                        (var 'z)))))))

;;; Define the sk equivalent of the above lambda expressions.
;(define sk-zero (sk-compile zero))
;(define sk-one (sk-compile one))
;(define sk-two (sk-compile two))
;(define sk-succ (sk-compile succ))

;;; Find the successor of zero
;(sk-eval (app sk-succ sk-zero))

;;;-file-examples
