;;;; -*-Scheme-*-
;;; $Header: /home/panda/pg/bevan/progs/scheme/reduction/RCS/beta.scm,v 1.1 91/05/26 12:51:47 bevan Exp $
;;;+file-summary
;;; Normal Form, Weak Head Normal Form, and Applicative order reducers
;;; for expressions of the form :-
;;;
;;; expr ::= (const int)
;;;        | (var var)
;;;        | (app expr expr)
;;;        | (lam var expr)
;;;
;;; See any good book on lambda calculus for what the above means.
;;; For example, I prefer :-
;;;   The Implementation of Functional Programming Languages
;;;   Simon L. Peyton Jones
;;;   Prentice Hall
;;;   ISBN 0-13-453325-9
;;; because, (as the title suggests) its about how functional languages can
;;; implemented as well as the theory behind them.
;;;
;;; The evaluators are a loose translation of ML/Miranda ones I wrote
;;; while I was taking a course on functional programming languages.
;;;
;;; Stephen J. Bevan <bevan@cs.man.ac.uk>
;;;-
;;;+extensions
;;; error
;;;   a la CommonLisp, or most Schemes for that matter.
;;;   A simple one is provided if your system hasn't got error.
;;; gensym
;;;   again as in CommonLisp.
;;;-

;;;+system scm
(require 'gensym)
;;;-
;;; If your system doesn't have "error", then try the following.  It should
;;; work ok for any errors produced in this file
;(define error
;  (lambda (function-error-occurs-in error-message-to-print . any-arguments)
;    (display "ERROR : ")
;    (display function-error-occurs-in)
;    (display " ")
;    (display error-message-to-print)
;    (for-each
;     (lambda (argument)
;       (display " ")
;       (display argument))
;     any-arguments)
;    (newline)
;    (let loop ()			; infinite loop
;      (loop))))			; to stop processing continuing.

;;; Definitions of the constructors and predicates for representing
;;; lambda expressions.  I've used lists, but for a bit more speed,
;;; you could try vectors.

(define var (lambda (var) (list 'var var)))
(define var? (lambda (expr) (eq? (car expr) 'var)))
(define var-id (lambda (var) (cadr var)))

(define const (lambda (int) (list 'const int)))
(define const? (lambda (expr) (eq? (car expr) 'const)))
(define const-value (lambda (const) (cadr const)))

(define app (lambda (expr1 expr2) (list 'app expr1 expr2)))
(define app? (lambda (expr) (eq? (car expr) 'app)))
(define app-lhs (lambda (expr) (cadr expr)))
(define app-rhs (lambda (expr) (caddr expr)))

(define lam (lambda (var expr) (list 'lam var expr)))
(define lam? (lambda (expr) (eq? (car expr) 'lam)))
(define lam-var (lambda (lam) (cadr lam)))
(define lam-expr (lambda (lam) (caddr lam)))

;;;+fs
;;; Does simple substitution without worrying about free variables.
;;; i.e. F=[M/x]E
;;;-
(define (substitute-simple expr m x)
  (cond
   ((var? expr) (if (eq? x (var-id expr)) m expr))
   ((const? expr) expr)
   ((app? expr) (app (substitute-simple (app-lhs expr) m x)
		     (substitute-simple (app-rhs expr) m x)))
   ((lam? expr) (if (eq? (lam-var expr) x)
		    expr
		    (lam (lam-var expr)
			 (substitute-simple (lam-expr expr) m x))))
   (else (error 'substitute-simple "wrong expression type" expr))))

;;;+fs
;;; Return a list of the free variables in an expression
;;;-
(define (free-vars expr)
  (let loop ((expression expr)
	     (currently-bound-variables '()))
    (cond
     ((var? expr) (if (memq (var-id expr) currently-bound-variables)
		      '()
		      (list (var-id expr))))
     ((const? expr) '())
     ((app? expr) (append (loop (app-lhs expr) currently-bound-variables)
			  (loop (app-rhs expr) currently-bound-variables)))
     ((lam? expr) (loop (lam-expr expr)
			(cons (lam-var expr) currently-bound-variables))))))

;;;+fs
;;; Given a lambda expression and a variable, returns true/false
;;; depending on whether the variable occurs free in the expression
;;; or not.
;;;-
(define (occurs-free? expression-to-check-in variable-to-look-for)
  (cond
   ((var? expression-to-check-in)
    (eq? (var-id expression-to-check-in) variable-to-look-for))
   ((const? expression-to-check-in) #f)
   ((app? expression-to-check-in)
    (or (occurs-free? (app-lhs expression-to-check-in)
		      variable-to-look-for)
	(occurs-free? (app-rhs expression-to-check-in)
		      variable-to-look-for)))
   ((lam? expression-to-check-in)
    (and (eq? variable-to-look-for (lam-var expression-to-check-in))
	 (occurs-free? (lam-expr expression-to-check-in)
		       variable-to-look-for)))))

;;;+fs
;;; Returns a new variable that hasn't been used in the list of
;;; `already-defined-variables'
;;; Note isn't functional, it uses (gensym)
;;;-
(define (new-var already-defined-variables)
  (let ((new-variable (gensym)))
    (if (memq new-variable already-defined-variables)
	new-variable
	(new-var already-defined-variables))))

;;;+fs
;;; Do complete variable substitution
;;; (including alpha conversion where necessary)
;;; F=[M/x]E
;;;-
(define (substitute-full expr m x)
  (cond
   ((var? expr) (if (eq? x (var-id expr)) m expr))
   ((const? expr) expr)
   ((app? expr) (app (substitute-full (app-lhs expr) m x)
		     (substitute-full (app-rhs expr) m x)))
   ((lam? expr)
    (let ((body (lam-expr expr))
	  (id (lam-var expr)))
      (cond ((eq? id x) expr)
	    ((not (occurs-free? body x)) expr)
	    ((not (occurs-free? m id)) (lam id (substitute-full body m x)))
	    (else
	     (let ((z (new-var (append (free-vars body) (free-vars m)))))
	       (lam z (substitute-full (substitute-full body (var z) id) m x)))))))))

;;;+fs
;;; Performs normal order evaluation on the lambda expression `expr'
;;; The lecturer I had for the course doesn't think the evaluator will
;;; work correctly for all inputs.  I tend to agree with him!
;;; Does anybody have a better version?
;;;-
(define (nf-eval expr)
  (cond
   ((lam? expr) (lam (lam-var expr) (nf-eval (lam-expr expr))))
   ((app? expr)
    (let ((y (app-lhs expr))
	  (z (app-rhs expr)))
      (cond ((lam? y)
	     (nf-eval (substitute-full (lam-expr y) z (lam-var y))))
	    ((app? y)
	     (let ((evald-inner-app (nf-eval y)))
	       (if (app? evald-inner-app)
		   (app evald-inner-app (nf-eval z))
		   (nf-eval (app evald-inner-app z)))))
	    (else (app y (nf-eval z))))))
   (else expr)))
	     
;;;+fs
;;; Performs Weak Head Normal Form reduction on `expr'
;;;-
(define (whnf-eval expr)
  (cond
   ((app? expr)
    (let ((y (app-lhs expr))
	  (z (app-rhs expr)))
      (cond
       ((lam? y)
	(whnf-eval (substitute-simple (lam-expr y) z (lam-var y))))
       ((app? y)
	(whnf-eval (app (whnf-eval y) z)))
       (else
	(error 'whnf-eval "illegal reduction" y)))))
   (else expr)))

;;;+fs
;;; Applicative Order reducer a la Scheme, Lisp, ML etc.
;;;-
(define (app-eval expr)
  (cond
   ((app? expr)
    (let ((y (app-lhs expr))
	  (z (app-rhs expr)))
      (cond
       ((app? y) (app-eval (app (app-eval y) (app-eval z))))
       ((lam? y) (app-eval (substitute-simple (lam-expr y)
					      (app-eval z)
					      (lam-var y))))
       (else (error 'app-eval "illegal reduction" y)))))
   (else expr)))

;;; End of the reducers.
;;;+file-examples
;;; The following are some misc. expressions with which to test out
;;; the evalutators.

;;; A misc. expression
(define expr1 (app (lam 'f 
                     (lam 'q
                       (app (var 'f) (var 'q))))
                   (lam 'a (var 'a))))

;;; Truth Values
(define true (lam 'x (lam 'y (var 'x))))
(define false (lam 'x (lam 'y (var 'y))))

;;; Negate a truth value.
(define nott (lam 'c (app (app (var 'c) false) true)))

;;; Some primitive numbers.
(define zero (lam 'x (lam 'y (var 'y))))
(define one (lam 'x (lam 'y (app (var 'x) (var 'y)))))
(define two (lam 'x (lam 'y (app (var 'x) (app (var 'x) (var 'y))))))

;;; The successor function for numbers.
(define succ
  (lam 'x
    (lam 'y 
      (lam 'z (app (var 'y)
                   (app (app (var 'x) (var 'y))
                        (var 'z)))))))

;;; The predecessor function for numbers.
(define pred
  (lam 'k
    (app (app (app (var 'k)
		   (lam 'p
		     (lam 'u (app (app (var 'u)
				       (app succ (app (var 'p) true)))
				  (app (var 'p) true)))))
	      (lam 'u (app (app (var 'u) zero) zero)))
	 false)))

;;; returns `true' if the argument is `zero'
(define isZero (lam 'k (app (app (var 'k) (app true false)) true)))

;;; Adds `n' and `m'
(define add (lam 'n (lam 'm (lam 'f (lam 'x 'u)))))

;;; Both of the following define multiplication.
;;; The reason there are two is that for most functions I found the same
;;; definition in the books I looked in, however for multiplication I found
;;; the following two defintions.
(define times
  (lam 'm
    (lam 'n
      (lam 'c (app (app (var 'm) (var 'n)) (var 'c))))))
(define mult
  (lam 'n
    (lam 'm
      (lam 'f (app (var 'n) (app (var 'm) (var 'f)))))))

;;; Y combinator
(define Y
  (lam 'h (app (lam 'x (app (var 'h) (app (var 'x) (var 'x))))
	       (lam 'x (app (var 'h) (app (var 'x) (var 'x)))))))

;;; if-then-else
(define ite (lam 'c (lam 'p (lam 'q (app (app (var 'c) (var 'p)) (var 'q))))))

;;; List processing functions.
(define ccons
  (lam 'a
    (lam 'b
      (lam 'f (app (app (var 'f) (var 'a)) (var 'b))))))
(define head (lam 'c (app (var 'c) (lam 'a (lam 'b (var 'a))))))
(define tail (lam 'c (app (var 'c) (lam 'a (lam 'b (var 'b))))))

;;; Return the head of a list of two items.
(define head-expr (app head (app (app ccons (const 2)) (const 1))))
;;; Return the tail of a list of two items.
(define tail-expr (app tail (app (app ccons (const 2)) (const 1))))

;;; Useful for testing full substitution.
(define free-var (lam 'x (app (lam 'y (lam 'x (var 'y))) (var 'x))))

;;; An infinite list of 1s
(define ones (app y (lam 'on (app (app ccons (const 1)) (var 'on)))))

;;; My attempt at a factorial function.
;;; This gives dubious results.
(define fac
  (lam 'f
    (lam 'n
      (app (app (app isZero (var 'n)) one)
	   (app (app times (var 'n))
		(app (var 'f)
		     (app pred (var 'n))))))))

;;; Make fact the fixed point of fac
(define fact (app y fac))

;;; Some simple things to try
;;; The lines prefixed by a single `;' are ones you might like to try
;;; on your system.

;;; Produce the head of a list to two items.
; (whnf-eval head-expr)
;;; Produce the tail of a list of two itmes.
; (whnf-eval tail-expr)

;;; find the head of an infinite list of 1s
; (whnf-eval (app head ones))

;;; Attempt to find the head of an infinite list using Scheme like
;;; reduction order.  This one will loop forever, so don't run it
;;; unless you can interrupt you scheme.
; (app-eval (app head ones))

;;;-file-examples
