;;;;
;;;; $Id: indent.scm,v 1.2 1992/10/17 11:28:19 pk Exp pk $
;;;;
;;;; sclint -- a Scheme lint.
;;;;
;;;; Pertti Kellom\"aki, 1992
;;;;
;;;; $Log: indent.scm,v $
;;;; Revision 1.2  1992/10/17  11:28:19  pk
;;;; Allow different indentation for named let and do.
;;;; Better handling of expression lists.
;;;; Handling of 'foo that expands to (quote foo) in the reader.
;;;;
;;;; Revision 1.1  1992/09/22  07:18:39  pk
;;;; Initial revision
;;;;
;;;; 

;;;;
;;;; These procedures check that the indentation agrees with the
;;;; logical structure of the program.
;;;;

;;;
;;; For now, we are only interested in indentation of lists with more
;;; than one element.
;;; 
;;;  The indentation rules for lists are:
;;;    * The first element of the list must be indented more than the
;;;      opening paren of the list
;;;    * The second element must not be indented less than the first one.
;;;    * The third etc. elements must be indented the same amount as
;;;      the second one.
;;;    * If more than one element is put on the same line, only the
;;;      first one is considered.
;;; 
;;;   In effect, these rules tolerate the following indentations:
;;; 
;;;     (elem elem ...)
;;; 
;;;     (elem elem
;;;           elem
;;;           ...)
;;; 
;;;     (elem elem elem ...     This is useful for procedures
;;;           elem elem ...     with lots of parameters
;;;           ...)
;;; 
;;;     (elem                   For cond clauses etc.
;;;      elem
;;;      ...)
;;; 
;;;     (elem elem ...          For list constants with lots of
;;;      elem elem ...          elements
;;;
;;;    For some constructs we allow different layout. These are:
;;;
;;;      (define (name params...)
;;;        expr
;;;        expr ...)
;;;
;;;      (case expr
;;;        ((literal...) ...)
;;;
;;;      (let (...)    ; or let* or letrec
;;;        expr
;;;        expr ...)
;;;
;;;      (let name (...)
;;;        expr
;;;        expr ...)
;;;
;;;      (do (...)
;;;          (...)
;;;        expr ...)
;;;
;;;      Actually, also the following is accepted:
;;;
;;;      (do (...)
;;;         (...)
;;;         expr ...)
;;;         
;;;      (lambda (...)
;;;        expr
;;;        expr ...)
;;;
;;;     Here the indentation level for the rest of the body is taken
;;;     from the third (or fourth) subexpression. 
   
(define (check-indentation form current-indentation)

  ;; recursively check subexpressions
  (if (psd-pair? form)
      (psd-map (lambda (subform) (check-indentation subform
                                                    (psd-expr-start-column subform)))
               form)
      #f)

  (cond

   ;; quit here, if form is not a list
   ((not (psd-pair? form))
    #f)

   ;; named let and do
   ((or (match '(do expr expr expr expr*) form)
	(match '(let symbol expr expr expr*) form))

    ;; check the indentation of first element
    (if (< (psd-expr-start-column (psd-car form))
	   current-indentation)
        (indent-warning (psd-car form))
        #f)

    ;; check the indentation of the second element
    (if (>= (psd-expr-start-column (psd-car form))
            (psd-expr-start-column (psd-cadr form)))
        (indent-warning (psd-cadr form))
        #f)

    ;; check that the third element is to the right of the first
    (if (>= (psd-expr-start-column (psd-car form))
            (psd-expr-start-column (psd-caddr form)))
        (indent-warning (psd-caddr form))
        #f)

    ;; check that the fourth element is to the right of the first
    (if (>= (psd-expr-start-column (psd-car form))
            (psd-expr-start-column (psd-caddr (psd-cdr form))))
        (indent-warning (psd-caddr (psd-cdr form)))
        #f)

    ;; the rest should be lined up below the fourth element
    (check-sequence-indentation (psd-expr-start-column (psd-caddr (psd-cdr form)))
				(psd-expr-start-line (psd-caddr (psd-cdr form)))
				(psd-cdddr (psd-cdr form))))


   ;; some more special cases: define, lambda, case, let, let* and letrec
   ((or (match '(define expr expr expr*) form)
        (match '(case expr expr expr*) form)
        (match '(lambda expr expr expr*) form)
        (match '(let expr expr expr*) form)
        (match '(let* expr expr expr*) form)
        (match '(letrec expr expr expr*) form))

    ;; check the indentation of first element
    (if (< (psd-expr-start-column (psd-car form))
	   current-indentation)
        (indent-warning (psd-car form))
        #f)

    ;; check the indentation of the second element
    (if (>= (psd-expr-start-column (psd-car form))
            (psd-expr-start-column (psd-cadr form)))
        (indent-warning (psd-cadr form))
        #f)

    ;; check that the third element is to the right of the first
    (if (>= (psd-expr-start-column (psd-car form))
            (psd-expr-start-column (psd-caddr form)))
        (indent-warning (psd-caddr form))
        #f)

    ;; the rest of the list should be lined up below the third element
    (check-sequence-indentation (psd-expr-start-column (psd-caddr form))
				(psd-expr-start-line (psd-caddr form))
				(psd-cdddr form)))

   ;; case (elem elem ...)
   ((and (psd-pair? form)
         (psd-pair? (psd-cdr form)))

    ;; check the indentation of first element
    (if (< (psd-expr-start-column (psd-car form))
           current-indentation)

        ;; watch out for quote, because our reader transforms 'foo to
        ;; (quote foo), with the opening paren and quote in the same
        ;; position
        (if (and (= (psd-expr-start-column (psd-car form))
                    current-indentation)
                 (= (psd-expr-start-line form)
                    (psd-expr-start-line (psd-car form)))
                 (psd-symbol? (psd-car form))
                 (eq? (pexp->sexp (psd-car form))
                      'quote))

            ;; it was quote, don't issue a warning
            #f

            ;; otherwise complain
            (indent-warning (psd-car form)))

        #f)

    ;; check the indentation of the second element
    (if (> (psd-expr-start-column (psd-car form))
           (psd-expr-start-column (psd-cadr form)))
        (indent-warning (psd-cadr form))
        #f)

    ;; the rest of the list should be lined up below the second element
    (check-sequence-indentation (psd-expr-start-column (psd-cadr form))
				(psd-expr-start-line (psd-cadr form))
				(psd-cddr form)))))

;;;
;;; Check that a sequence of expressions is lined up properly.
;;;

(define (check-sequence-indentation indent line form)

  (cond
   
   ;; the list ended
   ((psd-null? form) #f)
   
   ;; this was a dotted pair
   ((not (psd-pair? form))
    (if (or (= (psd-expr-start-line form)
	       line)
	    (>= (psd-expr-start-column (psd-car form))
		indent))
	
	;; everything ok
	#f
	
	(indent-warning form)))
   
   ;; this element is correctly indented
   ((= (psd-expr-start-column (psd-car form))
       indent)
    (check-sequence-indentation indent
				(psd-expr-start-line (psd-car form))
				(psd-cdr form)))

   ;; elements are on the same line
   ((= (psd-expr-start-line (psd-car form))
       line)
    (check-sequence-indentation indent
				line
				(psd-cdr form)))
        

   ;; this element was incorrectly indented
   (else
    (indent-warning (psd-car form))
    (check-sequence-indentation indent
				(psd-expr-start-line (psd-car form))
				(psd-cdr form)))))

;;;
;;; General warning about indentation.
;;;

(define (indent-warning form)
  (warning form "Indentation does not match the logical structure."))
                            