;;;
;;;  F R A M E S   T O O L B O X
;;;  = = = = = =   = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems
;;;    AssociativeNetworks

; This toolbox supports a simple frame toolbox catering for property
; inheritance, defaults and demon procedures. The toolbox is built
; upon the AN toolbox using the property inheritance links labelled by
; AKO to provide the inheritance and the user defined links are used
; to hold the user defined slots for the frame. Such a user defined
; frame contains a list of facets enabling the default and demon
; procedures to be specified via the following standard facet types:
; VALUE      - supply a particular value
; DEFAULT    - supply a default value in the absence of any other value
; IF-NEEDED  - a demon procedure to be called to supply a value if one
;                 is requested but not stored
; IF-ADDED   - a demon procedure to be called if a value is added to
;                 the frame
; IF-REMOVED - a demon procedure to be called if a value is deleted
;                 from the frame
 
; The following data types are used:

; FACET        :a list of <aType aValueList> where aType is one of the
;                  standard facet types
; FACET-LIST   :a list of FACETs
; AKO-SLOT     :a list of (AKO aListOfFrameNames)
; SLOT         :a list of (aSlotName aFacetList)
; SLOT-LIST    :a list of SLOTs including up to one AKO-SLOT
; FRAME        :a list of (aSymbol aSlotList)
; FN           :a list of FRAMEs                                


; ========== fns on FACETs ==================================
; a list of (aType aValueList) where aType is one of the standard
; facet types: VALUE, DEFAULT, IF-NEEDED, IF-ADDED, IF-REMOVED

(define MakeFacet (lambda (aType aValueList) (list aType aValueList)))

(define GetFacetType (lambda (aFacet) (car aFacet)))

(define GetValueList (lambda (aFacet) (cadr aFacet)))

(define VALUE (lambda (aValue) (list 'VALUE aValue)))

(define ValueFacet?
  (lambda (aFacet) (eq? 'VALUE (GetFacetType aFacet))))

(define DEFAULT (lambda (aValue) (list 'DEFAULT aValue)))

(define DefaultFacet?
  (lambda (aFacet) (eq? 'DEFAULT (GetFacetType aFacet))))

(define IF-NEEDED
  (lambda (aDemonProc) (list 'IF-NEEDED aDemonProc)))

(define IF-NEEDEDFacet?
  (lambda (aFacet) (eq? 'IF-NEEDED (GetFacetType aFacet))))

(define IF-ADDED
  (lambda (aDemonProc) (list 'IF-ADDED aDemonProc)))

(define IF-ADDEDFacet?
  (lambda (aFacet) (eq? 'IF-ADDED (GetFacetType aFacet))))

(define IF-REMOVED
  (lambda (aDemonProc) (list 'IF-REMOVED aDemonProc)))

(define IF-REMOVEDFacet?
  (lambda (aFacet) (eq? 'IF-REMOVED (GetFacetType aFacet))))

(define DemonFacet?
  (lambda (aFacet)
    (or (IF-NEEDEDFacet? aFacet)
        (IF-ADDEDFacet? aFacet)
        (IF-REMOVEDFacet? aFacet))))

(define GetDemonProc (lambda (aDemonFacet) (cadr aDemonFacet)))

(define InvokeDemon
  (lambda (aDemonFacet . optionalArgs)
    (apply (GetDemonProc aDemonFacet) optionalArgs)))

(define PrintFacet
  (lambda (aFacet)
    (DisplayList
      (cond ((ValueFacet? aFacet)      "(VALUE ")           
	    ((DefaultFacet? aFacet)    "(DEFAULT ")
	    ((IF-NEEDEDFacet? aFacet ) "(IF-NEEDED ")
	    ((IF-ADDEDFacet? aFacet)   "(IF-ADDED ")
	    ((IF-REMOVEDFacet? aFacet) "(IF-REMOVED "))
      (GetValueList aFacet)
      ")")))

; ========== fns on FACET-LISTs =============================
; lists of FACETs

(define NoMoreFacets? (lambda (aSlotList) (null? aSlotList)))

(define FirstFacet (lambda (aSlotList) (car aSlotList)))

(define RestOfFacets (lambda (aSlotList) (cdr aSlotList)))

; ========== fns on SLOTs ===================================
; These are different from the AN Toolbox ones except for the
; ones that act on AKO slots. Either an AKO slot, a list of the form:
;        (AKO aListOfFrameNames)
; or a user defined slot, a list of the form
;        (aSlotName aFacetList)

; AKO         - as for AN Toolbox, use as in (AKO any_number_of_SLOTs)

(define SLOT
  (lambda (aSlotName . aFacetList) (list aSlotName aFacetList)))

(define MakeSlotFromList
  (lambda (aSlotName aFacetList) (list aSlotName aFacetList)))

(define GetSlotName GetLinkName)
;               from AN Toolbox, use as in
;               (GetSlotName aSlot)

; AKOSlot?    - as for AN Toolbox, use as in
;              (AKOSlot? aSlot)

(define GetFrameList
  (lambda (anAKOSlot)
    (if (AKOSlot? anAKOSlot)
        (cdr anAKOSlot)
        (Fatal-Error "GetFrameList:" anAKOSlot
                     "is not an AKO slot"))))

(define GetFacetList
  (lambda (aSlot)
    (if (AKOSlot? aSlot)
        (Fatal-Error "GetFacetList:" "Action not valid on AKO slot" aSlot)
        (cadr aSlot))))

(define PrintSlot
  (lambda (aSlot)
    (if (AKOSlot? aSlot)
        (DisplayList "(AKO" (GetFrameList aSlot) ")")
        (begin (DisplayList "(SLOT" (GetSlotName aSlot))
               (for-each (lambda (aFacet)
                           (display " ")
                           (PrintFacet aFacet))
                         (GetFacetList aSlot))
               (display ")")))))

; ========== fns on SLOT-LISTs ==============================
; lists of SLOTs

; NoMoreSlots? - as for AN Toolbox, use as in
;               (NoMoreSlots? aSlotList)

; FirstSlot    - as for AN Toolbox, use as in
;               (FirstSlot aSlotList)

; RestOfSlots  - as for AN Toolbox, use as in
;               (RestOfSlots aSlotList)


; ========== fns on FRAMEs ==================================
; list of the form (aSymbol aSlotList)
; uses the fns on CONCEPTs from AN Toolbox

(define FRAME CONCEPT) 
;               from AN Toolbox, use as in
;               (FRAME aFrameName any_number_of_slots)

(define MakeFrameWithList MakeConceptWithList) 
;               from AN Toolbox, use as in
;               (MakeFrameWithList aFrameName aSlotList)

(define GetFrameSymbol GetConceptSymbol)
;               from AN Toolbox, use as in
;               (GetFrameSymbol aFrame)

; GetSlots    - as for AN Toolbox, use as in
;              (GetSlots aFrame)

(define PrintFrame
  (lambda (aFrame)
    (DisplayLine (GetFrameSymbol aFrame) ":")
    (for-each (lambda (aSlot)
                (display "      ")
                (PrintSlot aSlot)
                (newline))
              (GetSlots aFrame))))


; ========== fns on FNs =====================================
; lists of FRAMEs - uses the fns on ANs from AN Toolbox

(define MakeFN MakeAN)
;               from AN Toolbox, use as in
;               (MakeFN any_number_of_frames)

(define EmptyFN? EmptyAN?)
;               from AN Toolbox, use as in
;               (EmptyFN? anFN)

(define FirstFrame FirstConcept)
;               from AN Toolbox, use as in
;               (FirstFrame anFN)

(define RestOfFrames RestOfConcepts)
;               from AN Toolbox, use as in
;               (RestOfFrames anFN)

(define AddFrame AddConcept)
;               from AN Toolbox, use as in
;               (AddFrame anFN aFrame)

(define RemoveFrame RemoveConcept)
;               from AN Toolbox, use as in
;               (RemoveFrame anFN aFrame)

(define RemoveSlot RemoveLink)
;               from AN Toolbox, use as in
;               (RemoveSlot anFN aFrameSymbol aSlotName)

(define AddSlot AddLink)
;               from AN Toolbox, use as in
;               (AddSlot anFN aFrameSymbol aSlot)
    
; AddAKOLink     - as for AN Toolbox, use as in
;          (AddAKOLink anFN aFrameSymbol superClassFrameSymbol)

; RemoveAKOLink  - as for AN Toolbox, use as in
;          (RemoveAKOLink anFN aFrameSymbol superClassFrameSymbol)

(define PrintFN
  (lambda (anFN)
    (DisplayLine "*** Frame Network ***")
    (for-each (lambda (aFrame)
                (display "  ")
                (PrintFrame aFrame))
              anFN)
    (newline)))
               
; ========== query fns on FNs ===============================

(define FindAllFrameNames FindAllConcepts)
;               from AN Toolbox, use as in
;               (FindAllFrames anFN)

(define FindAllSlotNames FindAllLinkNames)
;               from AN Toolbox, use as in
;               (FindAllSlotNames anFN)

(define FindFrame FindConcept)
;               from AN Toolbox, use as in
;               (FindFrame anFN aFrameSymbol)

(define FindSlot
  (lambda (aFrame aSlotName) (assoc aSlotName (GetSlots aFrame))))

(define FindFacet
  (lambda (aSlot aFacetType) (assoc aFacetType (GetFacetList aSlot))))

; GetAKOSlot  - as for AN Toolbox, use as in
;               (GetAKOSlot anFN aFrameSymbol)

; GetAKOLinks - as for AN Toolbox, use as in
;               (GetAKOLinks anFN aFrameSymbol)

; GetAKOChain - as for AN Toolbox, use as in
;               (GetAKOChain anFN aFrameSymbol)

(define TriggerDemons
  (lambda (anFN aFrameName aSlotName aDemonType aValue)
    ; invoke the demons procedures of type aDemonType contained in the
    ; slots aSlotName on the chain of AKO links back from the aFrameName
    ; frame. Each demon is called with parameters:
    ;    (demon anFN aFrameName aSlotName aValue)
    ; and returns #f to suppress the adddition/deletion in which case
    ; TriggerDemons also returns #f imediately, OR the demon returns a
    ; new FN that will be passed to the next demon etc. If none of the
    ; demons return #f then the FN returned from the last demon is
    ; returned.
    (do ((frames (GetAKOChain anFN aFrameName) (cdr frames))
         (returnedFN anFN))
        ((or (not returnedFN) (null? frames)) returnedFN)
        (let ((aFrame (FindFrame returnedFN (car frames))))
          (if aFrame
              (let ((aSlot (FindSlot aFrame aSlotName)))
                (if aSlot
                    (let ((demon (FindFacet aSlot aDemonType)))
                      (if demon
                          (set! returnedFN
                                (InvokeDemon demon
                                             returnedFN
                                             aFrameName
                                             aSlotName
                                             aValue))
                          #f))
                    #f))
              #f)))))
  
(define RemoveFacet
  (lambda (anFN aFrameName aSlotName aFacetName)
    ; remove facet with name aFacetName from slot with name aSlotName
    ; of frame with aFrameName in anFN and return the updated FN. If
    ; aFacetName is a VALUE facet then invoke all IF-REMOVED demon
    ; procedures of this frame and in the AKO chain of super frames of
    ; this frame
    
    (define removeAux
      (lambda (aFacetList aFacetName)
        (cond ((NoMoreFacets? aFacetList) #f)
              ((eq? aFacetName
		    (GetFacetType (FirstFacet aFacetList)))
               (RestOfFacets aFacetList))
              (else (cons (FirstFacet aFacetList)
			  (removeAux (RestOfFacets aFacetList)
				     aFacetName))))))
    
    (let ((frame (FindFrame anFN aFrameName)))
      (if (not frame)
          anFN
          (let ((aSlot (FindSlot frame aSlotName)))
            (if (not aSlot)
                anFN
                (let ((facet (FindFacet aSlot aFacetName)))
                  (if (not facet)
                      anFN
                      (let
                        ((newFN (if (eq? 'VALUE aFacetName)
                                    (TriggerDemons anFN
						   aFrameName
						   aSlotName
						   'IF-REMOVED
						   (GetValueList facet))
                                    anFN)))
                        (if newFN
                            (AddSlot newFN
                                     aFrameName
                                     (MakeSlotFromList
                                       aSlotName
				       (removeAux (GetFacetList aSlot)
						  aFacetName)))
                            anFN))))))))))

(define RemoveAValue
  (lambda (anFN aFrameName aSlotName)
    ; remove the value facet from slot with the name aSlotName
    ; from the frame with the name aFrameName from the network
    ; anFN and return the updated network
    (RemoveFacet anFN aFrameName aSlotName 'VALUE)))

(define AddFacet
  (lambda (anFN aFrameName aSlotName aFacet)
    ; add aFacet to slot with aSlotName to frame with aFrameName in
    ; anFN and return the updated FN - if the facet is a VALUE facet
    ; and the slot has an IF-ADDED facet then invoke the demon in this
    ; frame and also all such demons in the AKO chain of super frames
    (let* ((newFN (RemoveFacet anFN
                               aFrameName
                               aSlotName
                               (GetFacetType aFacet)))
           (aFrame (FindFrame newFN aFrameName)))
      (if (not aFrame)
          (AddSlot newFN aFrameName (SLOT aFacet))
          (let ((aSlot (FindSlot aFrame aSlotName))
                (newerFN (if (ValueFacet? aFacet)
                             (TriggerDemons newFN
                                            aFrameName
                                            aSlotName
                                            'IF-ADDED
                                            (GetValueList aFacet))
                             newFN)))
            (if newerFN
                (if (not aSlot)
                    (AddSlot newerFN
                             aFrameName
                             (SLOT aSlotName aFacet))
                    (AddSlot newerFN
                             aFrameName
                             (MakeSlotFromList
                               aSlotName
			       (append (GetFacetList aSlot) (list aFacet)))))
                newFN))))))
  
(define AddAValue
  (lambda (anFN aFrameName aSlotName aValue)
    ; add a value facet (VALUE aVALUE) to slot with the name
    ; aSlotName of the frame with the name aFrameName in the
    ; network anFN and return the updated network
    (AddFacet anFN aFrameName aSlotName (VALUE aValue))))

(define GetAValue
  (lambda (anFN aFrameName aSlotName)
    ; return the value of the VALUE facet of the slot with the name
    ; aSlotName of the frame with the name aFrameName of the network
    ; anFN. If there is no VALUE facet return the value in the DEFAULT
    ; facet or, if none but there is an IF-NEEDED demon call that for
    ; the value. If all these fail try each frame on the chain of AKO
    ; super frames for such a slot looking for VALUE, DEFAULT and then
    ; IF-NEEDED facets to supply the value. If all these fail then a
    ; value of "NOTKNOWN" is returned
    (do ((nameList (GetAKOChain anFN aFrameName) (cdr nameList))
         (foundValue? #f)
         (returnedValue #f))
        ((or foundValue?
             (null? nameList)) (if foundValue?
                                   returnedValue
                                   "NOTKNOWN"))
        (let ((frame (FindFrame anFN (car nameList))))
          (if (not frame)
              #f
              (let ((slot (FindSlot frame aSlotName)))
                (if (not slot)
                    #f
                    (cond ((FindFacet slot 'VALUE)
                           (set! foundValue? #t)
                           (set! returnedValue (GetValueList
						(FindFacet slot 'VALUE))))
                          ((FindFacet slot 'DEFAULT) 
                           (set! foundValue? #t)
                           (set! returnedValue (GetValueList
						(FindFacet slot 'DEFAULT))))
                          ((FindFacet slot 'IF-NEEDED)
                           (set! foundValue? #t)
                           (set! returnedValue
                                 (InvokeDemon
                                  (FindFacet slot 'IF-NEEDED)
                                  anFN
                                  aFrameName
                                  aSlotName)))))))))))
