;;;
;;;       T O Y L A N D
;;;       = = = = = = =

; example from section 3.6.3

; This example is patterned after the famous SHRDLU program of
; T. Winograd. It uses FRAMES to store propeties about a series
; of blocks standing on a table and possibly stacked on top of
; each other. Only two-dimensional side views of the blocks are
; available. The data base knows about various properties of
; different shapes such as squares and types of triangles and
; and when gives a new block tries to decide what type of block
; it is with the information supplied. If necessary it tries to
; refine this information by requesting new information from the
; user.

; the code in the book on p341 uses ``case'' but some systems do
; not have this so it has been rewritten.
; (define CharacterizeWeight
;   (lambda (aNetwork aFrameName aSlotName)
;     (case (GetAValue ToyLand aFrameName 'Size)
;       ((Small Little Tiny) 'Light)
;       ((Large Big Huge)    'Heavy)
;       (("NOTKNOWN")         "NOTKNOWN"))))

(define CharacterizeWeight
  (lambda (aNetwork aFrameName aSlotName)
    (let ((val (GetAValue ToyLand aFrameName 'Size)))
     (cond ((member val '(Small Little Tiny)) 'Light)
	   ((member val '(Large Big Huge))    'Heavy)
	   ((eq? val "NOTKNOWN")              "NOTKNOWN")))))

(define CheckIfRemovalPossible
  (lambda (aNetwork aFrameName aSlotName aBlock)
    ; prevent removal of Base and First components of a Compound object
    ; (if a top there is present)
    (if (equal? (GetAValue aNetwork aFrameName 'Top)
		"NOTKNOWN")
        aNetwork
        (begin 
           (DisplayLine aBlock
			"supports a bridge -"
			"since I hate vandalism I won't remove it")
	   #f))))

(define CheckIfValidSupport
  (lambda (aNetwork aFrameName aSlotName anObject)
    ; check whether this object can support others
    (let ((supports (GetAValue aNetwork anObject 'Supports)))
      (if (eq? supports 'CanDo)
          aNetwork
          (begin 
	    (DisplayLine "Object" anObject "is not fit to support anything")
	    #f)))))

(define CheckIfHas2Supports
  (lambda (aNetwork aFrameName aSlotName anObject)
    ; check that this object has both a left and a right support
    (let ((left (GetAValue ToyLand aFrameName 'LeftSupport))
          (right (GetAValue ToyLand aFrameName 'RightSupport)))
      (if (or (equal? left  "NOTKNOWN")
              (equal? right "NOTKNOWN"))
          (begin 
	    (DisplayLine "Object" anObject
			 "can not be placed, since it"
			 "would be unsupported in this"
			 aFrameName "structure")
	    #f)
          aNetwork))))

(define ToyLand
  ; defines the initial state of our knowledge base:
  ; a few shapes and 2 block objects
  (MakeFN
   ; generics
   (FRAME 'Bridge 
          (SLOT 'LeftSupport  
                (IF-REMOVED CheckIfRemovalPossible)
                (IF-ADDED   CheckIfValidSupport))
          (SLOT 'RightSupport 
                (IF-REMOVED CheckIfRemovalPossible)
                (IF-ADDED   CheckIfValidSupport))
          (SLOT 'Top      
                (IF-ADDED   CheckIfHas2Supports))
          (SLOT 'Size (DEFAULT 'Large)))
   (FRAME 'Block 
          (SLOT 'Colour (DEFAULT 'White))
          (SLOT 'Size   (DEFAULT 'Small))
          (SLOT 'Weight (IF-NEEDED CharacterizeWeight))
          (SLOT 'Supports (DEFAULT 'CanDo)))
   (FRAME 'Rectangle (AKO 'Block)
	  (SLOT 'Angles      (VALUE 4))
	  (SLOT 'Sides       (VALUE 4))
	  (SLOT 'EqualSides  (VALUE 2))
	  (SLOT 'EqualAngles (VALUE 4)))
   (FRAME 'Triangle (AKO 'Block)
	  (SLOT 'Angles      (VALUE 3))
	  (SLOT 'Sides       (VALUE 3))
	  (SLOT 'Supports    (VALUE 'Nothing)))
   (FRAME 'Square (AKO 'Rectangle)
	  (SLOT 'EqualSides  (VALUE 4)))
   (FRAME 'Isosceles (AKO 'Triangle)
	  (SLOT 'EqualSides  (VALUE 2))
	  (SLOT 'EqualAngles (VALUE 4)))
   (FRAME 'Equilateral (AKO 'Triangle)
	  (SLOT 'EqualSides  (VALUE 3))
	  (SLOT 'EqualAngles (VALUE 3)))
   ; individuals
   (FRAME 'Square-1 (AKO 'Square)
	  (SLOT 'Colour      (VALUE 'Blue)))
   (FRAME 'Triangle-1 (AKO 'Triangle)
	  (SLOT 'Size        (VALUE 'Big))) ))

(define WhatIs
  (lambda initialAttributes
    
    (define satisfiesAttribute
      (lambda (aFrameList anAttribute)
        ; return only those frames in aFrameList that satisfy anAttribute
        (let ((attrname  (car anAttribute))
              (attrvalue (cadr anAttribute)))
          (do ((returnedList ())
               (toTestList aFrameList (cdr toTestList))
               (aFrameName #f))
              ((null? toTestList) returnedList)
              (set! aFrameName (car toTestList))
              (if (not (equal? attrvalue (GetAValue ToyLand
						    aFrameName
						    attrname)))
                  #f
                  (set! returnedList (append returnedList
					     (list aFrameName))))))))
    
    ; try and discover what kind of object it might be
    (let ((possibleObjects (FindAllFrameNames ToyLand)))
      (for-each 
       (lambda (anAttribute)
         ; reduce the list of candidates using this attribute
         (set! possibleObjects
               (satisfiesAttribute possibleObjects anAttribute)))
       initialAttributes)
      (cond ((null? possibleObjects)
             (DisplayLine "Sorry, no such object is known to me"))
            ((= 1 (length possibleObjects))
             (DisplayLine "It might be a" (car possibleObjects)))
            (else
             ; now keep asking for another attribute to reduce the list 			           
             ; until only one (or none) is left
             (do ((list possibleObjects)
                  (newAttribute #f))
                 ((or (null? list) (= 1 (length list)))
                  (newline)
                  (if (null? list)
                      (DisplayLine "Sorry, no such object is known to me")
                      (DisplayLine "It might be a" (car list))))
                 (DisplayLine "We've narrowed it down to one of:")
                 (DisplayLine list)
                 (DisplayLine "Please appy further discrimination")
                 (set! newAttribute (read))
                 (newline)
                 ; try and reduce the list further
                 (set! list
                       (satisfiesAttribute list newAttribute))))))))

(define AddObject
  (lambda (aName parentClass . anAttributeList)
    (let ((aSlotList (map (lambda (anAttribute)
			    (SLOT (car anAttribute)
				  (VALUE (cadr anAttribute))))
			  anAttributeList)))
      (set! ToyLand
            (AddFrame ToyLand
		      (apply FRAME
			     (cons aName (cons (AKO parentClass)
					       aSlotList)))))
      (DisplayLine "New object" aName "added to network"))))


(define Show
  ; shows the state of a frame
  (lambda (aName)
    (let ((aFrame (FindFrame ToyLand aName)))
      (if (not aFrame)
          #f
          (PrintFrame aFrame)))))

(define NewBridge
  ; this procedure adds an uninstantiated "bridge" object
  (lambda (aName)
    (set! ToyLand
          (AddFrame ToyLand (FRAME aName (AKO 'Bridge))))
    (DisplayLine "New bridge" aName "added to network")))

(define AddToBridge
  ; this procedure lets us "build" bridges incrementally.
  ; It expects the name of an existing bridge, followed by
  ; the "position" in which the new component should go
  ; (one of: left, right, top), followed by the name of
  ; the component (which we have to create before we use it).
  (lambda (aBridgeName where anObject)
    (DisplayLine "Adding" anObject "to position"
                 where "of bridge" aBridgeName "of network")
    (set! ToyLand
          (AddAValue ToyLand
                     aBridgeName
		     ; again ``case'' code replaced
                     ; (case where ((Left)  'LeftSupport)
                     ;             ((Right) 'RightSupport)
                     ;             ((Top)   'Top)
                     ;             (else    (DisplayLine 
                     ;                       "I don't know what kind of"
                     ;                       "position '" where
                     ;                       "' is supposed to be")
                     ;                      (DisplayLine 
                     ;                       "- let's start at the 'top'")
                     ;                      'Top))
                     (cond ((eq? where 'Left)  'LeftSupport)
			   ((eq? where 'Right) 'RightSupport)
			   ((eq? where 'Top) 'Top)
			   (else (DisplayLine "I don't know what kind of"
					      "position '" where
					      "' is supposed to be")
				 (DisplayLine "- let's start at the 'top'")
				 'Top))
                     anObject)) #t))

