; ========== a simple Discrete Event Simulator ==========

; ----- a MONITOR flavour -----

(defFlavour Monitor (ako Actor) 
  (ivars clockTime agenda simTime dataCollectors distributions
  tracing?) setivars getivars testivars)

; <clockTIme> holds the model's current clock time (a real number)
; <agenda> is an object of flavour "SequencingSet"
; <simTime> limits a simulation's duration
; <dataCollectors> and <distributions> are lists of entities - used for reset! and report!

(defMethod (simulate! Monitor) (aTimeLimit) 
  (set! clockTime 0)
  (set! simTime aTimeLimit)
  (if (not (self 'agenda?)) (set! agenda (SequencingSet 'new)))
  #t )

(defMethod (addDataCollectors! Monitor) anEntityList 
  (for-each (lambda (anEntity) 
             (self 'dataCollectors! 
                   (if (pair? dataCollectors)
                       (cons anEntity dataCollectors)
                       (List anEntity))) )
             anEntityList))

(defMethod (addDistributions! Monitor) anEntityList 
  (for-each (lambda (anEntity) 
             (self 'distributions!
                   (if (pair? distributions)
                       (cons anEntity distributions)
                       (List anEntity))) )
             anEntityList))

(defMethod (startUp! Monitor) ()
    (if (>? simTime clockTime)
        (self 'continue 'resumeProcesses)
        (displayLine "time limit too low - no simulation started !!!")))

(defMethod (schedule! Monitor) (anObject aMethodName aTime)
  (if (self 'tracing?) 
      (begin (displayLine "* MONITOR: at:" clockTime "another" aMethodName
                          " event scheduled for:" aTime)))               
  (if (<? aTime clockTime)
      (error "clocks can't run backwards !!!" clockTime)
      (agenda 'noteEvent! aTime anObject aMethodName) ))

(defCoroutine (resumeProcesses Monitor) ()
  (let ((next nil))
    (while (not (self 'finished?))          
           (begin 
            ; get first event on the agenda  
            (set! next (agenda 'nextEvent!))
            ; update the clock to this event            
            (set! clockTime (next 'timeOfOccurence))
            ; execute event
            (if (<=? clockTime simTime)
                    (begin (if (self 'tracing?)
                               (begin (displayLine "* MONITOR: --- executing ---")
                                      (next 'show)))                                       
                           ; transfer control to the appropriate process 
                           ; (coroutine) of the relevant actor
                           (self 'resume (next 'actorRef)
                                         (next 'processName))))))
      (displayLine "*** simulation stops at:" clockTime)))

(defMethod (finished?  Monitor) ()
  (or (agenda 'empty?) (>? clockTime simTime)))

(defMethod (reset!   Monitor) () 
  (set! tracing?  #f)
  (set! clockTime 0)
  (agenda 'clear!)
  (for-each (lambda (anEntity) (anEntity 'reset!)) dataCollectors)
  (for-each (lambda (anEntity) (anEntity 'reset!)) distributions)
  #t)

(defMethod (show Monitor) ()
  (define toGo (if (or (<=? simTime 0) (<=? clockTime 0)) 0 (- simTime clockTime)) ) 
  (displayLine "+++++ Monitor STATE +++++")
  (displayLine "... Clock Time:" clockTime "...") 
  (displayLine "->" (if (>? toGo 0) toGo 0) "to go !")
  (agenda 'show))

(defMethod (report Monitor) ()
  (self 'show) (newline)
  (displayLine "+++ data collectors:")
  (for-each (lambda (anEntity) (anEntity 'show) (newline)) dataCollectors)
  (displayLine "+++ distributions:") 
  (for-each (lambda (anEntity) (anEntity 'show) (newline)) distributions))

; ----- flavour SequencingSet ----- 
      
(defFlavour SequencingSet (ako Vanilla) (ivars contents) 
  setivars getivars testivars)

; Sequencing sets contain a list of event notices.
; Event notices are sorted, so that the "next imminent event" will always 
; be at the head of this list. 

(defMethod (empty? SequencingSet) () (not (self 'contents?)))

(defMethod (clear! SequencingSet) () (self 'contents! nil)) 

(defMethod (noteEvent! SequencingSet) (aTime anObject aMethodName)
  (define notice (EventNotice 'new))
  (define AgendaSearchedSoFar nil)   
  ; this is used to remember head of agenda while "cdr-ing" down the tail
                                  
  (define (insert! anEventNotice aSequencingSet)
    
    (define nextEvent (car aSequencingSet))    
            
    (if (not (null? aSequencingSet))
        ; then look for the right place to insert it !
        (begin
        (if (>? (nextEvent 'timeOfOccurence) 
                (anEventNotice 'timeOfOccurence))
            ; insert here, by appending to the tail of the list,
            ; and appending this structure to the unchanged head 
            (set! AgendaSearchedSoFar
                  (append AgendaSearchedSoFar 
                          (append (list anEventNotice)
                                   aSequencingSet)))
            ; keep on looking in the list's cdr - 
            ; remembering what we chopped off (in "AgendaSearchedSoFar")
            (begin (set! AgendaSearchedSoFar 
                         (append AgendaSearchedSoFar (list nextEvent))) 
                         (insert! anEventNotice (cdr aSequencingSet)))))
        ; else put it at the end
        (set! AgendaSearchedSoFar 
              (append AgendaSearchedSoFar (list anEventNotice))))
    AgendaSearchedSoFar)
  
  (notice 'timeOfOccurence! aTime)
  (notice 'actorRef!        anObject)
  (notice 'processName!     aMethodName)
  ; now insert "notice" at its proper place in the agenda 
  ; - "insert!" will return an updated SequencingSet
  (self 'contents! (insert! notice (self 'contents))) nil)

(defMethod (nextEvent SequencingSet) () (car contents))

(defMethod (nextEvent! SequencingSet) ()
  (define next nil)
  (if (self 'empty?)
      (displayLine "no more events in system !!!")
      (begin (set! next     (car contents))
             (set! contents (cdr contents)))) next)

(defMethod (show SequencingSet) ()
  (if (self 'empty?) 
      (displayLine "----- no events in system -----") 
      (begin (displayLine "----- scheduled for execution -----")
             (for-each (lambda (anEvent) (anEvent 'show))
                       (self 'contents)))))

; ----- flavour EventNotice -----                               

(defFlavour EventNotice (ako Vanilla) 
  (ivars timeOfOccurence actorRef processName)
  setivars getivars testivars)

; Event Notices encapsulate all information relevant to a particular event -
; its nature (defined by a process attached to an actor object), and 
; the clock time at which it is about to occur.

(defMethod (show EventNotice) ()
  (displayLine ". at:" (self 'timeOfOccurence) "a" (self 'processName) "event."))
    
; ----- flavour Process -----
  
; Processes define objects with life cycles

(defFlavour Process (ako Actor) (ivars name monitor) 
  setivars getivars testivars)

(defMethod (hold Process) (aMethodName aDuration)
  (if (<=? aDuration 0) (set! aDuration 0))
  (monitor 'schedule! self aMethodName (+ (monitor 'clockTime) aDuration))
  (self 'resume monitor 'resumeProcesses))

(defMethod (proceed Process) (aMethodName) 
  (monitor 'schedule! self aMethodName (monitor 'clockTime))
  (self 'resume monitor 'resumeProcesses))

(defMethod (passivate Process) () (self 'resume monitor 'resumeProcesses))

(defMethod (show Process) () 
  (displayLine "+" name ", a Process of flavour" (self 'flavourName)))

;
;
;