
% OL(P): Object Layer for Prolog -- interface objects
% Version 1.1 for SICStus Prolog and QUINTUS Prolog
% (C) 1993 Markus P.J. Fromherz.  All Rights Reserved.
% (C) 1993 Xerox Corporation.     All Rights Reserved.

% user_interface implements a simple framework for a
% user interface in a single window using gmlib.
% The window can be in different states with different GM objects (buttons,
% sliders etc.) being active. After the window is created, the main
% interaction loop accepts events (typically mouse events) and sends them
% to the event handler. The event handler defines operations and state
% transitions.
% See demo/sort_interface.pl.ol for a sample use.


:- ensure_loaded(library(gmlib)).

object user_interface.

   :- publish start/0, init_interface/2, init/1, interact/2, handle_event/5,
              views/1, buttons/1, button_style/1, sliders/1, outputs/1,
              values/1, window/3.

   % start -- start the user interface
   start :-
      ::init_interface(InitialState, Window),
      ::interact(InitialState, Window).

   init_interface(InitialState, WindowName) :-
      ( started -> end ; true ), start,
      ::button_style(Style),
      ::views(Views),     this::init_views(Views),
      ::buttons(Buttons), this::init_buttons(Buttons, Style),
      ::sliders(Sliders), this::init_sliders(Sliders),
      ::outputs(Outputs), this::init_outputs(Outputs),
      ::values(Values),   this::init_values(Values),
      ::disable_objects,
      ::window(WindowName,Title,Window),
      this::init_window(WindowName,Title,Window),
      ::init(InitialState),
      ::enable_state_objects(InitialState).

   % interact(+State, +Window) -- main interaction loop; State is the current
   %    state of the interface (cf. active_objects/2 and help_texts/4)
   interact(State, Window) :-
      Window => waitevent(Event),
      ::handle_event(State, Event, Window, NewState, Next),
      !,
      ( Next = quit ->
           true
         ;
           ( NewState \== State -> ::enable_state_objects(NewState) ; true ),
           ::interact(NewState, Window)
      ).


   % *** initializations

   init_views(Views) :-
      % the assertions of object/2 are a hack around a problem of
      % object identity in gmlib
      retractall(object(_,_)),
      forall_in_do((Name,Width,Height),
                   Views,
                   ( View <= view(Width, Height), assert(object(Name,View)) )).

   init_buttons(Buttons, Style) :-
      forall_in_do((Name,Text),
                   Buttons,
                   Name <= button(Text, Name, Style)).

   init_sliders(Sliders) :-
      forall_in_do((Name,Text),
                   Sliders,
                   ( Slider <= slider(Text), assert(object(Name,Slider)) )).

   init_outputs(Outputs) :-
      forall_in_do((Name,Text),
                   Outputs,
                   Name <= output(Text)).

   init_values(Values) :-
      forall_in_do(Value,
                   Values,
                   ( functor(Value, N, A), functor(EValue, N, A),
                     retractall(EValue), assert(Value) )).

   init_window(Name, Title, Window) :-
      Name <= window(Title, Window),
      Name => open.

   % some defaults for elements of the user interface

   % views(-Views) -- defines the views; element format: (Name,Width,Height)
   views([]).

   % buttons(-Buttons) -- defines the buttons; element format: (Name,Text)
   buttons([(bt_quit,"Quit")]).

   % button_style(-Style) -- defines the style for all buttons
   button_style([style(fancy)]).

   % sliders(-Sliders) -- defines the sliders; element format: (Name,Text)
   sliders([]).

   % outputs(-Outputs) -- defines the output texts; element format: (Name,Text)
   outputs([(o_message,"That's a default window.")]).

   % values(-Values) -- defines facts used to hold values (needs to be improved)
   values([]).

   % window(-Name, -Title, -WindowDef) -- defines the window
   window(default_win, "Default Window",
          vbox([space,
                hbox([space,vbox([o_message,space,hbox([space,bt_quit,space])]),space]),
                space,
                help])) :-
      help <= text(2, 50, 1, "times6"),
      help => readonly.

   % init(-InitialState) -- does custom initializations and defines the
   %     initial state
   init(wait_for_quit).

   % in the following, there is one fact per state ...
   % active_objects(+State, -Objects) -- defines active GM objects in State
   active_objects(wait_for_quit, [bt_quit]).
   % help_text(+State, -Text) -- defines help text in State
   help_text(wait_for_quit, "Press Quit to quit.").

   % handle_event(+State, +Event, +Window, -NewState, -Next) -- handle Event (GM format)
   %    in state State; the next state will be NewState; Next will be either
   %    next or quit (interaction stops in the latter case)
   handle_event(AnyState, Event, _, AnyState, next) :-
      \+ (Event = button(_,bt_quit)),
      format("Event ~w in state ~w is not handled by the application.~n", [Event,AnyState]).
   handle_event(_wait_for_quit, button(_,bt_quit), Window, _, quit) :-
      Window => close.


   % *** enabling and disabling objects

   :- publish enable_state_objects/1, disable_objects/0,
              active_objects/2, help_text/2.

   % Remember to add new objects (especially buttons) to predicate
   % active_objects/2, and new states to predicates active_objects/2
   % and help_texts/4.

   % enable_state_objects(+State) -- enable (only) objects for this State
   enable_state_objects(State) :-
      ::disable_objects,
      ::active_objects(State, Objects),
      forall_in_do(O, Objects, send(O, enable)),
      ::help_text(State, Text),
      help => clear,
      help => insert(Text).

   disable_objects :-
      ::views(Views),
      forall_in_do((N,_,_), Views, send(N, disable)),
      ::buttons(Buttons),
      forall_in_do((N,_), Buttons, send(N, disable)),
      ::sliders(Sliders),
      forall_in_do((N,_), Sliders, send(N, disable)).


   % *** some default dialogs

   :- publish ok/2, banner_start/1, banner_end/0.

   % ok(+Text, -Name) -- produce dialog window with Text, ok and cancel
   %    buttons, and a field to enter Name
   ok(Text, Name) :-
      name <= input("Name", times10),
      bt_ok     <= button("Okay",   bt_ok),
      bt_cancel <= button("Cancel", bt_cancel),
      OK <= window("",
                   hbox([space,
                   vbox([space,
                         output(Text),
                         space,
                         frame(name),
                         space,
                         hbox([bt_ok,space,bt_cancel]),
                         space]),
                         space])),
      OK => open,
      OK => waitevent(E),
      name => in(NameString), name(Name, NameString),
      OK => close,
      E = button(_,bt_ok).
   
   % banner_start(+Text) -- produce banner window with message Text
   banner_start(Text) :-
      banner <= window("",
                       vbox([space,hbox([space,output(Text),space]),space])),
      banner => open.
   
   % banner_end -- close banner
   banner_end :-
      banner => close.

end_object user_interface.

forall_in_do(X, Xs, G) :-
   \+ (on(X, Xs), \+ G).

on(X, [X|_]).
on(X, [_|Xs]) :-
    on(X, Xs).

% for a bug in gmlib: using constants doesn't work for views and sliders
send(Object, Message) :-
   ( object(Object, View) ->
        View => Message
      ;
        Object => Message
   ).
