/*
 * callpat.c
 * Processes user annotations for call/success patterns.
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *
 * HISTORY:
 *	6/25/88 - Created DAR
 *	7/01/88 - added list processing
 */

/* header files */

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "state.h"
#include "lattice.h"

/* macros */
#include "macros.c"
#include "pgm_macros.c"
#include "gpmacros.c"
#include "state_macros.c"

extern SLOT		ListFunc, NilAtom;
extern Procedure	procedures[MaxProcs];
extern char		cmperrmsg[];

/* maximum number of variables, defined in pgm_typedefs.h */
CALSUCPR		lub_result[MaxVarTbl];

/*
 * HIGH or LOW level debugging messages on if HIBUG and LOBUG
 * defined respectively.
 */

/*#define HIBUG 1 */ 		/* high level debugging */
/*#define LOBUG	1 */		/* low level debugging */




/*
 * CALSUCPR *lubOfProc(SHORT proc_index)
 * Given the index to a procedure, compute the least
 * upper bound of all associated calling patterns.
 *
 * Currently, the result is stored in an array whose contents
 * are destroyed upon each invocation of this function.
 * As the base address of the array is returned, this may
 * easily be changed to actually allocate space for each invocation
 * should this prove to be useful
 */
CALSUCPR *lubOfProc(proc_index)
     SHORT proc_index;
{
  SHORT		i;
  CALLSUCC	*pattern;
  CALSUCPR	*calsuc;

  pattern = procedures[proc_index].call_succ;
  if (pattern == NULL) {
    /* if no call/success info available */
    for (i = 0; i < procedures[proc_index].arity; i++) {
      lub_result[i].call = DONTKNOW;
      lub_result[i].succ = DONTKNOW;
    } /* end for i */
  } else {
    /* initialize lub_result */
    for (i = 0; i < procedures[proc_index].arity; i++) {
      lub_result[i].call = EMPTY;
      lub_result[i].succ = EMPTY;
    } /* end for i */
    /* determine least upper bound of all call/success patterns */
    while (pattern != NULL) {
      calsuc = pattern->patterns;
      for (i = 0; i < procedures[proc_index].arity; i++) {
	lub_result[i].call = lub_callpat(lub_result[i].call,
					       calsuc[i].call);
	lub_result[i].succ = lub_callpat(lub_result[i].succ,
					       calsuc[i].succ);
      } /* end for i */
      pattern = pattern->next;
    } /* end while pattern != NULL */
  } /* end else */
#ifdef HIBUG
  printf("Least upper bound of call/succ patterns for %s/%d\n",
	 procedures[proc_index].head, procedures[proc_index].arity);
  printf("CALL\tSUCC\n");
  for (i = 0; i < procedures[proc_index].arity; i++)
    printf("%d\t%d\n", lub_result[i].call, lub_result[i].succ);
#endif
  return lub_result;
}


/*
 * SHORT success_pattern(SLOT *term, CALSUCPR *calsuc)
 * Search the given term for a success pattern that matches
 * the calling pattern.  It is assumed that the calling
 * patterns have been ordered, and the first match is the
 * the best.
 * If an exact match is found, nonzero is returned and
 * calsuc is updated to the corresponding success pattern.
 * Otherwise, 0 is returned, and the sucess pattern is set
 * to the calling pattern.
 *
 * Currently, calling patterns are ordered by declaring them
 * from the most general case to the most specific.
 * e.g. for is/2	:- callpat([is(d/g,g), is(g,g)]).
 * It is hoped that this will be changed in the future.
 */
SHORT success_pattern(term, calsuc)
     SLOT	*term;
     CALSUCPR	*calsuc;
{
  SHORT		functor, i, match;
  CALLSUCC 	*call_succ_list;
  CALSUCPR	*pattern;

  functor = GET_FUNCTOR(term);
  call_succ_list = procedures[functor].call_succ;
  while (call_succ_list != NULL) {
    pattern = call_succ_list->patterns;
    call_succ_list = call_succ_list->next;
    for (i=0; i < procedures[functor].arity; i++) {
      if (pattern[i].call == DONTKNOW)
	calsuc[i].succ = lub_instantiation(DONTKNOW, calsuc[i].call);
      else
	if (pattern[i].call != calsuc[i].call)
	  break;	/* inexact match -- look at next pattern */
      else
	calsuc[i].succ = pattern[i].succ;
    } /* end for */
    if (i == procedures[functor].arity) /* if all terms matched */
      return 1;
  } /* end while call_succ_list != NULL */
  
  /* no match, set success to same as call pattern */
  for (i=0; i < procedures[functor].arity; i++)
    calsuc[i].succ = calsuc[i].call;
  return 0;
} /* end success_pattern */


/*
 * callpat_dec(SLOT term)
 * term contains the argument of the callpat directive.
 * The argument may either be a single term, or a list of terms
 * Each term must be a predicate name of arity n.
 *
 * Collectively, its arguments form a call and success pattern.
 * That is, if the predicate is called with the arguments of
 * the indicated type, when it is resolved the arguments will
 * be bound to success type.
 *
 * Legal type names may be found by examing the
 * structure type_table in lattice.c
 *
 * Each argument is of the form call_type/success_type.  If the
 * argument is not further instantiated, we may abbreviate by
 * simply writing the call_type.
 *
 * Example:	:- callpat([append(g,g,f/g), append(f/g,f/g,g)]).
 * The above example gives two calling patterns for append/3.
 * The first states that if append is called with the first
 * two arguments ground and the last free, it will return with
 * all three arguments ground.  This could also have been written
 * as append(g/g, g/g, f/g).  The second calling pattern says
 * that when append is called with the first two arguments
 * free and the third ground, everything will be returned ground.
 */

void callpat_dec(term)
     SLOT term;
{
  SHORT		arity, proc, functor, i, call, succ, call_typ, succ_typ;
  SLOT		slot, head, tail, arg, args;
  CALLSUCC	*csrec;
  CALSUCPR	*cspair;
  char		go;
  
  arity = ARITY((SLOT *) term);
  if (arity != 1) {
    sprintf(cmperrmsg, ":- callpat/%d - arity must be 1\n", arity);
    cmperror(0, cmperrmsg);
  } else {
    tail = term += sizeof(SLOT); /* move to argument of callpat */
    DEREF(tail);
    go = 1;
    while (go) {
      functor = GET_FUNCTOR((SLOT *) tail);
      arity = ARITY((SLOT *) tail);
      if (CONS_CELL(tail)) {
	head = tail + sizeof(SLOT);	/* head of list */
	DEREF(head);
	NEXT_CELL(tail);		/* tail of list */
	DEREF(tail);
      } else {
	head = tail;	/* no need to DEREF, done last iteration */
	go = 0; /* end of list or no list */
      } /* end if CONS_CELL */
      /* head points to the predicate to be processed */
      proc = GET_FUNCTOR((SLOT *) head);
      if (proc != GET_FUNCTOR(&NilAtom)) {
#	ifdef HIBUG
	printf("Functor being processed %s/%d\n",
	       procedures[proc].head, procedures[proc].arity);
#	endif /* end DEBUG */
	arity = ARITY((SLOT *) head);
	args = head;
	Malloc_CallSucc(csrec);
	Malloc_CalSucPR(csrec->patterns, arity);
	cspair = csrec->patterns;
	if (procedures[proc].call_succ == NULL) {
	  procedures[proc].call_succ = csrec;
	  csrec->next = NULL;
	} else {
	  csrec->next = procedures[proc].call_succ;
	  procedures[proc].call_succ = csrec;
	} /* end if call_succ == NULL */
	for (i=0; i < arity; i++) {
	  /* process arguments */
	  arg = args += sizeof(SLOT);
	  DEREF(arg);
	  functor = GET_FUNCTOR((SLOT *) arg);
	  /* check if / */
	  if (! strcmp("/",procedures[functor].head) &&
	      procedures[functor].arity == 2) {
	    slot = arg += sizeof(SLOT);
	    DEREF(slot);
	    call = GET_FUNCTOR((SLOT *) slot);
	    call_typ = find_type(procedures[call].head);
	    slot = arg += sizeof(SLOT);
	    DEREF(slot);
	    succ = GET_FUNCTOR((SLOT *) slot);
	    succ_typ = find_type(procedures[succ].head);
#	    ifdef LOBUG
	    printf("call/succ: %s/%d=%d %s/%d=%d\n",
		   procedures[call].head, procedures[call].arity, call_typ,
		   procedures[succ].head, procedures[succ].arity, succ_typ);
#	    endif /* end DEBUG */
	  } else {
	    call = succ = GET_FUNCTOR((SLOT *) arg);
	    call_typ = find_type(procedures[call].head);
	    succ_typ = call_typ;
#	    ifdef LOBUG
	    printf("call=succ: %s/%d=%d \n", procedures[call].head,
		   procedures[call].arity, succ_typ);
#	    endif /*end DEBUG */
	  } /* end if/else ( "/" && arity 2) */
	  /* Store call & succ pairs for this argument */
	  cspair->call = call_typ;
	  cspair->succ = succ_typ;
	  cspair++;	/* storage for next argument */
	} /* end for(i=0; i<arity; i++) */
#       ifdef HIBUG
	printf("Current callpats for %s/%d\n",
	       procedures[proc].head, procedures[proc].arity);
	csrec = procedures[proc].call_succ;
	while (csrec != NULL) {
	  printf("------call/succ pat --------\n");
	  for (i=0; i<arity; i++)
	    printf("call %d succ %d\n",
		   csrec->patterns[i].call, csrec->patterns[i].succ);
	  csrec = csrec->next;
	}
#	endif /* end DEBUG code */
      } /* end if (proc != GET_FUNCTOR(&NilAtom)) */
    } /* while (go) */
  } /* end if (arity != 1) else { */
} /* end callpat_dec */


      
	
  
  
