%{
/*
 * PCN Abstract Machine Emulator
 * Authors:     Steve Tuecke and Ian Foster
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * parse_pdb.y	- yacc based parser for PDB
 *
 * Note: We've had some problems getting this to work with bison.  You
 * may need to use yacc...
 */

/*
#define PDB_PARSE_DEBUG
#define YYDEBUG 1
int yydebug = 1;
*/
    
/* Bison on some machines does not include this, but needs to... */
#if defined(iris) || defined(ksr1)
#include <alloca.h>
#endif

#define MAX_PARSE_STRING_LENGTH 256    
typedef union  {
    pt_node_t *	ptr;
    int 	ival;
    double	dval;
    char	sval[MAX_PARSE_STRING_LENGTH];
} pdb_yystype;
#define YYSTYPE pdb_yystype

#define NULL_NODE ((pt_node_t *) NULL)

static void yyerror();
static pt_node_t *pt_alloc_node();
static void free_pt_nodes();
static char *pt_alloc_string();
static double *pt_alloc_double();
static void yyp_clear_input();

%}

%start command

/*
 * The following are all the command name tokens
 */
%token <ival> PDBT_ABORT
%token <ival> PDBT_ASSIGN_VAR
%token <ival> PDBT_BREAK
%token <ival> PDBT_CONTINUE
%token <ival> PDBT_DEBUG
%token <ival> PDBT_DELETE
%token <ival> PDBT_DISABLE
%token <ival> PDBT_ENABLE
%token <ival> PDBT_HELP
%token <ival> PDBT_LIST
%token <ival> PDBT_LOAD
%token <ival> PDBT_MODULES
%token <ival> PDBT_MOVE
%token <ival> PDBT_NEXT
%token <ival> PDBT_NODEBUG
%token <ival> PDBT_PRINT
%token <ival> PDBT_PROCEDURES
%token <ival> PDBT_QUIT
%token <ival> PDBT_SHOW
%token <ival> PDBT_SOURCE
%token <ival> PDBT_STATUS
%token <ival> PDBT_SUMMARY
%token <ival> PDBT_SWITCH
%token <ival> PDBT_VARS

/*
 * The following are all the non-command name tokens
 */
%token <ival> PDBT_ACTIVE
%token <ival> PDBT_ALL
%token <ival> PDBT_BLOCK_SPEC
%token <ival> PDBT_BLOCK_SPEC_LIST
%token <ival> PDBT_COMMAND_BRANCH
%token <ival> PDBT_ERROR
%token <ival> PDBT_GENERAL_HELP
%token <ival> PDBT_GLOBSUSP
%token <ival> PDBT_INSTANCE
%token <ival> PDBT_INTEGER
%token <ival> PDBT_INTEGER_LIST
%token <ival> PDBT_NULL
%token <ival> PDBT_PENDING
%token <ival> PDBT_PRINT_EXPR_BRANCH
%token <ival> PDBT_PROCESS_RANGE
%token <dval> PDBT_DOUBLE
%token <ival> PDBT_REDUCTION
%token <ival> PDBT_SHOW_PROCESS
%token <sval> PDBT_STRING
%token <ival> PDBT_SUSPENSION
%token <sval> PDBT_TOKEN
%token <ival> PDBT_UNDEF
%token <ival> PDBT_UNKNOWN_HELP
%token <ival> PDBT_VAR_PRINT_ARRAY_SIZE
%token <ival> PDBT_VAR_PRINT_TUPLE_DEPTH
%token <ival> PDBT_VAR_PRINT_TUPLE_WIDTH
%token <ival> PDBT_VAR_GLOBAL_DL
%token <ival> PDBT_VAR_EMULATOR_DL
%token <ival> PDBT_VAR_GC_DL
%token <ival> PDBT_VAR_PARALLEL_DL
%token <ival> PDBT_VAR_REDUCTION_BREAK
%token <ival> PDBT_VAR_EMPTY_QUEUE_BREAK
%token <ival> PDBT_VAR_PRINT_ORPHANED
%token <ival> PDBT_VAR_MODULE
%token <ival> PDBT_VAR_PROCEDURE
%token <ival> PDBT_VAR_ARGS
%token <ival> PDBT_VAR_INSTANCE
%token <ival> PDBT_VAR_REDUCTION
%token <ival> PDBT_VAR_CURRENT_REDUCTION
%token <ival> PDBT_VAR_UNKNOWN
%token <sval> PDBT_VARIABLE
%token <ival> PDBT_VARSUSP
%token <sval> PDBT_WILDCARD
%token <sval> PDBT_WILDCARD_WRAP

/*
 * The following are needed types for all the commands rules.
 */
%type <ival> abort_token
%type <ptr>  abort_command
%type <ptr>  assign_var_command
%type <ival> break_token
%type <ptr>  break_command
%type <ival> continue_token
%type <ptr>  continue_command
%type <ival> debug_token
%type <ptr>  debug_command
%type <ival> delete_token
%type <ptr>  delete_command
%type <ival> disable_token
%type <ptr>  disable_command
%type <ival> enable_token
%type <ptr>  enable_command
%type <ival> help_token
%type <ptr>  help_command
%type <ival> list_token
%type <ptr>  list_command
%type <ival> load_token
%type <ptr>  load_command
%type <ival> modules_token
%type <ptr>  modules_command
%type <ival> move_token
%type <ptr>  move_command
%type <ival> next_token
%type <ptr>  next_command
%type <ival> nodebug_token
%type <ptr>  nodebug_command
%type <ival> print_token
%type <ptr>  print_command
%type <ival> procedures_token
%type <ptr>  procedures_command
%type <ival> quit_token
%type <ptr>  quit_command
%type <ival> show_token
%type <ptr>  show_command
%type <ival> source_token
%type <ptr>  source_command
%type <ival> status_token
%type <ptr>  status_command
%type <ival> summary_token
%type <ptr>  summary_command
%type <ival> switch_token
%type <ptr>  switch_command
%type <ival> vars_token
%type <ptr>  vars_command

/*
 * The following are needed types for all the non-command rules.
 */
%type <ptr>  assign_var_value
%type <ptr>  block_spec
%type <ptr>  block_spec_list
%type <ptr>  block_spec_program
%type <ptr>  block_spec_module
%type <ptr>  command_block
%type <ptr>  command_item
%type <ptr>  command_list
%type <ptr>  general_token
%type <ival> help_topic
%type <ptr>  integer
%type <ptr>  integer_list
%type <ival> integer_value
%type <ival> move_switch_queue
%type <ptr>  move_switch_process
%type <ival> move_switch_where
%type <ptr>  print_expr
%type <ptr>  print_item
%type <ptr>  double
%type <ival> show_instance
%type <ptr>  show_process
%type <ival> show_queue
%type <ival> show_reduction
%type <ival> show_undef
%type <ptr>  string
%type <ptr>  string_or_token
%type <ptr>  variable
%type <ptr>  wildcard
%type <ptr>  wildcard_wrap

    
%%

command
	: /* empty */
	| command_block
		{   yyp_parse_tree = $1; }
	| command_list
		{   yyp_parse_tree = $1; }
	| error
		{
		    yyerrok;
		    yyclearin;
		    yyp_clear_input();	/* defined in tok_pdb.c */
		    free_pt_nodes();	/* free all allocated nodes */
		    return(1);
		}
	;

command_block
	: '{' command_list '}'
		{   $$ = $2; }
	| command_item
		{   $$ = $1; }
	;

command_list
	: command_item ',' command_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_COMMAND_BRANCH);
#ifdef PDB_PARSE_DEBUG		    
		    printf("command branch\n");
#endif
		    node->ptr1 = $1;
		    node->ptr2 = $3;
		    $$ = node;
		}
	| command_item
		{   $$ = $1; }
	;

command_item
	: abort_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| assign_var_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| break_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| continue_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| debug_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| delete_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| disable_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| enable_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| help_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| list_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| load_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| modules_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| move_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| next_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| nodebug_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| print_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| procedures_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| quit_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| show_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| source_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| status_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| summary_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| switch_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	| vars_command
		{   $$ = $1; yyp_start_new_command = TRUE; }
	;



/********************************************************************
 *
 * abort
 */
abort_token	: PDBT_ABORT { $$ = $1; } ;
abort_command
	: abort_token
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("abort command\n");
#endif
		    $$ = pt_alloc_node(PDBT_ABORT);
		}
	;


/********************************************************************
 *
 * assign_var_command
 *
 * $var = <value>
 */
assign_var_command
	: variable '=' assign_var_value
		{
		    pt_node_t *node = pt_alloc_node(PDBT_ASSIGN_VAR);
#ifdef PDB_PARSE_DEBUG		    
		    printf("assign_var command\n");
#endif
		    node->ptr1 = $1;
		    node->ptr2 = $3;
		    $$ = node;
		}
	;

assign_var_value
	: integer
		{   $$ = $1; }
	| double
		{   $$ = $1; }
	| string
		{   $$ = $1; }
	| general_token
		{   $$ = $1; }
	;

    

/********************************************************************
 *
 * break
 */
break_token	: PDBT_BREAK  { $$ = $1; } ;
break_command
	: break_token block_spec_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_BREAK);
#ifdef PDB_PARSE_DEBUG		    
		    printf("break command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * continue
 */
continue_token	: PDBT_CONTINUE  { $$ = $1; } ;
continue_command
	: continue_token
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("continue command\n");
#endif
		    $$ = pt_alloc_node(PDBT_CONTINUE);
		}
	;


/********************************************************************
 *
 * debug [<module_name>:<proc_name>]*
 */
debug_token	: PDBT_DEBUG  { $$ = $1; } ;
debug_command
	: debug_token block_spec_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_DEBUG);
#ifdef PDB_PARSE_DEBUG		    
		    printf("debug command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * delete [<integer>]*
 */
delete_token	: PDBT_DELETE  { $$ = $1; } ;
delete_command
	: delete_token integer_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_DELETE);
#ifdef PDB_PARSE_DEBUG		    
		    printf("delete command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;

integer_list
	: /* empty */	{ $$ = NULL_NODE; }
	| integer_list integer
		{
		    pt_node_t *node = pt_alloc_node(PDBT_INTEGER_LIST);
#ifdef PDB_PARSE_DEBUG		    
		    printf("integer list\n");
#endif
		    node->ptr1 = $1;
		    node->ptr2 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * disable [<integer>]*
 */
disable_token	: PDBT_DISABLE  { $$ = $1; } ;
disable_command
	: disable_token integer_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_DISABLE);
#ifdef PDB_PARSE_DEBUG		    
		    printf("disable command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * enable [<integer>]*
 */
enable_token	: PDBT_ENABLE  { $$ = $1; } ;
enable_command
	: enable_token integer_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_ENABLE);
#ifdef PDB_PARSE_DEBUG		    
		    printf("enable command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * help [<topic>]
 *
 * <topic> can be any of the commands.
 * If the <topic> is not a valid command, the <topic> is passed
 * through as a token.
 */
help_token	: PDBT_HELP  { $$ = $1; } ;
help_command
	: help_token help_topic
		{
		    pt_node_t *node = pt_alloc_node(PDBT_HELP);
#ifdef PDB_PARSE_DEBUG		    
		    printf("help command\n");
#endif
		    node->i1 = $2;
		    $$ = node;
		}
	| help_token general_token
		{
		    pt_node_t *node = pt_alloc_node(PDBT_HELP);
#ifdef PDB_PARSE_DEBUG		    
		    printf("help on general token\n");
#endif
		    node->i1 = PDBT_UNKNOWN_HELP;
		    node->ptr1 = $2;
		    $$ = node;
		}
	| help_token variable
		{
		    pt_node_t *node = pt_alloc_node(PDBT_HELP);
#ifdef PDB_PARSE_DEBUG		    
		    printf("help for a variable\n");
#endif
		    node->i1 = PDBT_VARIABLE;
		    node->ptr1 = $2;
		    $$ = node;
		}
	| help_token
		{
		    pt_node_t *node = pt_alloc_node(PDBT_HELP);
#ifdef PDB_PARSE_DEBUG		    
		    printf("help command\n");
#endif
		    node->i1 = PDBT_GENERAL_HELP;
		    $$ = node;
		}
	;

help_topic
	: '='
		{   $$ = PDBT_ASSIGN_VAR; }
	| abort_token
		{   $$ = $1; }
	| break_token
		{   $$ = $1; }
	| continue_token
		{   $$ = $1; }
	| debug_token
		{   $$ = $1; }
	| delete_token
		{   $$ = $1; }
	| disable_token
		{   $$ = $1; }
	| enable_token
		{   $$ = $1; }
	| help_token
		{   $$ = $1; }
	| list_token
		{   $$ = $1; }
	| load_token
		{   $$ = $1; }
	| modules_token
		{   $$ = $1; }
	| move_token
		{   $$ = $1; }
	| next_token
		{   $$ = $1; }
	| nodebug_token
		{   $$ = $1; }
	| print_token
		{   $$ = $1; }
	| procedures_token
		{   $$ = $1; }
	| quit_token
		{   $$ = $1; }
	| show_token
		{   $$ = $1; }
	| source_token
		{   $$ = $1; }
	| status_token
		{   $$ = $1; }
	| summary_token
		{   $$ = $1; }
	| switch_token
		{   $$ = $1; }
	| vars_token
		{   $$ = $1; }
	;


/********************************************************************
 *
 * list [<queue>] [<process>]
 *
 * This takes the same arguments as the 'show' command.
 */
list_token	: PDBT_LIST  { $$ = $1; } ;
list_command
	: list_token show_queue show_process
		{
		    pt_node_t *node = pt_alloc_node(PDBT_LIST);
#ifdef PDB_PARSE_DEBUG		    
		    printf("list command\n");
#endif
		    node->i1 = $2;
		    node->ptr1 = $3;
		    $$ = node;
		}
	| list_token show_process
		{
		    pt_node_t *node = pt_alloc_node(PDBT_LIST);
#ifdef PDB_PARSE_DEBUG		    
		    printf("list command\n");
#endif
		    node->i1 = PDBT_ALL;
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * load
 */
load_token	: PDBT_LOAD  { $$ = $1; } ;
load_command
	: load_token string_or_token
		{
		    pt_node_t *node = pt_alloc_node(PDBT_LOAD);
#ifdef PDB_PARSE_DEBUG		    
		    printf("load command; %s\n", $2);
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * modules
 */
modules_token	: PDBT_MODULES  { $$ = $1; } ;
modules_command
	: modules_token
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("modules command\n");
#endif
		    $$ = pt_alloc_node(PDBT_MODULES);
		}
	;


/********************************************************************
 *
 * move [<queue>] [<process>] [<where>]
 *
 * <queue>:  This is one of the following:
 *	active
 *	pending
 *
 * <process>:  Same as for the 'show' command.
 *
 * <where>:	Integer
 */
move_token	: PDBT_MOVE  { $$ = $1; } ;
move_command
	: move_token move_switch_queue move_switch_process move_switch_where
		{
		    pt_node_t *node = pt_alloc_node(PDBT_MOVE);
#ifdef PDB_PARSE_DEBUG		    
		    printf("move command\n");
#endif
		    node->i1 = $2;
		    node->i2 = $4;
		    node->ptr1 = $3;
		    $$ = node;
		}
	;

move_switch_queue
	: PDBT_TOKEN
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch_queue \"%s\"\n", $1);
#endif
		    if ((strcmp($1,"active") == 0)
			     || (strcmp($1,"a") == 0)
			     || (strcmp($1,"A") == 0))
			$$ = PDBT_ACTIVE;
		    else if ((strcmp($1,"pending") == 0)
			     || (strcmp($1,"p") == 0)
			     || (strcmp($1,"P") == 0))
			$$ = PDBT_PENDING;
		    else
			YYERROR;
		}
	;

move_switch_process
	: integer_value '-' integer_value
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PROCESS_RANGE);
		    node->i1 = $1;
		    node->i2 = $3;
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch__process range %d - %d\n", node->i1, node->i2);
#endif
		    $$ = node;
		}
	| integer_value
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PROCESS_RANGE);
		    node->i1 = $1;
		    node->i2 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch_process range %d - %d\n",node->i1,node->i2);
#endif
		    $$ = node;
		}
	| show_instance
		{
		    pt_node_t *node = pt_alloc_node(PDBT_INSTANCE);
		    node->i1 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch_process instance %d\n", node->i1);
#endif
		    $$ = node;
		}
	| show_reduction
		{
		    pt_node_t *node = pt_alloc_node(PDBT_REDUCTION);
		    node->i1 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch__process reduction %d\n", node->i1);
#endif
		    $$ = node;
		}
	| show_undef
		{
		    pt_node_t *node = pt_alloc_node(PDBT_UNDEF);
		    node->i1 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch__process undef %d\n", node->i1);
#endif
		    $$ = node;
		}
	| block_spec
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch_process block spec\n");
#endif
		    $$ = $1;
		}
	| PDBT_TOKEN
		{
		    if (   (strcmp($1,"all") == 0)
			|| (strcmp($1,"a"  ) == 0) )
		    {
			pt_node_t *node = pt_alloc_node(PDBT_ALL);
#ifdef PDB_PARSE_DEBUG		    
			printf("move_switch_process all\n");
#endif
			$$ = node;
		    }
		    else
		    {
			YYERROR;
		    }
		}
	;
	
move_switch_where
	: /* empty */
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch_where implicit-end\n");
#endif
		    $$ = -1;
		}
	| integer_value
		{
		    $$ = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch_where %d\n", $1);
#endif
		}
	| PDBT_TOKEN
		{
		    if ((strcmp($1,"end") == 0)
			     || (strcmp($1,"e") == 0))
			$$ = -1;
		    else
			YYERROR;
#ifdef PDB_PARSE_DEBUG		    
		    printf("move_switch_where \"%s\"\n", $1);
#endif
		}
	;


/********************************************************************
 *
 * next
 */
next_token	: PDBT_NEXT  { $$ = $1; } ;
next_command
	: next_token
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("next command\n");
#endif
		    $$ = pt_alloc_node(PDBT_NEXT);
		}
	;


/********************************************************************
 *
 * nodebug [<module_name>:<proc_name>]*
 */
nodebug_token	: PDBT_NODEBUG  { $$ = $1; } ;
nodebug_command
	: nodebug_token block_spec_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_NODEBUG);
#ifdef PDB_PARSE_DEBUG		    
		    printf("nodebug command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * print <item>
 * print ( <item>,<item>,...)
 *
 * <item> : A integer, double, string, or variable.
 */
print_token	: PDBT_PRINT  { $$ = $1; } ;
print_command
	: print_token '(' print_expr ')'
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PRINT);
#ifdef PDB_PARSE_DEBUG		    
		    printf("print command with print_expr\n");
#endif
		    node->ptr1 = $3;
		    $$ = node;
		}
	| print_token print_item
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PRINT);
#ifdef PDB_PARSE_DEBUG		    
		    printf("print command with print_item\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;

print_expr	
	: print_item
		{   $$ = $1; }
	| print_expr ',' print_expr
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PRINT_EXPR_BRANCH);
#ifdef PDB_PARSE_DEBUG		    
		    printf("expression branch\n");
#endif
		    node->ptr1 = $1;
		    node->ptr2 = $3;
		    $$ = node;
		}
	;

print_item
	: integer
		{   $$ = $1; }
	| double
		{   $$ = $1; }
	| string
		{   $$ = $1; }
	| variable
		{   $$ = $1; }
	;


/********************************************************************
 *
 * procedures [<module_name>:<proc_name>]*
 *
 */
procedures_token	: PDBT_PROCEDURES  { $$ = $1; } ;
procedures_command
	: procedures_token block_spec_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PROCEDURES);
#ifdef PDB_PARSE_DEBUG		    
		    printf("procedures command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * quit
 */
quit_token	: PDBT_QUIT  { $$ = $1; } ;
quit_command
	: quit_token
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("quit command\n");
#endif
		    $$ = pt_alloc_node(PDBT_QUIT);
		}
	;

/********************************************************************
 *
 * show [<queue>] [<process>]
 *
 * <queue>:  This is one of the following:
 *	active
 *	pending
 *	varsusp
 *	globsusp
 *	suspension
 *	all
 *
 * <process>:  This is one of the following (<n> and <m> are integers):
 *	<n>
 *	<n>-<m>
 *	#<n>
 *	^<n>
 *	<module_string>:<proc_string>
 *	all
 */
show_token	: PDBT_SHOW  { $$ = $1; } ;
show_command
	: show_token show_queue show_process
		{
		    pt_node_t *node = pt_alloc_node(PDBT_SHOW);
#ifdef PDB_PARSE_DEBUG		    
		    printf("show command\n");
#endif
		    node->i1 = $2;
		    node->ptr1 = $3;
		    $$ = node;
		}
	| show_token show_process
		{
		    pt_node_t *node = pt_alloc_node(PDBT_SHOW);
#ifdef PDB_PARSE_DEBUG		    
		    printf("show command\n");
#endif
		    node->i1 = PDBT_ALL;
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


show_queue
	: PDBT_TOKEN
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_queue \"%s\"\n", $1);
#endif
		    if (strcmp($1,"all") == 0)
			$$ = PDBT_ALL;
		    else if ((strcmp($1,"active") == 0)
			     || (strcmp($1,"a") == 0)
			     || (strcmp($1,"A") == 0))
			$$ = PDBT_ACTIVE;
		    else if ((strcmp($1,"pending") == 0)
			     || (strcmp($1,"p") == 0)
			     || (strcmp($1,"P") == 0))
			$$ = PDBT_PENDING;
		    else if ((strcmp($1,"suspension") == 0)
			     || (strcmp($1,"s") == 0)
			     || (strcmp($1,"S") == 0))
			$$ = PDBT_SUSPENSION;
		    else if ((strcmp($1,"varsusp") == 0)
			     || (strcmp($1,"vs") == 0)
			     || (strcmp($1,"VS") == 0))
			$$ = PDBT_VARSUSP;
		    else if ((strcmp($1,"globsusp") == 0)
			     || (strcmp($1,"gs") == 0)
			     || (strcmp($1,"GS") == 0))
			$$ = PDBT_GLOBSUSP;
		    else
			YYERROR;
		}
	;

show_process
	: /* empty */
		{
		    pt_node_t *node = pt_alloc_node(PDBT_ALL);
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_process implicit-all\n", node->i1);
#endif
		    $$ = node;
		}
	| integer_value '-' integer_value
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PROCESS_RANGE);
		    node->i1 = $1;
		    node->i2 = $3;
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_process range %d - %d\n", node->i1, node->i2);
#endif
		    $$ = node;
		}
	| integer_value
		{
		    pt_node_t *node = pt_alloc_node(PDBT_PROCESS_RANGE);
		    node->i1 = $1;
		    node->i2 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_process range %d - %d\n",node->i1,node->i2);
#endif
		    $$ = node;
		}
	| show_instance
		{
		    pt_node_t *node = pt_alloc_node(PDBT_INSTANCE);
		    node->i1 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_process instance %d\n", node->i1);
#endif
		    $$ = node;
		}
	| show_reduction
		{
		    pt_node_t *node = pt_alloc_node(PDBT_REDUCTION);
		    node->i1 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_process reduction %d\n", node->i1);
#endif
		    $$ = node;
		}
	| show_undef
		{
		    pt_node_t *node = pt_alloc_node(PDBT_UNDEF);
		    node->i1 = $1;
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_process undef %d\n", node->i1);
#endif
		    $$ = node;
		}
	| block_spec
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("show_process block spec\n");
#endif
		    $$ = $1;
		}
	| PDBT_TOKEN
		{
		    if (   (strcmp($1,"all") == 0)
			|| (strcmp($1,"a"  ) == 0) )
		    {
			pt_node_t *node = pt_alloc_node(PDBT_ALL);
#ifdef PDB_PARSE_DEBUG		    
			printf("show_process all\n");
#endif
			$$ = node;
		    }
		    else
		    {
			YYERROR;
		    }
		}
	;
	
show_instance
	: PDBT_INSTANCE
		{   $$ = $1; }
	;

show_reduction
	: PDBT_REDUCTION
		{   $$ = $1; }
	;

show_undef
	: PDBT_UNDEF
		{   $$ = $1; }
	;


/********************************************************************
 *
 * summary [<queue>] [<process>]
 *
 * This takes the same arguments as the 'show' command.
 */
summary_token	: PDBT_SUMMARY  { $$ = $1; } ;
summary_command
	: summary_token show_queue show_process
		{
		    pt_node_t *node = pt_alloc_node(PDBT_SUMMARY);
#ifdef PDB_PARSE_DEBUG		    
		    printf("summary command\n");
#endif
		    node->i1 = $2;
		    node->ptr1 = $3;
		    $$ = node;
		}
	| summary_token show_process
		{
		    pt_node_t *node = pt_alloc_node(PDBT_SUMMARY);
#ifdef PDB_PARSE_DEBUG		    
		    printf("summary command\n");
#endif
		    node->i1 = PDBT_ALL;
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * source
 */
source_token	: PDBT_SOURCE  { $$ = $1; } ;
source_command
	: source_token string_or_token
		{
		    pt_node_t *node = pt_alloc_node(PDBT_SOURCE);
#ifdef PDB_PARSE_DEBUG		    
		    printf("source command; %s\n", $2);
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * status [<module_name>:<proc_name>]*
 *
 */
status_token	: PDBT_STATUS  { $$ = $1; } ;
status_command
	: status_token block_spec_list
		{
		    pt_node_t *node = pt_alloc_node(PDBT_STATUS);
#ifdef PDB_PARSE_DEBUG		    
		    printf("status command\n");
#endif
		    node->ptr1 = $2;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * switch [<queue>] [<process>] [<where>]
 *
 * <queue>:  This is one of the following:
 *	active
 *	pending
 *
 * <process>:  Same as for the 'show' command.
 *
 * <where>:	Integer
 */
switch_token	: PDBT_SWITCH  { $$ = $1; } ;
switch_command
	: switch_token move_switch_queue move_switch_process move_switch_where
		{
		    pt_node_t *node = pt_alloc_node(PDBT_SWITCH);
#ifdef PDB_PARSE_DEBUG		    
		    printf("switch command\n");
#endif
		    node->i1 = $2;
		    node->i2 = $4;
		    node->ptr1 = $3;
		    $$ = node;
		}
	;


/********************************************************************
 *
 * vars
 */
vars_token	: PDBT_VARS  { $$ = $1; } ;
vars_command
	: vars_token
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("vars command\n");
#endif
		    $$ = pt_alloc_node(PDBT_VARS);
		}
	;




/********************************************************************
 *
 * The remaining rules are some general rules used all over the place.
 */


string_or_token
	: general_token
		{ $$ = $1; }
	| string
		{ $$ = $1; }
	;


/*
 * General block specification rules
 */
block_spec_list
	: /* empty */	{ $$ = NULL_NODE; }
	| block_spec_list block_spec
		{
		    pt_node_t *node = pt_alloc_node(PDBT_BLOCK_SPEC_LIST);
#ifdef PDB_PARSE_DEBUG		    
		    printf("block spec list\n");
#endif
		    node->ptr1 = $1;
		    node->ptr2 = $2;
		    $$ = node;
		}
	;

block_spec
	: block_spec_module ':' block_spec_program
		{
		    pt_node_t *node = pt_alloc_node(PDBT_BLOCK_SPEC);
#ifdef PDB_PARSE_DEBUG		    
		    printf("block spec with module and program\n");
#endif
		    node->ptr1 = $1;
		    node->ptr2 = $3;
		    $$ = node;
		}
	;

block_spec_program
	: '*'
		{   $$ = NULL_NODE; }
	| wildcard
		{   $$ = $1; }
	| wildcard_wrap
		{   $$ = $1; }
	| general_token
		{   $$ = $1; }
	;

block_spec_module
	: '*'
		{   $$ = NULL_NODE; }
	| wildcard
		{   $$ = $1; }
	| general_token
		{   $$ = $1; }
	;


/*
 * This rule assigns its value to be the integer's value.
 */
integer_value
	: PDBT_INTEGER
		{
#ifdef PDB_PARSE_DEBUG		    
		    printf("integer_value %d\n", $1);
#endif
		    $$ = $1;
		}
	;

/*
 * Construct a parse tree node which contains an integer.
 */
integer
	: integer_value
		{
		    pt_node_t *node = pt_alloc_node(PDBT_INTEGER);
#ifdef PDB_PARSE_DEBUG		    
		    printf("integer %d\n", $1);
#endif
		    node->i1 = $1;
		    $$ = node;
		}
	;

/*
 * Construct a parse tree node which contains a double.
 */
double
	: PDBT_DOUBLE
		{
		    pt_node_t *node = pt_alloc_node(PDBT_DOUBLE);
		    double *dtmp = pt_alloc_double($1);
#ifdef PDB_PARSE_DEBUG		    
		    printf("double %f\n", *dtmp);
#endif
		    node->ptr1 = (pt_node_t *) dtmp;
		    $$ = node;
		}
	;

/*
 * Construct a parse tree node which contains a string.
 */
string
	: PDBT_STRING
		{
		    pt_node_t *node = pt_alloc_node(PDBT_STRING);
		    char *stmp = pt_alloc_string($1);
#ifdef PDB_PARSE_DEBUG		    
		    printf("string \"%s\"\n", stmp);
#endif
		    node->ptr1 = (pt_node_t *) stmp;
		    $$ = node;
		}
	;

/*
 * Construct a parse tree node which contains a variable.
 */
variable
	: PDBT_VARIABLE
		{
		    pt_node_t *node = pt_alloc_node(PDBT_VARIABLE);
		    char *stmp;

		    if ((strcmp($1,"print_array_size") == 0) 
			|| (strcmp($1,"pas") == 0))
		    {
			node->i1 = PDBT_VAR_PRINT_ARRAY_SIZE;
			node->ptr2 = (pt_node_t *) &_p_print_array_size;
			stmp = pt_alloc_string("print_array_size");
		    }
		    else if ((strcmp($1,"print_tuple_depth") == 0)
			     || (strcmp($1,"ptd") == 0))
		    {
			node->i1 = PDBT_VAR_PRINT_TUPLE_DEPTH;
			node->ptr2 = (pt_node_t *) &_p_print_tuple_depth;
			stmp = pt_alloc_string("print_tuple_depth");
		    }
		    else if ((strcmp($1,"print_tuple_width") == 0)
			     || (strcmp($1,"ptw") == 0))
		    {
			node->i1 = PDBT_VAR_PRINT_TUPLE_WIDTH;
			node->ptr2 = (pt_node_t *) &_p_print_tuple_width;
			stmp = pt_alloc_string("print_tuple_width");
		    }
		    else if ((strcmp($1,"global_dl") == 0)
			     || (strcmp($1,"gdl") == 0))
		    {
			node->i1 = PDBT_VAR_GLOBAL_DL;
			node->ptr2 = (pt_node_t *) &_p_global_dl;
			stmp = pt_alloc_string("global_dl");
		    }
		    else if ((strcmp($1,"emulator_dl") == 0)
			     || (strcmp($1,"edl") == 0))
		    {
			node->i1 = PDBT_VAR_EMULATOR_DL;
			node->ptr2 = (pt_node_t *) &_p_em_dl;
			stmp = pt_alloc_string("emulator_dl");
		    }
		    else if ((strcmp($1,"gc_dl") == 0)
			     || (strcmp($1,"gcdl") == 0))
		    {
			node->i1 = PDBT_VAR_GC_DL;
			node->ptr2 = (pt_node_t *) &_p_gc_dl;
			stmp = pt_alloc_string("gc_dl");
		    }
		    else if ((strcmp($1,"parallel_dl") == 0)
			     || (strcmp($1,"pdl") == 0))
		    {
			node->i1 = PDBT_VAR_PARALLEL_DL;
			node->ptr2 = (pt_node_t *) &_p_par_dl;
			stmp = pt_alloc_string("parallel_dl");
		    }
		    else if ((strcmp($1,"reduction_break") == 0)
			     || (strcmp($1,"rb") == 0))
		    {
			node->i1 = PDBT_VAR_REDUCTION_BREAK;
			node->ptr2 = (pt_node_t *) &_pdb_reduction_break;
			stmp = pt_alloc_string("reduction_break");
		    }
		    else if ((strcmp($1,"empty_queue_break") == 0)
			     || (strcmp($1,"eqb") == 0))
		    {
			node->i1 = PDBT_VAR_EMPTY_QUEUE_BREAK;
			node->ptr2 = (pt_node_t *) &_pdb_empty_queue_break;
			stmp = pt_alloc_string("empty_queue_break");
		    }
		    else if ((strcmp($1,"print_orphaned") == 0)
			     || (strcmp($1,"po") == 0))
		    {
			node->i1 = PDBT_VAR_PRINT_ORPHANED;
			node->ptr2 = (pt_node_t *) &_pdb_print_orphaned;
			stmp = pt_alloc_string("print_orphaned");
		    }
		    else if ((strcmp($1,"module") == 0)
			     || (strcmp($1,"m") == 0))
		    {
			node->i1 = PDBT_VAR_MODULE;
			stmp = pt_alloc_string("module");
		    }
		    else if ((strcmp($1,"procedure") == 0)
			     || (strcmp($1,"p") == 0)
			     || (strcmp($1,"proc") == 0))
		    {
			node->i1 = PDBT_VAR_PROCEDURE;
			stmp = pt_alloc_string("procedure");
		    }
		    else if ((strcmp($1,"args") == 0)
			     || (strcmp($1,"a") == 0))
		    {
			node->i1 = PDBT_VAR_ARGS;
			stmp = pt_alloc_string("args");
		    }
		    else if ((strcmp($1,"instance") == 0)
			     || (strcmp($1,"i") == 0))
		    {
			node->i1 = PDBT_VAR_INSTANCE;
			stmp = pt_alloc_string("instance");
		    }
		    else if ((strcmp($1,"reduction") == 0)
			     || (strcmp($1,"r") == 0))
		    {
			node->i1 = PDBT_VAR_REDUCTION;
			stmp = pt_alloc_string("reduction");
		    }
		    else if ((strcmp($1,"current_reduction") == 0)
			     || (strcmp($1,"cr") == 0))
		    {
			node->i1 = PDBT_VAR_CURRENT_REDUCTION;
			node->ptr2 = (pt_node_t *) &_p_reduction;
			stmp = pt_alloc_string("current_reduction");
		    }
		    else
		    {
			node->i1 = PDBT_VAR_UNKNOWN;
			stmp = pt_alloc_string($1);
		    }
		    
#ifdef PDB_PARSE_DEBUG		    
		    printf("variable: \"%s\"\n", stmp);
#endif
		    node->ptr1 = (pt_node_t *) stmp;
		    $$ = node;
		}
	;


/*
 * Construct a parse tree node which contains a wildcard.
 */
wildcard
	: PDBT_WILDCARD
		{
		    pt_node_t *node = pt_alloc_node(PDBT_WILDCARD);
		    char *stmp = pt_alloc_string($1);
#ifdef PDB_PARSE_DEBUG		    
		    printf("wildcard \"%s\"\n", stmp);
#endif
		    node->ptr1 = (pt_node_t *) stmp;
		    $$ = node;
		}
	;

/*
 * Construct a parse tree node which contains a wildcard with a .wrap ending
 */
wildcard_wrap
	: PDBT_WILDCARD_WRAP
		{
		    pt_node_t *node = pt_alloc_node(PDBT_WILDCARD_WRAP);
		    char *stmp = pt_alloc_string($1);
#ifdef PDB_PARSE_DEBUG		    
		    printf("wildcard \"%s\"\n", stmp);
#endif
		    node->ptr1 = (pt_node_t *) stmp;
		    $$ = node;
		}
	;

/*
 * Construct a parse tree node which contains an arbitrary token.
 */
general_token
	: PDBT_TOKEN
		{
		    pt_node_t *node = pt_alloc_node(PDBT_TOKEN);
		    char *stmp = pt_alloc_string($1);
#ifdef PDB_PARSE_DEBUG		    
		    printf("token \"%s\"\n", stmp);
#endif
		    node->ptr1 = (pt_node_t *) stmp;
		    $$ = node;
		}
	;


%%

#include "lex.yy.c"

/*
 * void yyerror(s)
 *
 * This routine gets called from YACC when the parse fails.  This routine
 * just stubs it out.  Error handing is done at using the error case
 * in the command rule.
 */
static void yyerror(s)
char *s;
{
}


/*
 * pt_node_t *pt_alloc_node(int token)
 *
 * Allocate a parse tree node, and set its token to the passed value.
 * Add this node to the node list ('next' field of structure) so that
 * this node can be freed even if the parse fails.
 *
 * Return:	A pointer to the newly allocated parse tree node.
 */
static pt_node_t *pt_alloc_node(token)
int token;
{
    pt_node_t *node;
    if ((node = (pt_node_t *) malloc(sizeof(pt_node_t))) == NULL_NODE)
	_p_malloc_error();
    node->token = token;
    node->ptr1 = node->ptr2 = NULL_NODE;
    if (yyp_first_pt_node == NULL_NODE)
    {
	node->next = NULL;
	yyp_first_pt_node = node;
    }
    else
    {
	node->next = yyp_first_pt_node;
	yyp_first_pt_node = node;
    }
    return (node);
}


/*
 * void free_pt_nodes()
 *
 * Free up the allocated parse tree nodes that are on the node list
 * pointed to by yyp_first_pt_node.  This subroutine is called when
 * the parse fail in order to free up all allocated nodes.
 */
static void free_pt_nodes()
{
    pt_node_t *node = yyp_first_pt_node;
    pt_node_t *next_node;
    while (node != NULL)
    {
	if (   node->token == PDBT_DOUBLE
	    || node->token == PDBT_STRING
	    || node->token == PDBT_VARIABLE
	    || node->token == PDBT_WILDCARD
	    || node->token == PDBT_WILDCARD_WRAP
	    || node->token == PDBT_TOKEN    )
	{
	    free(node->ptr1);
	}
	next_node = node->next;
	free(node);
	node = next_node;
    }
}


/*
 * char *pt_alloc_string(char *str)
 *
 * Allocate space for the passed string and fill that space in with that
 * string.
 *
 * Return:	The newly allocate space for the string.
 */
static char *pt_alloc_string(str)
char *str;
{
    char *return_str;
    if ((return_str = (char *) malloc(strlen(str) + 1)) == (char *) NULL)
	_p_malloc_error();
    strcpy(return_str, str);
    return (return_str);
}


/*
 * double *pt_alloc_double(double d)
 *
 * Allocate space for the passed double and fill that space in with that
 * double.  Doubles in the parse tree are malloced because they rarely
 * occur in the parse tree.  Rather than complicate the pt_node_t to
 * accomidate a double, it is easier and just malloc space for it for
 * those few times when it is needed.
 *
 * Return:	The newly allocate space for the double.
 */
static double *pt_alloc_double(d)
double d;
{
    double *return_d;
    if ((return_d = (double *) malloc(sizeof(double))) == (double *) NULL)
	_p_malloc_error();
    *return_d = d;
    return (return_d);
}

