%Copyright (C) The University of Melbourne 1993
%All Rights Reserved

%Permission to use, copy, modify, and distribute this software and
%its documentation for any purpose and without fee is hereby
%granted, provided that the above copyright notice appear in all
%copies and that both that the copyright notice and this
%permission notice and warranty disclaimer appear in supporting
%documentation, and that the name of The University of Melbourne 
%or any of its entities not be used in advertising or publicity
%pertaining to distribution of the software without specific,
%written prior permission.

%THE UNIVERSITY OF MELBOURNE DISCLAIMS ALL WARRANTIES WITH REGARD
%TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
%MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY
%OF MELBOURNE OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
%SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
%WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
%IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
%ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
%THIS SOFTWARE.

%AUTHORS : Jason Lee (jasonl@cs.mu.oz.au)
%	    Andrew Davison (ad@cs.mu.oz.au)


%   Comments :
%	demo finds all solutions to goal using the program database.
%	demo can either be called with a single Goal  i.e g1
%	or with a conjunction of goals  i.e [g1, g2, g3, ...]
%	Goals can include cuts (!), not, and demo itself.
%	If a predicate is defined over more than one database
%	a warning will be issued
%	(NOTE : This means that more than
%	one definition in the database may cause infinite loops, 
%	infinite number of solutions, same solution more than
%	once etc may occur).
%	If a predicate in the database and has the same name and arity
%	as a system predicate the database definition will be used.

% Join databases checking for multiply definitions and also
% create a list of functors and matching arities for later.
demo(P + Q, Goal, Solutions) :-
	!,
	$bb_join_check(P + Q, [], [], Fn, Db, [], Mdef),
	( Mdef \= [] ->
		write(user_error, 'Bebop : warning join of database(s) in demo call contains'),
             	writeln(user_error, ' predicate(s) '),
	     	$bb_write_mdef(Mdef),
             	writeln(user_error, ' defined in more than one database of the join.'),
		nl(user_error),
		flushOutput(user_error)
	),
	$bb_demo(Fn, Db, Goal, Solutions).

demo(P, Goal, Solutions) :-
	$bb_exfun(P, [], Fn),
	$bb_demo(Fn, P, Goal, Solutions).

% This is the working interpreter.
% Goal is either single or a list
$bb_demo(Fn, ProgDb, Goal, Solutions) :-
	(isCons(Goal) ->
		findall(Goal, $bb_do_goals(Fn, ProgDb, Goal), Solutions)
	 ;
		findall(Goal, $bb_do_goal(Fn, ProgDb, Goal), Solutions)
	).


% No more goals to solve!!
$bb_do_goals(_, _, []).

% Solve each goal
$bb_do_goals(Fn, ProgDb, [Goal | Rest]) :-
	$bb_do_goal(Fn, ProgDb, Goal),
	$bb_do_goals(Fn, ProgDb, Rest).

% Is goal a system predicate and not defined in
% the database  call it, also if we have a not
% as the functor leave it.
$bb_do_goal(FnList, _, Goal) :-
	functor(Goal, Fn, Ar),
	Fn \= (not),
	systemPredicate(Fn, Ar),
	\+ member((Fn/Ar), FnList),
	!,
	call(Goal).

% Otherwise look up program for clause and solve
% But first check if we have a not goal and make it's
% arguments a list of goals to prove.
% Eg not(not(a), not(b)) == a ; b so do not(a), not(b)
% then not of them.
$bb_do_goal(Fn, ProgDb, Goal) :-
	(nonvar(Goal) ->
		Goal =.. [N | Args]
	 ;
		N = no
	),
	(N = (not) ->
		Body = [not(Args)]
	;
		member(Clause, ProgDb),
		duplicate(Clause, NewCl),
		$bb_parts(NewCl, Head, Body),
		Head = Goal
	),
	$bb_do_body(Fn, ProgDb, Body, AfterCut, HadCut),
	( HadCut = yes,
		!,
		$bb_do_body(Fn, ProgDb, AfterCut)
	; HadCut = no
	).

% Just get Head and Body of clause
$bb_parts((H :- B), H, B) :- !.

$bb_parts(H, H, []).


% Do body checking for Cuts, not or call to demo
$bb_do_body(Fn, ProgDb, Goal) :-
	$bb_do_body(Fn, ProgDb, Goal, AfterCut, HadCut),
	( HadCut = yes,
                !,
                $bb_do_body(Fn, ProgDb, AfterCut)
        ; HadCut = no
        ).

$bb_do_body(_, _, [], [], no) :- !.

$bb_do_body(_, _, [! | AfterCut], AfterCut, yes) :- !.

$bb_do_body(Fn, ProgDb, [demo(P, G, S) | Rest], AfterCut, HadCut) :-
	!,
	demo(P, G, S),
	$bb_do_body(Fn, ProgDb, Rest, AfterCut, HadCut).

$bb_do_body(Fn, ProgDb, [not(Goal) | Rest], AfterCut, HadCut) :-
	!,
	% Non-Logical used here for now seems to work better
	% than logical "not".
	% Note: Goal needs to be ground before being called
	% i.e if  f(A) :- not A > 2, .... is a pred in db
	% and f(A), A = 1 is a goal it will fail, BUT
	% A = 1, f(A) as a goal will succeed!!!
	% Other examples as shown by Andrew i.e married etc
	% Also Goal may a list conjunction or a single goal
	% check and either call $bb_do_goals or $bb_do_goal.
	( isCons(Goal) ->
		\+ $bb_do_goals(Fn, ProgDb, Goal)
	;
		\+$bb_do_goal(Fn, ProgDb, Goal)
	),
	$bb_do_body(Fn, ProgDb, Rest, AfterCut, HadCut).

$bb_do_body(Fn, ProgDb, [Goal | Rest], AfterCut, HadCut) :-
	$bb_do_goal(Fn, ProgDb, Goal),
	$bb_do_body(Fn, ProgDb, Rest, AfterCut, HadCut).


% write out the list of predicate names and arities that are
% multiply defined.
$bb_write_mdef([F/A]) :-
	!,
	write(user_error, F),
	write(user_error, '/'),
	write(user_error, A).

$bb_write_mdef([(F/A) | Rest]) :-
	 write(user_error, F),
	 write(user_error, '/'),
	 write(user_error, A),
	 write(user_error, ', '),
	 $bb_write_mdef(Rest).

% Extract functors and arities from database.
$bb_exfun([], Fn, Fn).

$bb_exfun([G | Rest], FnS, FnE) :-
	$bb_parts(G, H, _),
	functor(H, NFn, NAr),
	( member((NFn/NAr), FnS) ->
		NewFn = FnS
	;
		append(FnS, [(NFn/NAr)], NewFn)
	),
	$bb_exfun(Rest, NewFn, FnE).

% Join databases and functor/arity list looking for multiply
% defined predicates, collect them in Mdef.
$bb_check_app([], _, Fn, Fn, []) :- !.

$bb_check_app(Fn, [], [], Fn, []) :- !.

$bb_check_app([(Fn/Ar) | Rest], FnC, FnS, FnE, Mdef) :-
	( member((Fn/Ar), FnC) ->
		Mdef = [(Fn/Ar) | NMdef], NewFn = FnS
	;
		append([(Fn/Ar)], FnS, NewFn), NMdef = Mdef
	),
	$bb_check_app(Rest, FnC, NewFn, FnE, NMdef).
	

% Carry out the joining of databases
$bb_join_check(P + Q, FnS, DbS, FnE, DbE, OMdef, Mdef) :-
	!,
	append(Q, DbS, NewDb),
	$bb_exfun(Q, [], QFn),
	$bb_check_app(QFn, FnS, FnS, NewFn, NMdef),
	append(OMdef, NMdef, NNMdef),
	$bb_join_check(P, NewFn, NewDb, FnE, DbE, NNMdef, Mdef).

$bb_join_check(P, FnS, DbS, FnE, DbE, OMdef, Mdef) :-
	append(P, DbS, DbE),
	$bb_exfun(P, [], PFn),
	$bb_check_app(PFn, FnS, FnS, FnE, NMdef),
	append(OMdef, NMdef, NNMdef),
	sort(NNMdef, Mdef).


% end of demo


