/* $Id: hash.h,v 1.7 1993/10/20 15:09:13 boortz Exp $ */

/* ######################################################################## */
/* 			GENERAL HASH TABLE HANDLING 			    */
/* ######################################################################## */

/* Macros for general hashing by Kent Boortz. The actual hashing code was
 * written by Bjorn Danielsson.
 * You can choose hash function, but not rehash function. You put in
 * routines for comparing keys, allocating memory etc.
 * All routines are (strange) macros purely for efficiency.
 *
 * The table size is always a power of 2 to make modulus calculation cheap.
 * When a table becomes 75% full, it is automatically expanded.
 * The rehashing method used is linear rehashing, with an odd distance that
 * uses some more bits from the key.
 *
 * A deleted entry is only marked as deleted. It may be reused but in the
 * worst case the table could be owerflowed even if there are a lot of
 * deleted entrys.
 */

/* ########################################################################
 *
 *			USING THIS MACRO PACKAGE
 *
 * You define some routines, preferably as macros. These will be part
 * of the arguments to the routines in this package. The first is for
 * allocating memory in bytes. This routine should use the variables 
 *
 *	unsigned size;		/* Variables already defined *\
 *	char *memory;
 *
 * and could in a simple example be
 *
 *	#define MyHashAlloc {memory = malloc(size)}
 *
 * The second one is used for comparing your supplied key
 * with the key in the table. You shoud use the unsigned long variable 
 * 'key' for comparing and store the result in 'result'. A simple one is
 *
 *	unsigned long key;	/* Variables already defined *\
 *	int result;
 *
 *	#define MyKeyCompare {result = (mykey == key);}
 *
 * You also need to supply a routine for calculating the hash value
 *
 *	unsigned long hash;	/* Variable already defined *\
 *
 *	#define MyHashFunc {hash = (unsigned long)mykey;}
 *
 * If clearing, copying and freeing the table you have to use
 * additional routines
 *
 *	HashEntry *entry;	/* Variable already defined *\
 *
 *	#define MyClearEntry {}
 *
 *	HashEntry *entry;	/* Variables already defined *\
 *	HashEntry *newentry;
 *
 *	#define MyCopyEntry {*newentry = *entry;}
 *
 *	unsigned size;		/* Variables already defined *\
 *	char *memory;
 *
 *	#define MyFree {free(memory);}
 *
 * ######################################################################## */

#define VACANT_HASH_ENTRY	0 /* NOTE: YOU CAN'T USE 0 AS A KEY */
#define DELETED_HASH_ENTRY	1 /* NOTE: YOU CAN'T USE 1 AS A KEY */

/* ########################################################################
 *
 * NewHashTable(
 *		HashTable *Table,
 *		long Size,
 *		ProcAllocate)
 *
 * Create new hash table. The arguments are:
 *
 *	Table		Return value.
 *	Size		Max number of elements in table.
 *	ProcAllocate	Procedure to allocate memory fo table.
 *			You allocate 'size' bytes and set
 *			'memory' to this memory block. If malloc()
 *			was to be used ProcAllocate could be defined:
 *
 *				{memory = malloc(size);}
 *
 *			Remeber to set 'memory' to NULL if the memory
 *			couldn't be allocated.
 *
 * 'HashTable' is defined as:
 *
 *     typedef struct {
 *       unsigned long size;
 *       long limit;
 *       HashEntry entry[ANY];
 *     } HashTable;
 *
 * ------------------------------------------------------------------------ */

#define NewHashTable(Table,Size,ProcAllocate) \
{ \
  unsigned long __real_size = 0x4; \
\
  if (Size & ~((unsigned long)0xff)) /* Always start at 256 or less */ \
    __real_size = 0x100; \
\
  while (__real_size < Size)	/* Find power of 2 that fit */ \
    __real_size = (__real_size << 1); \
\
  AllocateHashTable(Table,__real_size,ProcAllocate) \
  if (Table != NULL) \
    ClearHashTable(Table,{}) \
}

/* ########################################################################
 *
 * AllocateHashTable(				(INTERNAL: DO NOT USE)
 *		HashTable *Table,
 *		long Size,
 *		ProcAllocate)
 *
 * Used by 'NewHashTable' and 'CopyHashTable'. Assumes
 * that the size is a power of 2
 * ------------------------------------------------------------------------ */

#define AllocateHashTable(Table,Size,ProcAllocate) \
{ \
  {\
    char *memory; \
    unsigned size = SizeOfHashTable(Size); \
    ProcAllocate; \
    (Table) = (HashTable *)memory; \
  } \
\
  if ((Table) != NULL) \
    (Table)->size = Size; \
}

/* ########################################################################
 *
 * ClearHashTable(
 *		HashTable *Table,
 *		ProcClearEntry)
 *
 * Clear the hash table. The arguments are:
 *
 *	Table		Table to clear
 *	ProcClearEntry	What to do with entry 'entry' in
 *			table. 'HashEntry' is defined as:
 *
 *				typedef struct {
 *				  unsigned long key;
 *				  unsigned long value;
 *				} HashEntry;
 *
 *			The entry key position will be cleared
 *			after this call.
 * ------------------------------------------------------------------------ */


#define ClearHashTable(Table,ProcClearEntry) \
{ \
  long n = (Table)->size; \
\
  (Table)->limit = (n >> 1) + (n >> 2); /* Set limit at 75% full */ \
\
  do { \
    HashEntry *entry; \
    n--; \
    entry = &(Table)->entry[n]; \
    if (entry->key != VACANT_HASH_ENTRY) /* Not empty */ \
      { \
	ProcClearEntry; \
      } \
    /* Always clear key to enable the whole statement above to go away */ \
    entry->key = VACANT_HASH_ENTRY; \
  } while (n != 0); \
}

/* ########################################################################
 *
 * CopyHashTable(
 *		HashTable NewTable,
 *		HashTable *Table,
 *		ProcAllocate,
 *		ProcCopyEntry)
 *
 * Allocate a new table and copy the elements in the hash table.
 * The arguments are:
 *
 *	NewTable	The new copy of the table.
 *	Table		Table to copy.
 *	ProcAllocate	Procedure to allocate memory fo table.
 *			See 'NewHashTable' above.
 *	ProcCopyEntry	Copy the old entry, 'entry' to the new
 *			entry 'newentry'.
 *			To do a simple element copy you write:
 *
 *				{*newentry = *entry;}
 *
 *			If you don't copy the entry will be cleared.
 * ------------------------------------------------------------------------ */

#define CopyHashTable(NewTable,Table,ProcAllocate,ProcCopyEntry) \
{ \
  int __size = (Table)->size; \
\
  AllocateHashTable((NewTable),__size,ProcAllocate);\
\
  if ((NewTable) != NULL) \
    { \
      int i; \
      for (i = 0; i < __size; i++) \
	{ \
	  HashEntry *entry = &(Table)->entry[i]; \
	  HashEntry *newentry = &(NewTable)->entry[i]; \
	  newentry->key =  VACANT_HASH_ENTRY;  /* Always clear */ \
\
	  if (entry->key != VACANT_HASH_ENTRY) /* Not empty */ \
	    { \
	      ProcCopyEntry; \
	    } \
	} \
    } \
}

/* ########################################################################
 *
 * GrowHashTable(
 *		HashTable NewTable,
 *		HashTable *Table,
 *		ProcHashFunc,
 *		ProcKeyComp,
 *		ProcAllocate,
 *		ProcFree,
 *		ProcCopyEntry)
 *
 * Allocate a new table twice as big as the old one and hash the
 * elements into the new hash table. The arguments are:
 *
 *	NewTable	The new copy of the table.
 *	Table		Table to copy.
 *	ProcHashFunc	Calculate the hash value into 'hash'
 *	ProcKeyComp	See 'HashLookupLoop' below.
 *	ProcAllocate	See 'NewHashTable' above.
 *	ProcFree	Free the memory pointed to by 'memory'.
 *			The block is 'size' bytes.
 * ------------------------------------------------------------------------ */

#define GrowHashTable(NewTable,Table,ProcHashFunc,ProcKeyComp,ProcAlloc,ProcFree) \
{ \
  AllocateHashTable(NewTable,2 * Table->size, ProcAlloc); \
\
  if (NewTable != NULL) \
    { \
      long __newlim = NewTable->limit;	/* 75% of new table */ \
      long __i = Table->size; \
      ClearHashTable(NewTable,{}) \
\
      do { \
        HashEntry *__entry = &Table->entry[0]; \
        unsigned long key = __entry->key; \
        if (key != VACANT_HASH_ENTRY) \
          { \
	    unsigned long hash; \
	    HashEntry *__newentry; \
	    ProcHashFunc; \
	    HashLookupLoop(NewTable,key,hash,__newentry,ProcKeyComp,{},{}); \
	    *__newentry = *__entry; \
	    __newlim -= 1;		/* one less before another grow */ \
          } \
        __entry += 1;			/* next entry */ \
        __i -= 1;			/* one less to rehash */ \
      } while (__i != 0); \
\
      NewTable->limit = __newlim; \
      { \
	char *memory = (char *)Table; \
	unsigned size = SizeOfHashTable((Table)->size); \
        ProcFree; \
      } \
    } \
}

/* ########################################################################
 *
 * HashLookupLoop(					(LOW LEVEL)
 *		HashTable *Table,
 *		unsigned long Key,
 *		unsigned long Hval,
 *		HashEntry *Entry,
 *		ProcKeyComp,
 *		IfFound,
 *		IfNotFound)
 *
 * Lookup entry in hash table. This is the low level lookup routine that
 * only leave you a pointer to an entry. Read about the arguments below
 * carefully, missuse may damage your hash table. The arguments are:
 *
 *	Table		Table to copy.
 *	Key		Hash key. Note that it can't have the value
 *			of the constant VACANT_HASH_ENTRY, default 0.
 *	Hval		Hash value for this key.
 *	Value		Value for this key.
 *	Entry		Returned pointer to key-value pair,
 *			 see 'CopyHashTable'.
 *	ProcKeyComp	Should compare a variable 'key' with
 *			the supplied 'Key' and set the a variable
 *			'result' to 0 if not equal and greater
 *			than 0 if equal.
 *	IfFound		Executed if entry found.
 *			A pointer to the entry is returned.
 *			You may change the value.
 *	IfNotFound	Executed if entry was not found.
 *			A pointer to a new entry is returned.
 *			You may assign Entry->key to Key and Entry->value
 *			to the value. If the old Key was not 
 *			DELETED_HASH_ENTRY you have to Decrement 
 *			Table->limit. If the new limit is zero you have 
 *			to call 'GrowHashTable'.
 * ------------------------------------------------------------------------ */

#define NO_EMPTY -1

#define HashLookupLoop(Table, Key, Hval, Entry, ProcKeyComp, IfFound, IfNotFound) \
{\
  unsigned long __mask = HashMask(Table); \
  long __index = ((Hval) & __mask); \
  long __last_empty = NO_EMPTY; \
\
  do { \
    register unsigned long key; \
    int result; \
    register unsigned long __c; \
\
    (Entry) = &(Table)->entry[__index]; \
    key = (Entry)->key; \
\
    if (key == DELETED_HASH_ENTRY) \
      __last_empty = __index; \
    else \
      if (key == VACANT_HASH_ENTRY) \
	{ \
	  if (__last_empty != NO_EMPTY) \
	    (Entry) = &(Table)->entry[__last_empty]; \
\
          IfNotFound; \
	  break; \
	} \
\
    { \
      ProcKeyComp; \
    } \
\
    if (result) \
      { \
        IfFound; \
	break; \
      } \
\
    __c = (((Hval) >> 3) | 0x1); \
    __index = ((__index + __c) & __mask); \
  } while (1); \
}

/* ########################################################################
 *
 * HashFind(
 *		HashTable *Table,
 *		unsigned long Key,
 *		unsigned long Value,
 *		ProcKeyComp,
 *		IfNotFound)
 *
 * Lookup entry in hash table. If not found, don't insert.
 * The arguments are:
 *
 *	Table		Table to copy.
 *	Key		Hash key. See note at 'HashLookupLoop'.
 *	Value		Value for this key.
 *	ProcHashFunc	Calculate hash value from Key and store in 'hash'
 *	ProcKeyComp	Should compare a variable 'key' with
 *			the supplied 'Key' and set the a variable
 *			'result' to 0 or 1.
 *	IfNotFound	Executed if entry wasn't found.
 *			'Value' will contain garbage.
 * ------------------------------------------------------------------------ */

#define HashFind(Table, Key, Val, ProcHashFunc, ProcKeyComp, IfNotFound) \
{ \
  HashEntry * __entry; \
  unsigned long hash; \
  { \
    ProcHashFunc; \
  } \
  HashLookupLoop(Table, Key, hash, __entry, ProcKeyComp, {}, IfNotFound); \
  (Val) = __entry->value; \
}

/* ########################################################################
 *
 * HashEnter(
 *		HashTable *Table,
 *		unsigned long Key,
 *		ProcHashFunc,
 *		ProcKeyComp,
 *		ProcAllocate,
 *		ProcFree)
 *
 * Will try to find the Key in Table. If found it will overwrite
 * overwrite the value with the new value. If not found it
 * will insert the new (Key,Value) pair into the table.
 * If the table become more than 75% full it will automaticly
 * expand.
 *
 * The arguments are:
 *
 *	Table		Table to copy.
 *	Key		Hash key. See note at 'HashLookupLoop'.
 *	Value		Value for this key.
 *	ProcHashFunc	Calculate hash value from Key and store in 'hash'
 *	ProcKeyComp	Should compare a variable 'key' with
 *			the supplied 'Key' and set the a variable
 *			'result' to 0 or 1.
 *	ProcAllocate	See 'NewHashTable' above.
 *	ProcFree	Free the memory pointed to by 'memory'.
 *			The block is 'size' bytes.
 * ------------------------------------------------------------------------ */

#define HashEnter(Table,Key,Val,ProcHashFunc,ProcKeyComp,ProcAllocate,ProcFree) \
{ \
  HashEntry * __entry; \
  unsigned long hash; \
  { \
    ProcHashFunc; \
  } \
  HashLookupLoop((Table), (Key), hash, __entry, ProcKeyComp, \
    { \
      __entry->value = (unsigned long)(Val); /* Overwrite old value */ \
    }, \
    { \
      __entry->key = (unsigned long)(Key); /* If not found, enter key */ \
      __entry->value = (unsigned long)(Val); /* Overwrite old value */ \
\
      if (--((Table)->limit) == 0) \
	{ \
	  HashTable *new; \
	  GrowHashTable(new,(Table),ProcHashFunc,ProcKeyComp,ProcAllocate,ProcFree); \
	  (Table) = new; \
	} \
    }) \
}


/* ########################################################################
 *
 * HashEntryLoop(
 *		HashTable *Table,
 *		ProcEntry)
 *
 * Will iterate over all entrys, i.e. (Key,Value) pairs, currently 
 * in the table.
 *
 * The arguments are:
 *
 *	Table		Table to copy.
 *	ProcEntry	Do whatever you like with 'entry'.
 * ------------------------------------------------------------------------ */

#define HashEntryLoop(Table,ProcEntry) \
{ \
  long __n = (Table)->size; \
  do { \
    HashEntry *entry = &(Table)->entry[--__n]; \
\
    if ((entry->key != VACANT_HASH_ENTRY) \
	  && (entry->key != DELETED_HASH_ENTRY)) \
      { \
	ProcEntry; \
      } \
  } while (__n != 0); \
}

/* ######################################################################## */

/* Information about the hash table */

#define SizeOfHashTable(n) \
	(sizeof(HashTable) + ((n)-ANY)*sizeof(HashEntry))

#define HashMask(Table) ((Table)->size - 1)


#ifndef ANY
 #define ANY 1
#endif


/* ######################################################################## */

/* Each entry is a tuple (Key,Value) */

typedef struct {
  unsigned long key;
  unsigned long value;
} HashEntry;


/* Data unique to each table */

typedef struct {
  unsigned long size;
  long limit;
  HashEntry entry[ANY];
} HashTable;
