/* $Id: array.c,v 1.20 1993/10/14 18:45:27 bd Exp $ */

#include "include.h"
#include "term.h"
#include "tree.h"
#include "predicate.h"
#include "exstate.h"
#include "engine.h"
#include "storage.h"
#include "unify.h"
#include "copy.h"
#include "gc.h"
#include "initial.h"
#include "config.h"
#include "names.h"
#include "display.h"
#include "error.h"
#include "array.h"

static bool method_typeof();
static bool method_sizeof();
static bool method_get_array();
static bool method_set_array();
static bool method_swap_array();
static bool method_sub_array();

static Gvainfo	newarray();
static bool 	unifyarray();
static int 	printarray();
static Gvainfo	copyarray();
static Gvainfo	gcarray();
static int	uncopiedarray();
static envid	*deallocatearray();
static int	killarray();
static bool	sendarray();

typedef struct array {
  gvainfo	header;
  Term		data;		/* Pointer to structure with content */
  Term		original;	/* Port to original array object or NullTerm */
  int		size;		/* Number of cells in array */
  int		offset;		/* Start of array */
  int		start;		/* subtract index */
  int		scale;		/* multiply index */
} Array;

/* This fileds gives us some more information. We call the memory
 * content in an array for 'cells' and the cells that belongs to
 * an array for 'elements'. Number of 'elements' <= number of 'cells'.
 *
 *   offset			First element in array
 *   index - start		Zero based index (zindex)
 *   zindex * scale + offset	Indexed cell
 *   size * scale + offset	Points out cell after last cell
 *   
 *   newoff = oldoff + oldscale * offset
 *   
 *   newscale = oldscale * scale
 *   
 *   
 */   
static gvamethod arraymethods = {
  newarray,
  unifyarray,
  printarray,
  copyarray,
  gcarray,
  uncopiedarray,  
  deallocatearray,
  killarray,
  NULL,				/* Close only called if in close list */
  sendarray,
  NULL,
  NULL
};


static Atom atom_nil;
static Term term_array;
static Functor functor_typeof;
static Functor functor_sizeof;
static Functor functor_subarray_3;
static Functor functor_subarray_4;
static Functor functor_subarray_5;
static Functor functor_get_2;
static Functor functor_set_2;
static Functor functor_set_3;

#define IsArray(Prt) (IsCvaTerm(Prt) && \
		      RefCvaMethod(Ref(Prt)) == &arraymethods)

#define GvaArr(g)  ((Array*)(g))
#define Arr(x)     GvaArr(RefGva(Ref(x)))


/* ------------------------------------------------------------------------ */
/* LOW LEVEL OBJECT METHODS */

static Gvainfo newarray(old)
     Array *old;
{
  Array *new;
  NEW(new,Array);		/* Called before copy and gc */

  new->size   = old->size;	/* Copy all but terms */
  new->offset = old->offset;
  new->start    = old->start;
  new->scale    = old->scale;

  return (Gvainfo) new;
}


/* If we are the original object we copy the actual data */
/* If we are a subarray we copy the original. If the original */
/* is already copied we get a pointer to the new object anyway */

static Gvainfo copyarray(old,new)
     Array *old, *new;
{
  new->original   = old->original;
  new->data   = old->data;

  if (!Eq(new->original, NullTerm))
    copy_location(&new->original);
    
  copy_location(&new->data);

  return (Gvainfo) new;
}


static Gvainfo gcarray(old,new,gcst)
     Array *old, *new;
     gcstatep gcst;
{
  new->original   = old->original;
  new->data   = old->data;

  if (!Eq(new->original, NullTerm))
    gc_location(&new->original, gcst);
    
  gc_location(&new->data, gcst);
    
  return (Gvainfo) new;
}


static envid *deallocatearray(prt)
     Array *prt;
{
  /* dispose the storage */
  return NULL;
}


static int uncopiedarray(prt)
     Array *prt;
{
  return 0;
}
    

static bool unifyarray(x0,x1,andb,exs)
     Term x0, x1;
     andbox *andb;
     exstate *exs;
{
  if (Eq(x0, x1))		/* Not as simple as I thought */
    return TRUE;		/* so just test for equality */
  return FALSE;
}


static int killarray(prt)
     Array *prt;
{
  return 1;
}


static int printarray(file,p,tsiz)
     FILE *file;
     Array *p;
     int tsiz;
{
  int i;
  Term *x = StrArgRef(Str(p->data),p->offset);

  if (p->scale < 0)
    {
      fprintf(file,"{array[%d..%d]: ",p->size/p->scale+p->start,p->start-1);

      display_term(x[p->size + p->scale], tsiz-1);

      for (i = 2; i != tsiz-1 ; i++)
	{
	  int j = -(p->scale) * i;

	  if (j > p->size)
	    break;

	  fprintf(file,",");
	  display_term(x[p->size - j]);
	}
    }
  else
    {
      fprintf(file,"{array[%d..%d]: ",p->start,p->size/p->scale+p->start-1);

      display_term(x[0], tsiz-1);

      for (i = 1; i != tsiz-1 ; i++)
	{
	  int j = p->scale * i;

	  if (j >= p->size)
	    break;

	  fprintf(file,",");
	  display_term(x[j]);
	}
    }


  if (i == tsiz-1)
    fprintf(file,",...}");
  else
    fprintf(file,"}");

  return 1;
}


/* ------------------------------------------------------------------------ */
/* LOW LEVEL OBJECT SEND METHOD */

static bool sendarray(message, self, exs)
     Term message, self;
     exstate *exs;
{
  Deref(message, message);

  if(IsSTR(message)) {
    Functor op = StrFunctor(Str(message));
    Term X0, X1, X2, X3, X4;
    
    if (op == functor_get_2)
      {
	GetStrArg(X0,Str(message),0);
	GetStrArg(X1,Str(message),1);
	return method_get_array(X0,X1,self,exs);
      }
    else
      if(op == functor_set_2)
	{
	  GetStrArg(X0,Str(message),0);
	  GetStrArg(X1,Str(message),1);
	  return method_set_array(X0,X1,self,exs);
	}
      else
	if(op == functor_set_3)
	  {
	    GetStrArg(X0,Str(message),0);
	    GetStrArg(X1,Str(message),1);
	    GetStrArg(X2,Str(message),2);
	    return method_swap_array(X0,X1,X2,self,exs);
	  }
	else
	  if(op == functor_subarray_3)
	    {
	      GetStrArg(X0,Str(message),0);
	      GetStrArg(X1,Str(message),1);
	      GetStrArg(X2,Str(message),2);
	      return method_sub_array(X0,X1,NULL,NULL,self,X2,exs);
	    }
	  else
	    if(op == functor_subarray_4)
	      {
		GetStrArg(X0,Str(message),0);
		GetStrArg(X1,Str(message),1);
		GetStrArg(X2,Str(message),2);
		GetStrArg(X3,Str(message),3);
		return method_sub_array(X0,X1,X2,NULL,self,X3,exs);
	      }
	    else
	      if(op == functor_subarray_5)
		{
		  GetStrArg(X0,Str(message),0);
		  GetStrArg(X1,Str(message),1);
		  GetStrArg(X2,Str(message),2);
		  GetStrArg(X3,Str(message),3);
		  GetStrArg(X4,Str(message),4);
		  return method_sub_array(X0,X1,X2,X3,self,X4,exs);
		}
	      else
		if(op == functor_typeof)
		  {
		    GetStrArg(X0,Str(message),0);
		    return method_typeof(X0,exs);
		  }
		else
		  if(op == functor_sizeof)
		    {
		      GetStrArg(X0,Str(message),0);
		      return method_sizeof(X0,self,exs);
		    }
		  else
		    return FALSE;
    
  }
  
  IfVarSuspend(message);
  USAGE_FAULT("Wrong message to send");
}

/* ------------------------------------------------------------------------ */

/* array(+Array) */

bool akl_array(Arg)
     Argdecl;
{
  Term X0;
  Deref(X0,A(0));
  
  if(IsArray(X0))
    return TRUE;
  IfVarSuspend(X0);
  return FALSE;
}
  
/* ------------------------------------------------------------------------ */

static bool akl_new_array(Size,Origin,Scale,New,exs)
     Term Size, Origin, Scale, New;
     exstate *exs;
{
  int size, i;
  int start = 0;
  int scale = 1;
  Array *new;
  Term tmpv;

  Deref(Size,Size);		/* The size of the new array */
  IfVarSuspend(Size);

  if (!IsINT(Size))
    USAGE_FAULT("new_array/[234]: Size is not a number");

  if ((size = IntVal(Int(Size))) < 1)
    USAGE_FAULT("new_array/[234]: Size is less than minimum 1");

  if (!Eq(Origin,NullTerm))
    {
      Deref(Origin,Origin);		/* The start index of the new array */
      IfVarSuspend(Origin);

      if (!IsINT(Origin))
	USAGE_FAULT("new_array/[234]: Start of index is not a number");

      start = IntVal(Int(Origin));
    }

  if (!Eq(Scale,NullTerm))
    {
      Deref(Scale,Scale);		/* The start index of the new array */
      IfVarSuspend(Scale);

      if (!IsINT(Scale))
	USAGE_FAULT("new_array/[234]: Scale of index is not a number");

      scale = IntVal(Int(Scale));
    }

  MakeGvainfo(new,Array,&arraymethods,exs->andb);
  MakeStructTerm(new->data,store_functor(atom_nil, size),exs->andb);

  /* CHECK OVERFLOW */

  new->original = NullTerm;	/* Not a subarray */
  new->size = size;
  new->offset = 0;		/* original has the whole array */
  new->start = start;
  new->scale = scale;

  {
    Term *x = StrArgRef(Str(new->data),0);
    for (i = 0; (i < size); i++)
      *(x++) = term_nil;
  }

/*  add_gvainfo_to_close((Gvainfo)new,exs);   ONLY IF MALLOC THE DATA */

  Deref(New,New);
  MakeCvaTerm(tmpv, (Gvainfo)new);
  return unify(New,tmpv,exs->andb,exs);
}

/* ------------------------------------------------------------------------ */

/* new_array(+Size, -Array) */

bool akl_new_array_2(Arg)
     Argdecl;
{
  return akl_new_array(A(0),NullTerm,NullTerm,A(1),exs);
}


/* ------------------------------------------------------------------------ */

/* new_array(+Size, +Start, -Array) */

bool akl_new_array_3(Arg)
     Argdecl;
{
  return akl_new_array(A(0),A(1),NullTerm,A(2),exs);
}


/* ------------------------------------------------------------------------ */

/* new_array(+Size, +Start, +Scale, -Array) */

bool akl_new_array_4(Arg)
     Argdecl;
{
  return akl_new_array(A(0),A(1),A(2),A(3),exs);
}


/* ------------------------------------------------------------------------ */

/* subarray(+Size, +Offset, -New)@ +Old */
/* subarray(+Size, +Offset, +Start, -New)@ +Old */
/* subarray(+Size, +Offset, +Start, +Scale, -New)@ +Old */

static bool method_sub_array(Size, Offset, Origin, Scale, Old, New, exs)
     Term Size, Offset, Origin, Scale, Old, New;
     exstate *exs;
{
  Array *old, *new;
  int size, offset, start, scale;
  Term tmpv;

  /* Old is this object and already dereferenced */
  /* and known to be a CVA variable */

  if(!IsLocalGVA(Ref(Old),exs->andb))
    {
      /* The array is not local to the current andbox
       * A constraint is constructed without a suspension.
       * The constraint will be retried when promoted.
       */
      return SUSPEND;
    }

  old = Arr(Old);

  Deref(Size, Size);		/* The size of the sub array */
  IfVarSuspend(Size);

  Deref(Offset, Offset);	/* The offset of the sub array */
  IfVarSuspend(Offset);

  if (!Eq(Origin,NullTerm))
    {
      Deref(Origin, Origin);	/* The first index */
      IfVarSuspend(Origin);

      if (!IsINT(Origin))
	USAGE_FAULT("subarray/[345]: Start of index is not a number");

      start = IntVal(Int(Origin));
    }
  else
    start = old->start;

  if (!Eq(Scale,NullTerm))
    {
      Deref(Scale, Scale);	/* The first index */
      IfVarSuspend(Scale);

      if (!IsINT(Scale))
	USAGE_FAULT("subarray/[345]: Scale of index is not a number");

      scale = IntVal(Int(Scale)) * old->scale;
    }
  else
    scale = old->scale;

  if (!IsINT(Size))
    USAGE_FAULT("subarray/[345]: Size is not a number");

  if (!IsINT(Offset))
    USAGE_FAULT("subarray/[345]: Offset is not a number");

  if (((size = IntVal(Int(Size))) < 1))
    USAGE_FAULT("subarray/[345]: Size is less than minimum 1");

  offset = IntVal(Int(Offset));
  offset -= old->start;		/* Adjust to zero based index */
  offset = old->offset + old->scale * offset;

  if (scale < 0)
    size *= -scale;		/* Let size be cells */
  else
    size *= scale;

  if ((offset < old->offset) || ((offset + size) > (old->offset + old->size)))
    USAGE_FAULT("subarray/[345]: New array out of bounds");

  MakeGvainfo(new,Array,&arraymethods,exs->andb);
  new->size = size;
  new->offset = offset;
  new->start = start;
  new->scale = scale;
  new->data = old->data;
  new->original = Old;

/*  add_gvainfo_to_close((Gvainfo)new,exs);   ONLY IF MALLOC THE DATA */

  Deref(New, New);		/* The term unified with the array */
  MakeCvaTerm(tmpv, (Gvainfo)new);
  return unify(New,tmpv,exs->andb,exs);
}

/* ------------------------------------------------------------------------ */

/* get(+Index, -Term)@ +Array */

static bool method_get_array(Off, ArgOld, ArrIn, exs)
     Term Off, ArgOld, ArrIn;
     exstate *exs;
{
  Term ArgTmp;
  Array *p;
  int i;

  /* ArrIn is self and already dereferenced and checked */

  if(!IsLocalGVA(Ref(ArrIn),exs->andb))
    {
      /* The array is not local to the current andbox
       * A constraint is constructed without a suspension.
       * The constraint will be retried when promoted.
       */
      return SUSPEND;
    }

  Deref(Off, Off);
  IfVarSuspend(Off);

  if (!IsINT(Off))
    USAGE_FAULT("get/2: Index is not an integer");

  p = Arr(ArrIn);
  i = IntVal(Int(Off));
  i -= p->start;		/* Adjust to zero based index */
  i *= p->scale;		/* Scale to cell position */

  if ((i < 0) || (i >= p->size))
    USAGE_FAULT("get/2: Index out of range");

  i += p->offset;		/* Adjust to original index */

  ArgTmp = StrArgument(Str(p->data),i);

  Deref(ArgOld, ArgOld);
  Deref(ArgTmp, ArgTmp);

  return unify(ArgOld,ArgTmp,exs->andb,exs);
}

/* ------------------------------------------------------------------------ */

/* set(+Index, +Term)@ +Array */

static bool method_set_array(Off, ArgNew, ArrIn, exs)
     Term Off, ArgNew, ArrIn;
     exstate *exs;
{
  Term *x;
  Array *p;
  int i;

  /* ArrIn is self and already dereferenced and checked */

  if(!IsLocalGVA(Ref(ArrIn),exs->andb))
    {
      /* The array is not local to the current andbox
       * A constraint is constructed without a suspension.
       * The constraint will be retried when promoted.
       */
      return SUSPEND;
    }

  Deref(Off, Off);
  IfVarSuspend(Off);

  if (!IsINT(Off))
    USAGE_FAULT("set/2: Index is not an integer");

  p = Arr(ArrIn);
  i = IntVal(Int(Off));
  i -= p->start;		/* Adjust to zero based index */
  i *= p->scale;		/* Scale to cell position */

  if ((i < 0) || (i >= p->size))
    USAGE_FAULT("set/2: Index out of range");

  i += p->offset;		/* Adjust to original index */

  x = StrArgRef(Str(p->data),i);
  *x = ArgNew;			/* Destructive set */
	      
  return TRUE;
}

/* ------------------------------------------------------------------------ */

/* set(+Index, -OldTerm, +Term)@ +Array */

static bool method_swap_array(Off, ArgOld, ArgNew, ArrIn, exs)
     Term Off, ArgOld, ArgNew, ArrIn;
     exstate *exs;
{
  Term ArgTmp, *x;
  Array *p;
  int i;

  /* ArrIn is self and already dereferenced and checked */

  if(!IsLocalGVA(Ref(ArrIn),exs->andb))
    {
      /* The array is not local to the current andbox
       * A constraint is constructed without a suspension.
       * The constraint will be retried when promoted.
       */
      return SUSPEND;
    }

  Deref(Off, Off);
  IfVarSuspend(Off);

  if (!IsINT(Off))
    USAGE_FAULT("set/3: Index is not an integer");

  p = Arr(ArrIn);
  i = IntVal(Int(Off));
  i -= p->start;		/* Adjust to zero based index */

  if ((i < 0) || (i >= p->size))
    USAGE_FAULT("set/3: Index out of range");

  i += p->offset;		/* Adjust to original index */

  x = StrArgRef(Str(p->data),i);
  ArgTmp = *x;
  *x = ArgNew;
	      
  Deref(ArgOld, ArgOld);
  Deref(ArgTmp, ArgTmp);

  return unify(ArgOld,ArgTmp,exs->andb,exs);
}

/* ------------------------------------------------------------------------ */

/* typeof(-Type)@ +Array */

static bool method_typeof(type, exs)
     Term type;
     exstate *exs;     
{
  Deref(type,type);
  return unify(type, term_array, exs->andb, exs);
}

/* ------------------------------------------------------------------------ */

/* sizeof(-Size)@ +Array */

static bool method_sizeof(size, self, exs)
     Term size, self;
     exstate *exs;     
{
  int i;
  Term tmp;
  Array *arr = Arr(self);

  i = arr->size / arr->scale;	/* number of elements */

  MakeIntegerTerm(tmp, i);
  Deref(size,size);
  return unify(size, tmp, exs->andb, exs);
}

/* ------------------------------------------------------------------------ */

void initialize_array() {

  atom_nil = store_atom("[]");
  term_array = TagAtm(store_atom("array"));

  functor_typeof = store_functor(store_atom("typeof"),1);
  functor_sizeof = store_functor(store_atom("sizeof"),1);

  functor_subarray_3 = store_functor(store_atom("subarray"),3);
  functor_subarray_4 = store_functor(store_atom("subarray"),4);
  functor_subarray_5 = store_functor(store_atom("subarray"),5);
  functor_get_2 = store_functor(store_atom("get"),2);
  functor_set_2 = store_functor(store_atom("set"),2);
  functor_set_3 = store_functor(store_atom("set"),3);

  define("new_array",akl_new_array_2,2);
  define("new_array",akl_new_array_3,3);
  define("new_array",akl_new_array_4,4);
  define("array",akl_array,1);
}
