;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;;    FILE:
;; 	aspect-mode.el
;; 
;;    DESCRIPTION:
;; 		First try at a major mode for editing ASpecT-sources.
;; 	
;; 
;; 	$Author: holly $   $Revision: 1.11 $
;; 	$Date: 1992/08/20 13:16:20 $
;;
;;	For comments, questions and ideas for enhancements, write to:
;;		holly@pc-labor.uni-bremen.de
;;	and/or to:
;;		hds@informatik.uni-bremen.de
;;
;;
;;  To use this package put something like the following 
;;  in your .emacs file:
;;
;;  (autoload 'aspect-mode "aspect-mode" "Major mode for editing ASpecT sources" t)
;;  (setq auto-mode-alist
;;      (append  '(("\\.AS$" . aspect-mode))
;;     	       auto-mode-alist))
;;  (setq aspect-mode-hook '(lambda ()
;;  			  (abbrev-mode 1)
;;  			  ))
;;
;;  To make use of the aspect-next-error function, you have to apply
;;  the "compile.el-diff"-Patch that should have come along with this
;;  file. To do this, cd into the directory where all emacs-lisp files
;;  are stored and say :
;;	patch < compile.el-patch
;;  after that, byte-compile the changed compile.el.
;;
;;  You will also need to change the compilation-error-regexp.
;;  In your .emacs file just after the other stuff from above say:
;;	(require 'compile)	;; force def. of compilation-error-regexp
;;	(setq compilation-error-regexp 
;;	      (concat "\\(^[^ :\n]+:.*at line [0-9]+\\)\\|" compilation-error-regexp))
;;  This will add another regexp to match the ASpecT-error messages.
;;
;;
;; 	$Log: aspect-mode.el,v $
;;  Revision 1.11  1992/08/20  13:16:20  holly
;;  Fixed on little bug.
;;
;;  Revision 1.10  1992/08/18  17:49:19  holly
;;  Added parsing of column-info in error text.  Error parsing now useful?
;;
;;  Revision 1.9  1992/08/18  16:37:01  holly
;;  Changed output in aspect-run output-buffer.
;;
;;  Revision 1.8  1992/08/18  15:53:04  holly
;;  Now allow for parsing of compilation errors.  Also: changed abbrevs.
;;
;;  Revision 1.7  1992/08/12  18:36:30  holly
;;  Now use gmhist on interactive call to aspect-find-spec
;;
;;  Revision 1.6  1992/08/12  17:15:49  holly
;;  Added aspect-find-spec function
;;
;;  Revision 1.5  1992/08/12  15:51:28  holly
;;  Added aspect-run function
;;
;;  Revision 1.4  1992/08/11  14:28:53  holly
;;  Minor fixes / enhancements.
;;
;;  Revision 1.3  1992/08/07  17:21:43  holly
;;  Little help msg in header.  Another abbrev-expansion.
;;
;;  Revision 1.2  1992/08/07  16:12:14  holly
;;  Now with abbrev table.
;;
;;  Revision 1.1  1992/08/07  15:28:09  holly
;;  First Version. aspect-gen and syntax-table.
;;
;; 
;;
;;  $Id: aspect-mode.el,v 1.11 1992/08/20 13:16:20 holly Exp $
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(require 'gmhist)
(require 'compile)

(defvar aspect-load-hook nil
  "Hooks to run after loading the aspect.el file.")

(defvar aspect-gen-command-name "gen" 
  "Name of the command to call gen")

(defvar aspect-mode-abbrev-table nil
  "Abbrev table in use in Aspect-mode buffers.")

(define-abbrev-table 'aspect-mode-abbrev-table ())

(defvar aspect-mode-map nil
  "Keymap used in Aspect mode.")


(defvar compilation-current-error-pos nil) ; in case we don't use the patched compile.el


(if (not aspect-mode-map)
    (progn
      (setq aspect-mode-map (make-sparse-keymap))
      (define-key aspect-mode-map "\C-c\C-f" 'aspect-find-spec)
      (define-key aspect-mode-map "\C-cf" 'aspect-find-spec)
      (define-key aspect-mode-map "\C-cg" 'aspect-gen)
      (define-key aspect-mode-map "\C-cr" 'aspect-run)
      (define-key aspect-mode-map "\C-c!" 'aspect-run)
      (define-key aspect-mode-map "\C-x`" 'aspect-next-error)
      ))

(defvar aspect-mode-syntax-table nil
  "Syntax table in use in Aspect-mode buffers.")
(if (not aspect-mode-syntax-table)
    (progn
      (setq aspect-mode-syntax-table (make-syntax-table))
      (modify-syntax-entry ?. "." aspect-mode-syntax-table)
      (modify-syntax-entry ?, "." aspect-mode-syntax-table)
      (modify-syntax-entry ?\\ "\\" aspect-mode-syntax-table)
      (modify-syntax-entry ?# "w" aspect-mode-syntax-table)
      (modify-syntax-entry ?_ "_" aspect-mode-syntax-table)
      (modify-syntax-entry ?{ "(}1" aspect-mode-syntax-table)
      (modify-syntax-entry ?} "){3" aspect-mode-syntax-table)
      (modify-syntax-entry ?* ". 23" aspect-mode-syntax-table)
      (modify-syntax-entry ?+ "." aspect-mode-syntax-table)
      (modify-syntax-entry ?- "." aspect-mode-syntax-table)
      (modify-syntax-entry ?/ ". 14" aspect-mode-syntax-table)
      (modify-syntax-entry ?' "\"" aspect-mode-syntax-table)
      (modify-syntax-entry ?< "." aspect-mode-syntax-table)
      (modify-syntax-entry ?> "." aspect-mode-syntax-table)
      (modify-syntax-entry ?= "." aspect-mode-syntax-table)
      (modify-syntax-entry ?| "." aspect-mode-syntax-table)
      )
)

;;; Abbreviations for some of the keywords of ASpecT
(defvar aspect-mode-abbrev-table nil
  "Abbreviation table for aspect-mode buffers")
(define-abbrev-table 'aspect-mode-abbrev-table '(
  ("##a" "ACTUAL" nil 1)
  ("##e" "EQNS" nil 1)
  ("##f" "FORALL" nil 1)
  ("##g" "GLOBAL" nil 1)
  ("##l" "LOCAL" nil 1)
  ("##m" "MACROS" nil 1)
  ("##n" "END" nil 1)		;; 'E' was already taken... sorry.
  ("##o" "OPNS" nil 1)
  ("##r" "FORMAL" nil 1)
  ("##s" "SORTS" nil 1)
  ("##t" "THEOREMS" nil 1)
  ("##x" "EXTERN" nil 1)
  ))

(defun aspect-gen ()
  "* Call gen for this file"
  (interactive)
  (let* ((fname (buffer-file-name))
	 (dummy (string-match "\\([^/]*\\).AS$" fname))
	 (name (substring fname (match-beginning 1) (match-end 1)))
	 (command (concat aspect-gen-command-name " " name))
	 )
    (compile command)
    )
)


(defun aspect-run ()
  "* Run program generated from current specification"
  (interactive)
  (let* ((fname (buffer-file-name))
	 (dummy (string-match "\\(.*\\)/\\([^/]*\\).AS$" fname))
	 (dirname (substring fname (match-beginning 1) (match-end 1)))
	 (uppername (substring fname (match-beginning 2) (match-end 2)))
	 (lowername (downcase uppername))
	 (shell (getenv "SHELL"))
	 (simplecommand (concat dirname "/" lowername))
	 (out-buffer-name (concat "*Output of " lowername "*"))
	 (out-buffer (get-buffer-create out-buffer-name))
	 (old-buffer (current-buffer))
	 )
    (display-buffer out-buffer)
    (save-excursion
      (set-buffer out-buffer)
      ;;; (erase-buffer)
      (with-output-to-temp-buffer out-buffer-name
	(if shell
	    (progn 
	      (princ "cd ") 
	      (princ dirname) 
	      (princ " ; ")
	      (princ lowername))
	  (princ simplecommand))
	(terpri))
      (goto-char (point-max))
      (sleep-for 0)
      )
      (if shell
	  (call-process shell nil out-buffer t "-c" (concat "( cd " dirname " ; " lowername ")"))
	(call-process (concat dirname "/" lowername) nil out-buffer t)
	)
      (save-excursion
	(set-buffer out-buffer)
	(insert-string "\ndone.\n")
	)
      )
  )

(defun id (a) a)
(defun string-to-charlist (str)
  (mapcar 'id str))
(defun charlist-to-string (ls)
  (mapconcat 'char-to-string ls ""))
(defun takewhile (pred list)
  (cond ((eq list ()) ())
	((funcall pred (car list)) (cons (car list) (takewhile pred (cdr list))))
	(t ())
	))
(defun dropwhile (pred list)
  (cond ((eq list ()) ())
	((funcall pred (car list)) (dropwhile pred (cdr list)))
	(t list)
	))

(defun split-string-at (str split)
  (if (string= str "")
      ()
    (let* ((strlist (mapcar 'char-to-string str))
	   (firstword (takewhile '(lambda(s) (not (string= s split))) strlist))
	   (restwords (dropwhile '(lambda(s) (not (string= s split))) strlist))
	   (rest (if (eq restwords ()) () (cdr restwords)))
	   )
      (cons (mapconcat 'id firstword "")
	    (split-string-at (mapconcat 'id rest "") split ))
      )))
(defun aspect-find-specpath (fname pathlist)
  (cond ((eq pathlist ()) nil)
	((file-exists-p (concat (car pathlist) "/" fname)) (concat (car pathlist) "/" fname))
	(t (aspect-find-specpath fname (cdr pathlist)))))


(defun aspect-find-spec (specname)
  "* Find a specification.
Finds a specification in path described by env-var $ASPECT."
  (interactive (gmhist-interactive "sSpec to find: " 'aspect-spec-history))
  (let* ((path (split-string-at (getenv "ASPECT") ":"))
	 (uspecname (upcase specname))
	 (fname (or
		 (aspect-find-specpath specname path)
		 (aspect-find-specpath (concat specname ".AS") path)
		 (aspect-find-specpath uspecname path)
		 (aspect-find-specpath (concat uspecname ".AS") path)))
	 )
    (if fname
	(find-file-other-window fname)
      (message (concat "Cannot find a spec " specname "!")))
    ))


(defun aspect-find-spec-noselect (specname)
  "* Find a specification via find-file-noselect.
Finds a specification in path described by env-var $ASPECT."
  (interactive (gmhist-interactive "sSpec to find: " 'aspect-spec-history))
  (let* ((path (split-string-at (getenv "ASPECT") ":"))
	 (uspecname (upcase specname))
	 (fname (or
		 (aspect-find-specpath (concat specname ".AS") path)
		 (aspect-find-specpath specname path)
		 (aspect-find-specpath uspecname path)
		 (aspect-find-specpath (concat uspecname ".AS") path)))
	 )
    (if fname
	(find-file-noselect fname)
      (progn (message (concat "Cannot find spec by name of " specname)) nil))
    ))

(defun aspect-next-error (&optional argp)
  "Find spec with next error.
NOTE!!! This function really needs the patches for compile.el"
  (interactive "P")
  (let ((compile-file-finder 'aspect-find-spec-noselect)
	column
	)
    (next-error argp)
    ;;; Now decode the column info.
    ;;; We rely on the fact that the patched compile.el will store 
    ;;; the point of the last error-text in buffer *compilation* into the
    ;;; variable compilation-current-error-pos.
    ;;; From here on we decode the column info and act accordingly.
    (save-excursion
      (set-buffer "*compilation*")
      (if  compilation-current-error-pos
	  (progn
	    (goto-char compilation-current-error-pos)
	    (end-of-line)
	    (backward-char)	; Some errors end w/ a period, some don't.
	    (if (not(looking-at "\\."))
		(forward-char))
	    (let ((endp (point))
		  columnstring)
	      (skip-chars-backward "[0-9]")
	      (setq columnstring (buffer-substring (point) endp))
	      (setq column (string-to-int columnstring))
	      )
	    )))
    (move-to-column (- column 1))
    ))



(defun aspect-mode ()
  "'Major mode for editing ASpecT-code.

Current Key bindings as follows:
\\{aspect-mode-map}

Local variable to modify:
	aspect-gen-command-name  (default: \"gen\") = command to use for aspect-gen

A mode-local abbrev-table has been provided.
After entering mode aspect-mode-hook will be run if non-nil.
After loading the aspect-mode.el file, aspect-load-hook will be run if non-nil.
		"
  (interactive)
  (kill-all-local-variables)
  (use-local-map aspect-mode-map)
  (setq mode-name "ASpecT")
  (setq major-mode 'aspect-mode)
  (set-syntax-table aspect-mode-syntax-table)
  (setq local-abbrev-table aspect-mode-abbrev-table)
  (run-hooks 'aspect-mode-hook)
  )



(run-hooks 'aspect-load-hook)
