/*  $Id: pce-lisp.c,v 1.6 1993/01/22 15:40:43 anjo Exp $

    File	pce-lisp.c
    Part of	PCE/Lisp interface
    Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
    Purpose	Definition of C interface between Lisp and PCE
    Works with	SCL 3.0, SCL 4.0, LispWorks 3.1
    		PCE 4.5, gcc 2.2
    Notice	Copyright (c) 1992  University of Amsterdam

    History	17/06/92  (Created)
		26/10/92  (Receiver and selector pushed as arguments)
   		14/12/92  (Last Modified)
*/


#include <stdio.h>
#include <ctype.h>
#include <varargs.h>
#include <sys/types.h>

#include "itf-interface.h"		/* PCE foreign language interface */


#define MAX_NAME_LENGTH         1024	/* Maximum name-length (assocs) */
#define PCE_MAX_ARGS		100	/* used to be in PCE */

#define Object   PceObject

#ifndef TRUE
#       define TRUE     1
#       define FALSE    0
#endif


/* #define MARKING			FOR THE TIME BEING */

Object	CONSTANT_abort;			/* Name: "abort" */
Object	CONSTANT_backtrace;		/* Name: "backtrace" */
Object	CONSTANT_break;			/* Name: "break" */
Object	CONSTANT_class;			/* Name: "class" */
Object	CONSTANT_convert;		/* Name: "convert" */
Object	CONSTANT_fatal;			/* Name: "fatal" */
Object	CONSTANT_halt;			/* Name: "halt" */
Object	CONSTANT_lisp_string;		/* Name: "lisp_string" */
Object	CONSTANT_lisp_symbol;		/* Name: "lisp_symbol" */
Object	CONSTANT_lisp_special;		/* Name: "lisp_special" */
Object	CONSTANT_lookup_lisp_symbol;	/* Name: "lookup_lisp_symbol" */
Object	CONSTANT_name;			/* Name: "name" */
Object	CONSTANT_on;			/* Object: @on */
Object	CONSTANT_object;		/* Name: "object" */
Object	CONSTANT_package;		/* Name: "package" */
Object	CONSTANT_pce;			/* Object: @pce */
Object	CONSTANT_protect;		/* Name: "protect" */
Object	CONSTANT_register_lisp_symbol;	/* Name: "register_lisp_symbol" */
Object	CONSTANT_string;		/* Name: "string" */
Object	CONSTANT_syntax;		/* Name: "syntax" */
Object	CONSTANT_unchecked;		/* Name: "uncheked" */
Object	CONSTANT_uppercase;		/* Name: "uppercase" */
Object	CONSTANT_value;			/* Name: "value" */


typedef void (*VoidFunc)();		/* pointer to void function */


#define SetArgc(x)		CurrentVector->status->argc = x
#define GetArgc()		CurrentVector->status->argc
#define IncArgc()		CurrentVector->status->argc++

#define SetError(ps, x)		ps->error = x
#define SetReturnType(ps, x)	ps->return_type = x
#define GetReturnType(ps)	ps->return_type
#define SetResult(ps, x)	ps->result.as_object = (Object) x


typedef struct m_vector *	Mvector;
typedef struct pcelisp_status *	PcelispStatus;

struct pcelisp_status
{ int		argc;			/* Argument counter */
  int		error;			/* Error code */
  int		name;			/* If TRUE name for new */
  int		return_type;		/* Type of return value */
  union
  {  int	as_int;
     float	as_float;
     char *	as_charp;
     Object	as_object;
     PceITFSymbol as_symbol;
  } result;				/* Result of computation */
  Object	return_as;		/* How to return a value (pce-get) */
};


struct m_vector
{ int		instruction;		/* Instruction code */
  int		argc;			/* Number of arguments in vector */
  Object        argv[PCE_MAX_ARGS];	/* Argument vector */
  PcelispStatus	status;
  char		name[MAX_NAME_LENGTH];	/* Assoc for object (new) */
};


Mvector		CurrentVector;


/*----------------------------------------------
 *  Instructions
 *----------------------------------------------*/

#define PCELISP_SEND			1
#define PCELISP_GET			2
#define PCELISP_NEW			3


/*----------------------------------------------
 *  Receiver Types
 *----------------------------------------------*/

#define PCELISP_INTEGER			1 /* PCE_INTEGER */
#define PCELISP_NAME			2 /* PCE_NAME */
#define PCELISP_REF			3 /* PCE_REFERENCE */
#define PCELISP_ASSOC			4 /* PCE_ASSOC */
#define PCELISP_REAL			5 /* PCE_REAL */
#define PCELISP_NATIVE			6
#define PCELISP_STRING			7
#define PCELISP_SYMBOL			8 /* Symbols if non-static */
#define PCELISP_SPECIAL			9 /* Lisp native special values */


/*----------------------------------------------
 *  Error Codes
 *----------------------------------------------*/

#define PCELISP_NO_ERROR		0
#define PCELISP_UNKNOWN_ASSOC		1
#define PCELISP_UNKNOWN_NAME		2
#define PCELISP_UNKNOWN_REFERENCE	3
#define PCELISP_RECEIVER_NOT_AN_OBJECT	4 /* NO LONGER IN USE */
#define PCELISP_MESSAGE_FAILED	       	5
#define PCELISP_UNKNOWN_SYMBOL		6
#define PCELISP_ILLEGAL_ASSOC		7
#define PCELISP_MESSAGE_SUCCEEDED      	8
#define PCELISP_UNKNOWN_RETURN_AS	9


/*----------------------------------------------
 *  Prototype Declarations
 *----------------------------------------------*/

void *		alloc(int);		/* PCE allocation */
void 		unalloc(int, void *);	/* PCE de-allocation */
AnswerMark	pcelisp_mark(void);
void		pcelisp_rewind(AnswerMark);
void *		define_lisp_symbol(void *name, void *package);
void *		pcelisp_define_symbol(char *name, char *package);


#define PCELISP_ASSOC_HANDLE		0
#define PCELISP_NAME_HANDLE		1
#ifdef PCELISP_STATIC_SYMBOLS
#	define PCELISP_SYMBOL_HANDLE	PCELISP_NAME_HANDLE
#else
#	define PCELISP_SYMBOL_HANDLE	2
#endif


/*----------------------------------------------
 *  Public Interface Functions (from Lisp)
 *----------------------------------------------*/

int
pcelisp_c_initialise()
{ CurrentVector = alloc(sizeof(struct m_vector));

  if (pceInitialise(PCELISP_SYMBOL_HANDLE+1, 0, NULL) != PCE_SUCCEED)
  { fprintf(stderr, "Failed to initialise PCE...\nAborted\n");
    exit(0);
  }

  CONSTANT_abort = cToPceName("abort");
  CONSTANT_backtrace = cToPceName("backtrace");
  CONSTANT_break = cToPceName("break");
  CONSTANT_class = cToPceName("class");
  CONSTANT_convert = cToPceName("convert");
  CONSTANT_fatal = cToPceName("fatal");
  CONSTANT_halt = cToPceName("halt");
  CONSTANT_lisp_string = cToPceName("lisp_string");
  CONSTANT_lisp_symbol = cToPceName("lisp_symbol");
  CONSTANT_lisp_special = cToPceName("lisp_special");
  CONSTANT_lookup_lisp_symbol = cToPceName("lookup_lisp_symbol");
  CONSTANT_name = cToPceName("name");
  CONSTANT_on = cToPceAssoc("on");
  CONSTANT_object = cToPceName("object");
  CONSTANT_package = cToPceName("package");
  CONSTANT_pce = cToPceAssoc("pce");
  CONSTANT_protect = cToPceName("protect");
  CONSTANT_register_lisp_symbol = cToPceName("register_lisp_symbol");
  CONSTANT_string = cToPceName("string");
  CONSTANT_syntax = cToPceName("syntax");
  CONSTANT_unchecked = cToPceName("unchecked");
  CONSTANT_uppercase = cToPceName("uppercase");
  CONSTANT_value = cToPceName("value");

  /* Define internal classes such that the pce-<class> functions
     are defined.
   */
  { Object argv[2];

    argv[0] = CONSTANT_lisp_symbol;	/* Predefine class lisp-symbol */
    argv[1] = CONSTANT_name;

    pceNew(NULL, CONSTANT_class, 2, argv);

    argv[0] = CONSTANT_lisp_string;	/* Predefine class lisp-string */
    argv[1] = CONSTANT_string;

    pceNew(NULL, CONSTANT_class, 2, argv);

    argv[0] = CONSTANT_lisp_special;	/* Predefine class lisp-special */
    argv[1] = CONSTANT_object;

    pceNew(NULL, CONSTANT_class, 2, argv);
  }

  { Object argv[2];

    argv[0] = CONSTANT_uppercase;
    argv[1] = cToPceInteger('-');
    pceSend(CONSTANT_pce, CONSTANT_syntax, 2, argv);
  }

  return TRUE;
}


int
pcelisp_c_new_assoc(char *name, void *symbol)
{ Object rval;
  char buf[strlen(name)+1];

  strcpy(buf, name);

  if (rval = cToPceAssoc(buf))
  { pceRegisterAssoc(PCELISP_ASSOC_HANDLE, symbol, rval);
    return TRUE;
  }

  SetError(CurrentVector->status, PCELISP_ILLEGAL_ASSOC);
  return FALSE;
}


int
pcelisp_c_new_name(char *name, void *symbol)
{ Object rval;

  char buf[strlen(name)+1];

  strcpy(buf, name);

  if (rval = cToPceName(name))
  { pceRegisterName(PCELISP_NAME_HANDLE, symbol, rval);
    return TRUE;
  }

  return FALSE;
}


int
pcelisp_c_new_symbol(char *name, char *package)
{ char buf_name[strlen(name)+1];
  char buf_package[strlen(package)+1];
  Object argv[2];

  strcpy(buf_name, name);
  strcpy(buf_package, package);
  argv[0] = cToPceName(buf_name);
  argv[1] = cToPceName(buf_package);

  pceNew(NULL, CONSTANT_lisp_symbol, 2, argv);

  return TRUE;
}


/*----------------------------------------------
 *  Argument Pushing
 *----------------------------------------------*/

int
pcelisp_push(Object p)
{ int argc;

  argc = CurrentVector->argc;

  if (argc < PCE_MAX_ARGS)
  { CurrentVector->argv[CurrentVector->argc++] = p;
    if (CurrentVector->argc == CurrentVector->status->argc)
      pcelisp_execute(CurrentVector->status);
    return TRUE;
  }

  return FALSE;
}


int
pcelisp_c_int(int i)
{ return pcelisp_push(cToPceInteger(i));
}


int
pcelisp_c_special(int i)
{ Object special;
  Object argv[1];
  PceCValue rval;

  argv[0] = cToPceInteger(i);

  special = pceNew(NULL, CONSTANT_lisp_special, 1, argv);
  pcelisp_push(special);
  pceToC(special, &rval);
  
  return rval.integer;
}


int
pcelisp_c_real(float f)
{ return pcelisp_push(cToPceReal(f));
}


int
pcelisp_c_string(char *s)
{ Object argv[1];
  Object str;
  char buf[strlen(s)+1];
  AnswerMark mark;

  strcpy(buf, s);
  
  argv[0] = cToPceTmpCharArray(buf);
  str = pceNew(NULL, CONSTANT_lisp_string, 1, argv);
  donePceTmpCharArray(argv[0]);

  return pcelisp_push(str);
}


int
pcelisp_c_name(void *symbol)
{ PceITFSymbol assoc;

  if ((assoc = pceLookupHandle(PCELISP_NAME_HANDLE, symbol)) && assoc->name)
    return pcelisp_push(assoc->name);
  
  SetError(CurrentVector->status, PCELISP_UNKNOWN_NAME);
  return FALSE;
}


int
pcelisp_c_symbol(void *symbol)
{ PceITFSymbol assoc;

  if ((assoc = pceLookupHandle(PCELISP_SYMBOL_HANDLE, symbol)) && assoc->name)
    return pcelisp_push(assoc->name);

  return FALSE;
}


int
pcelisp_c_assoc(void *symbol)
{ PceITFSymbol assoc;
  
  if ((assoc = pceLookupHandle(PCELISP_ASSOC_HANDLE, symbol)) && assoc->object)
    return pcelisp_push(assoc->object);

  SetError(CurrentVector->status, PCELISP_UNKNOWN_ASSOC);
  return FALSE;
}


int
pcelisp_c_ref(int i)
{ Object ref;

  if (ref = cToPceReference(i))
    return pcelisp_push(ref);
  
  SetError(CurrentVector->status, PCELISP_UNKNOWN_REFERENCE);
  return FALSE;
}


/*----------------------------------------------
 *  Existence Testing
 *----------------------------------------------*/

int
pcelisp_c_is_assoc(void *symbol)
{ PceITFSymbol assoc;

  assoc = pceLookupHandle(PCELISP_ASSOC_HANDLE, symbol);
  return assoc && assoc->object;
}
     

int
pcelisp_c_is_ref(int p)
{ return (int) cToPceReference(p);
}
     

/*----------------------------------------------
 *  Virtual Machine Instructions
 *----------------------------------------------*/

void
pcelisp_answer(PcelispStatus ps, Object result)
{ PceCValue rval;
  int rtype;

  if (ps->return_as != CONSTANT_unchecked)
  { Object tmp;
    Object argv[2];

    argv[0] = result;
    argv[1] = ps->return_as;
    if (tmp = pceGet(CONSTANT_pce, CONSTANT_convert, 2, argv))
      result = tmp;
    else
      fprintf(stderr, "PCE/Lisp result (%s) not convertible to a %s\n",
	      pcePP(result),
	      pcePP(ps->return_as));
  }

  rtype = pceToC(result, &rval);

  SetReturnType(ps, rtype);
  SetResult(ps, rval.integer);

  switch (rtype) 
  { case PCE_INTEGER:   return;
    case PCE_REAL:      return;
    case PCE_REFERENCE:			/* EFFICIENCY ??? */
    { if (pceInstanceOf(result, CONSTANT_lisp_string))
      { SetResult(ps, pceCharArrayToC(result));
        SetReturnType(ps, PCELISP_STRING);
      }
      if (pceInstanceOf(result, CONSTANT_lisp_special))
        SetReturnType(ps, PCELISP_SPECIAL);
      return;
    }
    
    case PCE_ASSOC:
    { PceITFSymbol assoc;

      assoc = rval.itf_symbol;
      if (assoc->handle[PCELISP_ASSOC_HANDLE])
      { SetResult(ps, assoc->handle[PCELISP_ASSOC_HANDLE]);
        SetReturnType(ps, PCELISP_NATIVE);
	return;
      }
      return;
    }

    case PCE_NAME:
    { PceITFSymbol assoc;

      assoc = rval.itf_symbol;
      if (assoc->handle[PCELISP_NAME_HANDLE])
      { SetResult(ps, assoc->handle[PCELISP_NAME_HANDLE]);
#ifdef PCELISP_STATIC_SYMBOLS
        SetReturnType(ps, PCELISP_NATIVE);
#else
        SetReturnType(ps, PCELISP_SYMBOL);
#endif
      }
#ifndef PCELISP_STATIC_SYMBOLS
      if (assoc->handle[PCELISP_SYMBOL_HANDLE])
      { SetResult(ps, assoc->handle[PCELISP_SYMBOL_HANDLE]);
        SetReturnType(ps, PCELISP_SYMBOL);
      }
#endif
      return;
    }

    default:
	fprintf(stderr, "<<< Unknown return type %d >>>\n", rtype);
	exit(0);
  }
}


pcelisp_execute(PcelispStatus ps)
{ switch(CurrentVector->instruction)
  { case PCELISP_SEND:
    { int rval;

      rval = pceSend(CurrentVector->argv[0],
		     CurrentVector->argv[1],
		     CurrentVector->argc-2,
		     &CurrentVector->argv[2]) == PCE_SUCCEED;

      if (rval)
      { SetError(ps, PCELISP_MESSAGE_SUCCEEDED);
	return TRUE;
      } else
      { SetError(ps, PCELISP_MESSAGE_FAILED);
	return TRUE;
      }
    }

    case PCELISP_GET:
	{ Object result;

	  ps->return_as = CurrentVector->argv[0];
	  result = pceGet(CurrentVector->argv[1],
			  CurrentVector->argv[2],
			  CurrentVector->argc-3,
			  &CurrentVector->argv[3]);

	  if (result == PCE_FAIL)
	  { SetError(ps, PCELISP_MESSAGE_FAILED);
	    return FALSE;
	  }

	  SetError(ps, PCELISP_MESSAGE_SUCCEEDED);
	  pcelisp_answer(ps, result);
	  return;
	}

    case PCELISP_NEW:
	{ Object result;

	  result = pceNew((ps->name ? CurrentVector->name : NULL),
			  CurrentVector->argv[0],
			  CurrentVector->argc-1,
			  &CurrentVector->argv[1]);

	  if (result == PCE_FAIL)
	  { SetError(ps, PCELISP_MESSAGE_FAILED);
	    return FALSE;
	  }

	  SetError(ps, PCELISP_MESSAGE_SUCCEEDED);
	  pcelisp_answer(ps, result);
	  return;
	}

    default:
	fprintf(stderr, "PCE/Lisp internal error: unknown instruction %d\n",
		CurrentVector->instruction);
        exit(0);
  }
}


int
pcelisp_c_send(PcelispStatus ps)
{ CurrentVector->status = ps;
  CurrentVector->instruction = PCELISP_SEND;
  CurrentVector->argc = 0;

  return TRUE;
}


int
pcelisp_c_get(PcelispStatus ps)
{ Object result;

  CurrentVector->status = ps;
  CurrentVector->instruction = PCELISP_GET;
  CurrentVector->argc = 0;
}

     
int
pcelisp_c_new_named(char *name)
{ strcpy(CurrentVector->name, name);
  return TRUE;
}


int
pcelisp_c_new(PcelispStatus ps)
{ PceITFSymbol assoc;
  Object class;

  CurrentVector->status = ps;
  CurrentVector->instruction = PCELISP_NEW;
  CurrentVector->argc = 0;
  ps->return_as = CONSTANT_unchecked;

  return TRUE;
}

     
/*----------------------------------------------
 *  Returning Values
 *----------------------------------------------*/

int
pcelisp_c_fetch_integer(PcelispStatus ps)
{ return (int) ps->result.as_int;
}


float
pcelisp_c_fetch_real(PcelispStatus ps)
{ return (float) ps->result.as_float;
}


char *
pcelisp_c_fetch_string(PcelispStatus ps)
{ return ps->result.as_charp;
}


void *
pcelisp_c_fetch_assoc(PcelispStatus ps)
{ PceITFSymbol assoc;

  assoc = ps->result.as_symbol;
  return assoc->handle[PCELISP_ASSOC_HANDLE];
}


char *
pcelisp_c_fetch_assoc_name(PcelispStatus ps)
{ PceITFSymbol assoc;

  assoc = ps->result.as_symbol;
  return pceCharArrayToC(assoc->name);
}


void *
pcelisp_c_fetch_name(PcelispStatus ps)
{ return (void *) ps->result.as_object;
}


void *
pcelisp_c_fetch_native(PcelispStatus ps)
{ return (void *) ps->result.as_object;
}


/*----------------------------------------------
 *  Special Functions
 *----------------------------------------------*/

void
pcelisp_reset()
{ pceReset();
}


/*----------------------------------------------
 *  Host Functions
 *----------------------------------------------*/

static int	_hostAction(int action, va_list args);

int
hostSend(lisp, selector, argc, argv)
Object lisp;
Object selector;
int argc;
Object argv[];
{ int i;
  Object function;
  char *s, *t;
  char *name;
  PceCValue pt;

  if (selector == CONSTANT_register_lisp_symbol)
  { Object symbol = argv[0];
    Object symbol_name = pceGet(symbol, CONSTANT_value, 0, NULL);
    Object symbol_package = pceGet(symbol, CONSTANT_package, 0, NULL);
    void *lisp_symbol;

    pceSend(symbol, CONSTANT_protect, 0, NULL);
    lisp_symbol = define_lisp_symbol(symbol_name, symbol_package);
    pceRegisterName(PCELISP_SYMBOL_HANDLE, lisp_symbol, symbol);
    return PCE_SUCCEED;
  }

  pceToC(selector, &pt);

  if (!pceInstanceOf(selector, CONSTANT_lisp_symbol))
  { fprintf(stderr, "PCE/Lisp: Illegal selector in hostSend()\n");
    return NULL;
  } else
  { PceITFSymbol assoc;

    assoc = pt.itf_symbol;
    if (assoc->handle[PCELISP_SYMBOL_HANDLE])
      function = assoc->handle[PCELISP_SYMBOL_HANDLE];
    else
    { fprintf(stderr, "PCE/Lisp internal: LispSymbol %s::%s unknown\n",
	      pcePP(pceGet(selector, CONSTANT_package, 0, NULL)),
	      pcePP(pceGet(selector, CONSTANT_name, 0, NULL)));
      exit(0);
    }
  }

  {  Mvector vector;

/*
fprintf(stderr, ";;; Calling %s(", pcePP(function));
for (i=0; i<argc; i++) fprintf(stderr, "%s ", pcePP(argv[i]));
fprintf(stderr, ")\n");
*/

     vector = alloc(sizeof(struct m_vector));

     for (i=0; i<argc; i++)
       vector->argv[i] = argv[i];

     vector->argc = argc;
 
     if (pce_call_back(argc, function, vector))
     { unalloc(sizeof(struct m_vector), vector);
       return PCE_SUCCEED;
     } else
     { unalloc(sizeof(struct m_vector), vector);
       return PCE_FAIL;
     }
   }
}


/*  void *define_lisp_symbol(Object name, Object package)
 *
 *  This function is called when an unknown symbol is detected
 *  by the PCE/Lisp interface.  This happens when the
 *  Lisp symbol is generated by PCE, or when a PCE saved state
 *  is loaded back into PCE.  define_lisp_symbol/# calls
 *  Lisp to force the symbol over the PCE/Lisp interface.
 *  Return the Lisp symbol reference returned by Lisp.
 */
void *
define_lisp_symbol(Object name, Object package)
{ return pcelisp_define_symbol(pceCharArrayToC(name),
			       pceCharArrayToC(package));
}


int
pcelisp_c_pull(PcelispStatus ps, Mvector mv, int argc)
{ ps->return_as = CONSTANT_unchecked;
  if (argc > mv->argc)
    return FALSE;
  pcelisp_answer(ps, mv->argv[argc]);
  return TRUE;
}


Object
hostGet(Object lisp, Object name, int argc, Object argv[])
{ if (name == CONSTANT_lookup_lisp_symbol)
  { void *lisp_symbol;
    PceITFSymbol assoc;

    lisp_symbol = define_lisp_symbol(argv[0], argv[1]);
    if (    (assoc = pceLookupHandle(PCELISP_SYMBOL_HANDLE, lisp_symbol))
         && assoc->name)
      return assoc->name;
    
    return PCE_FAIL;
  }

  fprintf(stderr, "[PCE/Lisp warning: hostGet(%s, %d, ...) called\n",
	  pcePP(name), argc);

  return PCE_FAIL;
}


int     
hostQuery(i, val)
int i;
PceCValue *val;
{ switch(i)
  { case HOST_GETC:
        return PCE_FAIL;

    case HOST_SYMBOLFILE:
	val->string = PCE_SYMBOLFILE;
	return PCE_SUCCEED;

    default:
        fprintf(stderr, "[PCE/Lisp warning: hostQuery(%d, ...) called\n", i);
  }
  
  return PCE_FAIL;
}


int
hostAction(va_alist)
va_dcl
{ va_list args;
  int action, rval;

  va_start(args);
  action = va_arg(args, int);
  rval = _hostAction(action, args);
  va_end(args);
  return rval;
}


void *
nameToLispKeyword(Object name)
{ PceITFSymbol assoc;

  assoc = getITFSymbolName(name);
  return assoc->handle[PCELISP_NAME_HANDLE];
}


static int
_hostAction(int action, va_list args)
{ switch(action)
  { case HOST_ABORT:			/* Abort to top-level */
      pcelisp_call_back(nameToLispKeyword(CONSTANT_abort));
      return PCE_SUCCEED;

    case HOST_BACKTRACE:		/* Lisp stacktrace */
      pcelisp_call_back(nameToLispKeyword(CONSTANT_backtrace));
      return PCE_SUCCEED;

    case HOST_BREAK:			/* Interactive top-level */
      pcelisp_call_back(nameToLispKeyword(CONSTANT_break));
      return PCE_SUCCEED;

    case HOST_FLUSH:
      fflush(stdout);
      return PCE_SUCCEED;

    case HOST_HALT:			/* Exit */
      pcelisp_call_back(nameToLispKeyword(CONSTANT_halt));

#ifdef sun
    case HOST_ONEXIT:			/* Called when PCE exists */
    { VoidFunc function;
      caddr_t f_arg;

      function = va_arg(args, VoidFunc);
      f_arg = va_arg(args, caddr_t);

      on_exit(function, f_arg);
      return PCE_SUCCEED;
    }
#endif

    case HOST_RECOVER_FROM_FATAL_ERROR:	/* Effectively same as abort */
      pcelisp_call_back(nameToLispKeyword(CONSTANT_fatal));

    case HOST_SIGNAL:			/* Set signal function */
    { int sig = va_arg(args, int);
      VoidFunc func = va_arg(args, VoidFunc);
      signal(sig, func);
      return PCE_SUCCEED;
    }

    case HOST_TRACE:			/* Interactive tracing */
      fprintf(stderr, ";;; No interactive trace in Lisp...\n");
      return PCE_SUCCEED;

    case HOST_WRITE:			/* Write a string on console */
      printf("%s", va_arg(args, char *));
      return PCE_SUCCEED;

  }
  fprintf(stderr, "[PCE/Lisp warning: hostAction(%d, ...) called\n", action);
  return PCE_FAIL;
}


/*----------------------------------------------
 *   Marking and Rewinding
 *----------------------------------------------*/

AnswerMark
pcelisp_mark()
{ AnswerMark mark;

  markAnswerStack(mark);
  return mark;
}


void
pcelisp_rewind(AnswerMark mark)
{ rewindAnswerStack(mark, NULL);
}

