%parser.pl
%
% contains all general predicates pertaining to  parse trees etc.
%

%
% gr2nl( +Basename)
%
% Translate the grammar rules in <Basename>.gr and print them out
% into <Basename.nl>
%
gr2nl( BaseName) :-
	atom_chars( BaseName, Chars),
	append( Chars, ".gr", InputFile),
	append( Chars, ".nl", OutputFile),
	read_clauses( InputFile, Rules),
	translate_rules( Rules, Terms, GrSyms),
	copy_term( Rules, Rules1),	
	copy_term( GrSyms, GrSyms1),
	remove_duplicates( GrSyms1, GrSyms2),
	generate_descendents( Rules1, GrSyms2, Desc),
	remove_duplicates( Desc, Desc1),
	open( OutputFile, write, Stream1),
	write_terms( Stream1, Terms),
	write_terms( Stream1, Desc1),
	close( Stream1).

% translate the rules
%
% as per Clocksin & Mellish / Gazdar & Mellish too!

translate_rules( [], [], []):-!.
translate_rules( [Rule|Rules], [Term|Terms], [GrSym|GrSyms]) :-
	translate( Rule, Term, GrSym),
	translate_rules( Rules, Terms, GrSyms).

translate( (P1-->P2), (G1:-G3), P1) :-
	right_hand_side( P2, S0, S, [], Nodes, G2),
	left_hand_side( P1, S0, S, Nodes, G1),
	ptn_flatten( G2, G3).

left_hand_side( (P1, P2), S0, Sn, Node, G) :-
	nonvar( P1),
	list( P2),
	lhs_tag( P1, S0, S1, Node, G),
	append( P2, Sn, S1).
left_hand_side( P0, S0, S, Node, G) :-
	nonvar( P0),
	lhs_tag( P0, S0, S, Node,  G).

right_hand_side( V, S0, Sn, Nodes, Nodes, phrase( V, S0, Sn)) :-
	var( V), !.
right_hand_side( (P1, P2), S0, S, NodesSoFar, Nodes, G) :- !,
	right_hand_side( P2, S1, S, NodesSoFar, N1, G2),
	right_hand_side( P1, S0, S1, N1, Nodes, G1),
	and( G1, G2, G).
right_hand_side( (P1;P2), S0, S, NodesSoFar, Nodes, (G1;G2)) :- !,
	or( P1, S0, S, NodesSoFar, N1, G1),
	or( P2, S0, S, N1, Nodes, G2).
right_hand_side( {P}, S0, S0, Nodes, Nodes, P) :- !.
right_hand_side( !, S0, S0, Nodes, Nodes, !) :- !.
right_hand_side( [], S, S, Nodes, [empty|Nodes], true) :-!.	
right_hand_side(  P, S0, S, Nodes, [tnode( atom, P)|Nodes], true) :-
	P = [_|_], !,
	append( P, S, S0).
right_hand_side( P, S0, S, Nodes, [Node|Nodes], G) :-
	rhs_tag( P, S0, S, Node, G).

lhs_tag( P, S0, S, [N], G) :-
	nonvar(N),	
	N = tnode( atom, List),
	P =..[ F|A],
	append( A, [S0, S], AX),
	append( AX, [tnode( P, List)], AY),
	G =..[ F| AY].
lhs_tag( P, S0, S, Childs, G) :-
	P =..[ F|A],
	append( A, [S0, S], AX),
	append( AX, [ node( P, Childs)], AY),
	G =..[ F| AY].

rhs_tag( P, S0, S, N, G) :-
	P =..[ F|A],
	append( A, [S0, S], AX),
	append( AX, [N], AY),
	G =..[ F| AY].
 
and( true, G, G) :- !.
and( G, true, G) :- !.
and( G1, G2, ( G1, G2)).

or( P, S0, S, N, N1, G) :-
	right_hand_side( P, S0a, S, N, N1, Ga),
	( var( S0a), S0a = S, !, S0 = S0a, G = Ga;
	G = (S0=S0a, Ga) ).

ptn_flatten( A, A) :-
	var( A), !.
ptn_flatten( ( A, B), C) :- !,
	flatten1( A, C, R),
	ptn_flatten( B, R).
ptn_flatten( A, A).

flatten1( A, (A, R), R) :- 
	var( A), !.
flatten1( ( A, B), C, R) :-!,
	flatten1( A, C, R1),
	flatten1( B, R1, R).
flatten1( A, ( A, R), R).

phrase( Category, String, Left) :-
	Category =..List,
	append( List, [ String, Left], New),
	Goal =.. New,
	call( Goal).

% generate a set of descendent/2 facts for use when building
% parse trees from (sub) parse trees, without reparsing
%
generate_descendents( [], _, []).
generate_descendents( [Rule|Rules], GrSyms, [ Desc1|RDescs]) :-
	generate_desc( Rule, GrSyms, Desc1), !,
	generate_descendents( Rules, GrSyms, RDescs).
generate_descendents( [_|Rules], GrSyms, RDescs) :-
	generate_descendents( Rules, GrSyms, RDescs).

generate_desc( (LHS-->RHS), GrSyms, descendent( LHS, Symbols)) :-
	separate_symbols( RHS, Symbols),
	one_member( Symbols, GrSyms).

separate_symbols( (R1, R2), Symbols) :-
	!, separate_symbols( R1, Sym1),
	separate_symbols( R2, Sym2),
	append( Sym1, Sym2, Symbols).
separate_symbols( (R1; R2), Symbols) :-
	!, separate_symbols( R1, Sym1),
	separate_symbols( R2, Sym2),
	append( Sym1, Sym2, Symbols).
separate_symbols( [R], [R]):- !.
separate_symbols( {_}, []):- !.
separate_symbols( R, [R]).

one_member( List1, List2) :-
	member( Z, List1),
	nonvar(Z),
	member( Z, List2).

remove_duplicates( [], []).
remove_duplicates( [H|T], Rest) :-
	member( H, T), !, 
	remove_duplicates( T, Rest) ;
	( Rest = [H|T1], remove_duplicates( T, T1)).

%
% parse the list of Tokens according to GrSymbol, and
% return a parsetree, Pt
%
parse( Grsymbol, Tokens, Pt) :-
	nonvar( Grsymbol),
	nonvar( Tokens),!,
	Grsymbol =..List,
	append( List, [Tokens, [], Pt], Args),
	Pred =..Args,
	call( Pred).

% parse_string( +GrSymbol, +String, -Parsetree)
% parse String according to GrSymbol, and return a parsetree
%
parse_string( atom, String, Pt) :-
	!, atom_chars( Atom, String),
	Pt = tnode( atom, [Atom]).
parse_string( number, String, Pt) :-
	!, number_chars( Int, String),
	Pt = tnode( number, [ Int]).
parse_string( var, String, Pt) :-
	nonvar( String), !, 
	stringToTokens( String, [[S|var]]),
	Pt = tnode( var, [ S]).
parse_string( Grsymbol, String, Pt) :-
	nonvar( Grsymbol),
	nonvar( String), !,
	build_tokens( String, Tokens),
	Grsymbol =..List,
	append( List, [Tokens, [], Pt], Args),	
	Pred =..Args,
	call( Pred).

parse_atom( Atom, tnode( atom, [Atom])).

parse_var( Var, tnode( var, [ Var])).

parse_number( Num, tnode( number, [Num])).

%
% build a parsetree, NewPt, according to Gr (Grammar rule symbol),
% from a list of the composite parsetrees, or Strings, or atoms
% uses descendent to optimize the process.
% 
build_parse( Gr, Pts, NewPt) :-
	get_names( Pts, Names),
	descendent( Gr, Names), !,
	all_parsed( Pts, PPts),
	NewPt = node( Gr, PPts).
build_parse( Gr, Pts, NewPt) :-
	!, all_parsed( Pts, PPts),
	extract_tokens( PPts, Toks),
	parse( Gr, Toks, NewPt).

%
% check all components of list are parsed
%
all_parsed( [], []) :- !.
all_parsed( [ node( T, Chs)|RPts], [ node( T, Chs)|RPPts]) :-
	!, all_parsed( RPts, RPPts).
all_parsed( [tnode( T, Chs)|RPts], [ tnode( T, Chs)|RPPts]) :-
        !, all_parsed( RPts, RPPts).
all_parsed( [String|RPts], PPts) :-
	string( String), !, 
	string_parsetrees( String, Pts),
	all_parsed( RPts, RPPts),
	append( Pts, RPPts, PPts).
all_parsed( [Atom|RPts], [ Pt|RPPts]) :-
	atom( Atom), !,
	parse_atom( Atom, Pt),
	all_parsed( RPts, RPPts).

% given a string, returns a list of parse trees of each token
% 
string_parsetrees( String, Pts) :-
	stringToTokens( String, Tokens),
	tokens_pts( Tokens, Pts).

tokens_pts( [], []).
tokens_pts( [ H|T], [ Pt|Pts]) :-
        token_pt( H, Pt),
        tokens_pts( T, Pts).
 
token_pt( [ Item|T], tnode( T, [Item])).

% get the grammar rule symbols from the parse tree, - used by
% build parse in conjunction with descendent, in order to
% avoid reparsing.
%
get_names( [], []):- !.
get_names( [tnode( Gr, _)|Rest], [Gr|Names]) :-
	!, get_names( Rest, Names).
get_names( [node( Gr, _)|Rest], [Gr|Names]) :- 
	get_names( Rest, Names). 
get_names( [ A|Rest], [A|Names]) :- 
	atom( A), !, 
	get_names( Rest, Names).
get_names( [ S|Rest], Names) :-
	string( S), !,
	build_tokens( S, Toks),
	get_names( Rest, Rnames),
	append( Toks, Rnames, Names).
	
%
% true if P is a node.
isNode( P) :-
	(P = node( T, _) ; P = tnode( T, _)),
	ground( T).
isNode( P) :-
	nonvar(P),
	P = gap( _).





