/* This is scanner.pl
**
** Purpose: Scanning makefiles for PROM
**
** (c) T.Kielmann, 92-04-22
**
*/



/*
** Section: setting/resetting special operators for makefile reading
*/

setops :- op(1150,xfx,-->),
	  op(1050,xfx,=).

resetops :- op(1200,xfx,-->),
	    op(700,xfx,=).



/*
** Section: Scanning makefiles
*/



/* process_file scans the contents of a file */

process_file(File) :-
        test_and_open(File,Stream), !,
        repeat,
            read(Stream,Term),
            process_term(Term), !,
        close(Stream).



/* try to open a file, abort execution if not possible */
test_and_open(File,Stream) :-
        access_file(File,read),
        open(File,read,Stream), !.

test_and_open(File,_) :-
        write('prom: cannot open '),
	write(File),nl,
        abort.



/* process makefile entries
** syntax errors will be reported by the Prolog runtime system
*/

/* end_of_file terminates scanning */

process_term(end_of_file).



/* Handle included files */
process_term((include File)) :-
        expand(File,[FileExp]),
        process_file(FileExp),!,
        fail.

process_term((include File)) :-
        write('prom: invalid filename '),
	write(File),nl,
        abort.

process_term((search include File)) :-
        expand(File,[FileExp]),
        searchfor(FileExp,Fullname),
        process_file(Fullname),!,
        fail.

process_term((search include File)) :-
        write('prom: invalid filename '),
	write(File),nl,
        abort.


/* Handle makefile entries */

process_term(Term) :-
        is_valid_entry(Term,Functor,TermToEnter),
        assert_entry(Functor,TermToEnter),!,
        fail.

/* if Term is invalid, we abort scanning */

process_term(Term) :-
        write('prom: invalid makefile entry:'),nl,
        write(Term),nl,
        abort.



/* lookup search path */

searchfor(File,Fullname) :-
        recorded(searchpath,P),
        expand(P,[PE]),
        concat_atom([PE,'/',File],Fullname),
        access_file(Fullname,read).

searchfor(File,_) :-
        write('prom: cannot find '),
	write(File),nl,
        abort.



/* test for validity of makefile entries */

is_valid_entry((depend T : D),dep,dep(Target,Deplist)) :-
        term(Target,[T],[]),
        terms(Deplist,[D],[]).
is_valid_entry((define D = E),def,def(Defname,Expansion)) :-
	term(Defname,[D],[]),
	terms(Expansion,[E],[]).
is_valid_entry((default define D = E),defaultdef,def(Defname,Expansion)) :-
	term(Defname,[D],[]),
	terms(Expansion,[E],[]).
is_valid_entry((create T : D --> A),rule,rule(Target,DepList,ActList,No)) :-
	term(Target,[T],[]),
	terms(DepList,[D],[]),
	terms(ActList,[A],[]),
	getruleno(No).


/* default definitions must be checked before recording */

assert_entry(defaultdef,TermToEnter) :-
        test_and_record(defaultdef,TermToEnter).
assert_entry(Functor,TermToEnter) :-
        recordz(Functor,TermToEnter).


test_and_record(defaultdef,def(Defname,_)) :-
        recorded(def,def(Defname,_)).
test_and_record(defaultdef,def(Defname,Expansion)) :-
        recordz(def,def(Defname,Expansion)).


/* primitive syntax checks for makefile entries... */

terms([T]) --> term(T).
terms([T|Tlist]) --> [X],{X=..[',',T1|TL],
			  term(T,[T1],[]),
			  terms(Tlist,TL,[])
			 }.

term(T) --> constant(T).
term(T) --> structure(T).

constant(C) --> [C], { atom(C) ; var(C) }.

structure(S) --> [S], { S =.. [C|T], C \== ',', T \== [] }.


/* Maintain unique numbering for creation rules */
getruleno(No) :-
        gensym(rule,No).
