/*  SYSTEM.PL  */


:- module system.


:- public open_scratch_output/2,
          command_line_parameter_is/1,
          initialise_operating_system_interface/0.


/*
SPECIFICATION
-------------

These predicates do various operating-system dependent things. The
predicates in module FILES could be generally useful to lots of
programs. These ones in contrast are more specific to the needs of the
Tutor.


PUBLIC open_scratch_output( FullName-, Status- ):
-------------------------------------------------

Creates a scratch file, and sets FullName to its full name. If OK,
leaves Status ok, else sets it to an atom indicating failure, from the
same set as for open_output.


PUBLIC command_line_parameter_is( Param- ):
-------------------------------------------

The Tutor should have been called with one parameter, the name of the
lesson to display. This predicate sets Param to that name, as an atom.
If it can't determine the parameter, it sets Param to
'no_parameter_found'.

At Oxford, the Tutor is invoked from a command file which assigns the
lesson name to a VAX/VMS logical variable called P1. This predicate
calls (via Pop-11) a VAX system routine to get back the value assigned.


PUBLIC initialise_operating_system_interface:
---------------------------------------------

To be called once at the start of a run, before any I/O is done, in
order to initialise whatever the operating system needs doing.

Here, it just sets the number of characters allowed per line of output
to 78 (the default is 70). Elsewhere, it might re-size windows, reset
the terminal, etc.
*/


/*
IMPLEMENTATION
--------------

All these are highly system-dependent, and may have to be changed to run on
your system.

open_scratch_output creates a file called SCRATCH. and returns its full
name. This should work on any system. If you can though, see whether
your system allows you to create "invisible" scratch files with names
that can't clash with any other files'.

command_line_parameter_is calls the Pop-11 routine
command_line_parameter_is in SYSTEM.P. This in turn calls the built-in
routine systranslate (see HELP SYSTRANSLATE) to get back the value of
the logical name P1, which is set up by the command file that invokes
the Tutor.

initialise_operating_system_interface does whatever initialistion is
necessary, and is assumed to be called once at the beginning (from
"go"). Here, it sets the default output line width to 78 rather than 70.
On your system, it may just be a no-op.

*/


:- needs exists_file/2, open_output/2.


/*  Pop-11 bits are in SYSTEM.P  */
:- pop_compile( 'system.p' ).


open_scratch_output( FullName, Status ) :-
    open_output( 'scratch.', Status ),
    (
        Status = ok
    ->
        exists_file( 'scratch.', FullName )
    ;
        true
    ).


command_line_parameter_is( Param ) :-
    prolog_eval(apply(valof(command_line_parameter_is)), Param).


initialise_operating_system_interface :-
    prolog_eval( apply(valof(initialise_operating_system_interface)) ).


:- endmodule.
