/*  CHECK_CLAUSE.PL  */


:- module check_clause.


:- public check_clause/2,
          check_goal/2.


/*
SPECIFICATION
-------------

PUBLIC check_clause( Clause+, Error- ):

This is called to check facts that, being syntactically correct, have
been translated into Prolog. There might still be semantic errors, or
conditions that aren't errors in Prolog, but that indicate a likely
student error (we know that students are unlikely to use certain
constructions). If the predicate fails, there are no errors detected.
Otherwise, Error is unified with an error code.                 

The three errors currently checked for are:
(1) Redefining 'is'.
(2) Redefining ','.
(3) Redefining a system predicate.

These give rise to errors
    is_as_predicate_in_fact
    comma_as_predicate_in_fact(Fact)
    system_predicate_in_fact/[Pred,Arity]

(1) is very common, because 'is' is such a tempting word to use in
classification facts. (2) is also quite common, either because the
student typed a dot instead of a question mark, or because they are
trying to assert two things at once.


PUBLIC check_goal( Clause+, Error- ):

As above, but for questions. In the present version it's a dummy, and
always fails.
*/


/*
IMPLEMENTATION
--------------

'check_clause' checks for redefinition of system predicates, and 'is' in
particular. It calls 'is_system_pred' from USEFUL for this.

I haven't thought of any worthwhile checks for goals, so 'check_goal'
below is just a dummy.
*/


:- needs
    clause_vs_pred_arity / 3,
    is_system_pred / 2.


check_clause( Clause, is_as_predicate_in_fact ) :-
    clause_vs_pred_arity( Clause, is, _ ),
    !.

check_clause( Clause, comma_as_predicate_in_fact(Clause) ) :-
    clause_vs_pred_arity( Clause, ',', _ ),
    !.

check_clause( Clause, Error ) :-
    clause_vs_pred_arity( Clause, Pred, Arity ),
    is_system_pred( Pred, Arity ),
    Error = system_predicate_in_fact/[Pred,Arity].
    /*  Poplog bug: if Error is instantiated in the head of this
        predicate, it get bound even if the call fails.
        I believe "Bletchhhhhh" is the usual response to this.
    */


check_goal( _, _ ) :-
    fail.


:- endmodule.
