/*
    string.d -- String routines.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

object Kstart1;
object Kend1;
object Kstart2;
object Kend2;
object Kinitial_element;
object Kstart;
object Kend;

object
alloc_simple_string(int l)
{
	object x;

	x = alloc_object(t_string);
	x->st.st_hasfillp = FALSE;
	x->st.st_adjustable = FALSE;
	x->st.st_displaced = Cnil;
	x->st.st_dim = (x->st.st_fillp = l) + 1;
	x->st.st_self = NULL;
	return(x);
}

/*
	Make_simple_string(s) makes a simple string from C string s.
*/
object
make_simple_string(char *s)
{
	object x;

	x = alloc_simple_string(strlen(s));
	x->st.st_self = s;
	
	return(x);
}

/*
	This correponds to string= (just the string equality).
*/
bool
string_eq(object x, object y)
{
	int i, j;

	i = x->st.st_fillp;
	j = y->st.st_fillp;
	return ( i == j && strncmp(x->st.st_self, y->st.st_self, i) == 0);
}



/*
	This corresponds to string-equal
	(string equality ignoring the case).
*/
bool
string_equal(object x, object y)
{
	int i, j;
	char *p, *q;

	i = x->st.st_fillp;
	j = y->st.st_fillp;
	if (i != j)
		return(FALSE);
	p = x->st.st_self;
	q = y->st.st_self;
	for (i = 0;  i < j;  i++)
		if ((isLower(p[i]) ? p[i] - ('a' - 'A') : p[i])
		 != (isLower(q[i]) ? q[i] - ('a' - 'A') : q[i]))
			return(FALSE);
	return(TRUE);
}

/*
	Copy_simple_string(x) copies string x to a simple string.
*/
object
copy_simple_string(object x)
{
	object y;

	y = alloc_simple_string(x->st.st_fillp);
	y->st.st_self = alloc_contblock(x->st.st_fillp+1);
	memcpy(y->st.st_self, x->st.st_self, x->st.st_fillp+1);
	return(y);
}

object
coerce_to_string(object x)
{
	object y;
	int i;

	switch (type_of(x)) {
	case t_symbol:
		y = alloc_simple_string(x->s.s_fillp);
		if (x->s.s_self < heap_end)
			y->st.st_self = x->s.s_self;
		else {
			y->st.st_self = alloc_contblock(x->s.s_fillp+1);
			memcpy(y->s.s_self, x->st.st_self, x->s.s_fillp+1);
		}
		return(y);

	case t_fixnum:
		x = coerce_to_character(x);
		/* continue the execution of the switch */
	case t_character:
		y = alloc_simple_string(1);
		y->st.st_self = alloc_contblock(2);
		y->st.st_self[1] = '\0';
		y->st.st_self[0] = char_code(x);
		return(y);

	case t_string:
		return(x);
	}
	FEerror("~S cannot be coerced to a string.", 1, x);
}

@(defun char (s i)
	int j;
@
	check_type_string(&s);
	if (!FIXNUMP(i))
		illegal_index(s, i);
	if ((j = fix(i)) < 0 || j >= s->st.st_fillp)
		illegal_index(s, i);
	@(return `code_char(s->ust.ust_self[j])`)
@)

siLchar_set(int narg, object str, object index, object c)
{
	int j;

	check_arg(3);
	check_type_string(&str);
	if (!FIXNUMP(index))
		illegal_index(str, index);
	if ((j = fix(index)) < 0 || j >= str->st.st_fillp)
		illegal_index(str, index);
	check_type_character(&c);
	str->st.st_self[j] = char_code(c);
	VALUES(0) = c;
	RETURN(1);
}

get_string_start_end(object string, object start, object end,
	int *ps, int *pe)
{
	if (Null(start))
		*ps = 0;
	else if (!FIXNUMP(start))
		goto E;
	else {
		*ps = fix(start);
		if (*ps < 0)
			goto E;
	}
	if (Null(end)) {
		*pe = string->st.st_fillp;
		if (*pe < *ps)
			goto E;
	} else if (!FIXNUMP(end))
		goto E;
	else {
		*pe = fix(end);
		if (*pe < *ps || *pe > string->st.st_fillp)
			goto E;
	}
	return;

E:
	FEerror("~S and ~S are illegal as :START and :END~%\
for the string ~S.", 3, start, end, string);
}

@(defun string_eq (string1 string2
		   &key start1 end1 start2 end2)
	int s1, e1, s2, e2;
@
	string1 = coerce_to_string(string1);
	string2 = coerce_to_string(string2);
	get_string_start_end(string1, start1, end1, &s1, &e1);
	get_string_start_end(string2, start2, end2, &s2, &e2);
	if (e1 - s1 != e2 - s2)
		@(return Cnil)
	while (s1 < e1)
		if (string1->st.st_self[s1++] !=
		    string2->st.st_self[s2++])
			@(return Cnil)
	@(return Ct)
@)

@(defun string_equal (string1 string2
		      &key start1 end1 start2 end2)
	int s1, e1, s2, e2;
	int i1, i2;
@
	string1 = coerce_to_string(string1);
	string2 = coerce_to_string(string2);
	get_string_start_end(string1, start1, end1, &s1, &e1);
	get_string_start_end(string2, start2, end2, &s2, &e2);
	if (e1 - s1 != e2 - s2)
		@(return Cnil)
	while (s1 < e1) {
		i1 = string1->st.st_self[s1++];
		i2 = string2->st.st_self[s2++];
		if (isLower(i1))
			i1 -= 'a' - 'A';
		if (isLower(i2))
			i2 -= 'a' - 'A';
		if (i1 != i2)
			@(return Cnil)
	}
	@(return Ct)
@)


Lstring_cmp(int narg, int sign, int boundary, object *ARGS)
{
	object string1 = ARGS[0], string2 = ARGS[1];
	int s1, e1, s2, e2, i1, i2, s;
	object KEYS[4];
#define start1 KEY_VARS[0]
#define end1 KEY_VARS[1]
#define start2 KEY_VARS[2]
#define end2 KEY_VARS[3]
	object KEY_VARS[8];

	if (narg < 2) FEtoo_few_arguments(&narg);
	KEYS[0]=Kstart1;
	KEYS[1]=Kend1;
	KEYS[2]=Kstart2;
	KEYS[3]=Kend2;
	parse_key(narg-2, ARGS+2, 4, KEYS, KEY_VARS, OBJNULL, 0);

	string1 = coerce_to_string(string1);
	string2 = coerce_to_string(string2);
	get_string_start_end(string1, start1, end1, &s1, &e1);
	get_string_start_end(string2, start2, end2, &s2, &e2);
	while (s1 < e1) {
		if (s2 == e2) {
		  VALUES(0) = sign>0 ? Cnil : MAKE_FIXNUM(s1);
		  RETURN(1);
		}
		i1 = string1->ust.ust_self[s1];
		i2 = string2->ust.ust_self[s2];
		if (sign == 0) {
			if (i1 != i2) {
			  VALUES(0) = MAKE_FIXNUM(s1);
			  RETURN(1);
			}
		} else {
			s = sign*(i2-i1);
			if (s > 0) {
			  VALUES(0) = MAKE_FIXNUM(s1);
			  RETURN(1);
			}
			if (s < 0) {
			  VALUES(0) = Cnil;
			  RETURN(1);
			}
		}
		s1++;
		s2++;
	}
	if (s2 == e2)
	  VALUES(0) = boundary==0 ? MAKE_FIXNUM(s1) : Cnil;
	else
	  VALUES(0) = sign>=0 ? MAKE_FIXNUM(s1) : Cnil;
	RETURN(1);
#undef start1
#undef end1
#undef start2
#undef end2
      }

Lstring_l(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_cmp(narg, 1, 1, (object *)args)); }
Lstring_g(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_cmp(narg,-1, 1, (object *)args)); }
Lstring_le(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_cmp(narg, 1, 0, (object *)args)); }
Lstring_ge(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_cmp(narg,-1, 0, (object *)args)); }
Lstring_neq(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_cmp(narg, 0, 1, (object *)args)); }


Lstring_compare(int narg, int sign, int boundary, object *ARGS)
{
	object string1 = ARGS[0], string2 = ARGS[1];
	int s1, e1, s2, e2, i1, i2, s;

	object KEYS[4];
#define start1 KEY_VARS[0]
#define end1 KEY_VARS[1]
#define start2 KEY_VARS[2]
#define end2 KEY_VARS[3]
	object KEY_VARS[8];

	if (narg < 2) FEtoo_few_arguments(&narg);
	KEYS[0]=Kstart1;
	KEYS[1]=Kend1;
	KEYS[2]=Kstart2;
	KEYS[3]=Kend2;
	parse_key(narg-2, ARGS+2, 4, KEYS, KEY_VARS, OBJNULL, 0);

	string1 = coerce_to_string(string1);
	string2 = coerce_to_string(string2);
	get_string_start_end(string1, start1, end1, &s1, &e1);
	get_string_start_end(string2, start2, end2, &s2, &e2);
	while (s1 < e1) {
		if (s2 == e2) {
		  VALUES(0) = sign>0 ? Cnil : MAKE_FIXNUM(s1);
		  RETURN(1);
		}
		i1 = string1->ust.ust_self[s1];
		if (isLower(i1))
			i1 -= 'a' - 'A';
		i2 = string2->ust.ust_self[s2];
		if (isLower(i2))
			i2 -= 'a' - 'A';
		if (sign == 0) {
			if (i1 != i2) {
			  VALUES(0) = MAKE_FIXNUM(s1);
			  RETURN(1);
			}
		} else {
			s = sign*(i2-i1);
			if (s > 0) {
			  VALUES(0) = MAKE_FIXNUM(s1);
			  RETURN(1);
			}
			if (s < 0) {
			  VALUES(0) = Cnil;
			  RETURN(1);
			}
		}
		s1++;
		s2++;
	}
	if (s2 == e2) {
		  VALUES(0) = boundary==0 ? MAKE_FIXNUM(s1) : Cnil;
		  RETURN(1);
		}
	VALUES(0) = sign>=0 ? MAKE_FIXNUM(s1) : Cnil;
	RETURN(1);
#undef start1
#undef end1
#undef start2
#undef end2
      }

Lstring_lessp(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_compare(narg, 1, 1, (object *)args)); }
Lstring_greaterp(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_compare(narg,-1, 1, (object *)args)); }
Lstring_not_greaterp(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_compare(narg, 1, 0, (object *)args)); }
Lstring_not_lessp(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_compare(narg,-1, 0, (object *)args)); }
Lstring_not_equal(int narg, ...)
{ va_list args; va_start(args, narg);
 RETURN(Lstring_compare(narg, 0, 1, (object *)args)); }


@(defun make_string (size &key (initial_element `code_char(' ')`)
		     &aux x)
	int i;
@
	while (!FIXNUMP(size) || fix(size) < 0)
		size
		= wrong_type_argument(TSnon_negative_integer, size);
		/*  bignum not allowed, this is PRACTICAL!!  */
	while (type_of(initial_element) != t_character ||
	       char_bits(initial_element) != 0 ||
	       char_font(initial_element) != 0)
		initial_element
		= wrong_type_argument(Sstring_char, initial_element);
	x = alloc_simple_string(fix(size));
	x->st.st_self = alloc_contblock(fix(size)+1);
	x->st.st_self[fix(size)] = '\0';
	for (i = 0;  i < fix(size);  i++)
		x->st.st_self[i] = char_code(initial_element);
	@(return x)
@)

bool
member_char(int c, object char_bag)
{
	int i, f;

	switch (type_of(char_bag)) {
	case t_symbol:
	case t_cons:
		while (!endp(char_bag)) {
			if (type_of(CAR(char_bag)) != t_character)
				continue;
			if (c == char_code(CAR(char_bag)))
				return(TRUE);
			char_bag = CDR(char_bag);
		}
		return(FALSE);

	case t_vector:
		for (i = 0, f = char_bag->v.v_fillp;  i < f;  i++) {
			if (type_of(char_bag->v.v_self[i]) != t_character)
				continue;
			if (c == char_code(char_bag->v.v_self[i]))
				return(TRUE);
		}
		return(FALSE);

	case t_string:
		for (i = 0, f = char_bag->st.st_fillp;  i < f;  i++) {
			if (c == char_bag->st.st_self[i])
				return(TRUE);
		}
		return(FALSE);

	case t_bitvector:
		return(FALSE);

	default:
		FEerror("~S is not a sequence.", 1, char_bag);
	}
}

Lstring_trim(int narg, object char_bag, object strng)
	{ RETURN(Lstring_trim0(narg, TRUE, TRUE, char_bag, strng)); }
Lstring_left_trim(int narg, object char_bag, object strng)
	{ RETURN(Lstring_trim0(narg, TRUE, FALSE, char_bag, strng)); }
Lstring_right_trim(int narg, object char_bag, object strng)
	{ RETURN(Lstring_trim0(narg, FALSE, TRUE, char_bag, strng));}

@(defun `string_trim0(bool left_trim, bool right_trim)`
  (char_bag strng &aux res)
	int i, j, k;
@
	strng = coerce_to_string(strng);
	i = 0;
	j = strng->st.st_fillp - 1;
	if (left_trim)
		for (;  i <= j;  i++)
			if (!member_char(strng->st.st_self[i], char_bag))
				break;
	if (right_trim)
		for (;  j >= i;  --j)
			if (!member_char(strng->st.st_self[j], char_bag))
				break;
	k = j - i + 1;
	res = alloc_simple_string(k);
	res->st.st_self = alloc_contblock(k+1);
	res->st.st_self[k] = '\0';
	memcpy(res->st.st_self, strng->st.st_self+i, k);
	@(return res)
@)

static char_upcase(int c, int *bp)
{
	if (isLower(c))
		return(c - ('a' - 'A'));
	else
		return(c);
}

static char_downcase(int c, int *bp)
{
	if (isUpper(c))
		return(c + ('a' - 'A'));
	else
		return(c);
}

static char_capitalize(int c, int *bp)
{
	if (isLower(c)) {
		if (*bp)
			c -= 'a' - 'A';
		*bp = FALSE;
	} else if (isUpper(c)) {
		if (!*bp)
			c += 'a' - 'A';
		*bp = FALSE;
	} else if (!isDigit(c))
		*bp = TRUE;
	return(c);
}

Lstring_case(int narg, int (*casefun)(), object *ARGS)
{
	object strng = ARGS[0];
	int s, e, i;
	bool b;
	object KEYS[2];
#define start KEY_VARS[0]
#define end KEY_VARS[1]
	object conv;
	object KEY_VARS[4];

	if (narg < 1) FEtoo_few_arguments(&narg);
	KEYS[0]=Kstart;
	KEYS[1]=Kend;
	parse_key(narg-1, ARGS+1, 2, KEYS, KEY_VARS, OBJNULL, 0);
	conv = Cnil;

	strng = coerce_to_string(strng);
	get_string_start_end(strng, start, end, &s, &e);
	conv = copy_simple_string(strng);
	b = TRUE;
	for (i = s;  i < e;  i++)
		conv->st.st_self[i] = (*casefun)(conv->st.st_self[i], &b);
	VALUES(0) = conv;
	RETURN(1);
#undef start
#undef end
}

Lstring_upcase(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_case(narg, char_upcase, (object *)args)); }
Lstring_downcase(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_case(narg, char_downcase, (object *)args)); }
Lstring_capitalize(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lstring_case(narg, char_capitalize, (object *)args)); }


Lnstring_case(int narg, int (*casefun)(), object *ARGS)
{
	object strng = ARGS[0];
	int s, e, i;
	bool b;
	object KEYS[2];
#define start KEY_VARS[0]
#define end KEY_VARS[1]
	object KEY_VARS[4];

	if (narg < 1) FEtoo_few_arguments(&narg);
	KEYS[0]=Kstart;
	KEYS[1]=Kend;
	parse_key(narg-1, ARGS+1, 2, KEYS, KEY_VARS, OBJNULL, 0);

	check_type_string(&strng);
	get_string_start_end(strng, start, end, &s, &e);
	b = TRUE;
	for (i = s;  i < e;  i++)
		strng->st.st_self[i] = (*casefun)(strng->st.st_self[i], &b);
	VALUES(0) = strng;
	RETURN(1);
#undef start
#undef end
}

Lnstring_upcase(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lnstring_case(narg, char_upcase, (object *)args)); }
Lnstring_downcase(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lnstring_case(narg, char_downcase, (object *)args)); }
Lnstring_capitalize(int narg, ...)
{ va_list args; va_start(args, narg);
  RETURN(Lnstring_case(narg, char_capitalize, (object *)args)); }


@(defun string (x)
@
	@(return `coerce_to_string(x)`)
@)

siLstring_concatenate(int narg, ...)
{
	int i, l;
	object v, *strings;
	va_list args;
	char *vself;

	va_start(args, narg);
	strings = (object *)args;
	for (i = 0, l = 0;  i < narg;  i++) {
		strings[i] = coerce_to_string(va_arg(args, object));
		l += strings[i]->st.st_fillp;
	}
	v = alloc_simple_string(l);
	v->st.st_self = alloc_contblock(l+1);
	v->st.st_self[l] = '\0';
	for (i = 0, vself = v->st.st_self;  i < narg;  i++, vself += l) {
		l = strings[i]->st.st_fillp;
		memcpy(vself, strings[i]->st.st_self, l);
		}
	VALUES(0) = v;
	RETURN(1);
}

init_string_function()
{
	Kstart1 = make_keyword("START1");
	Kend1 = make_keyword("END1");
	Kstart2 = make_keyword("START2");
	Kend2 = make_keyword("END2");
	Kinitial_element = make_keyword("INITIAL-ELEMENT");
	Kstart = make_keyword("START");
	Kend = make_keyword("END");

	make_function("CHAR", Lchar);
	make_si_function("CHAR-SET", siLchar_set);
	make_function("SCHAR", Lchar);
	make_si_function("SCHAR-SET", siLchar_set);
	make_function("STRING=", Lstring_eq);
	make_function("STRING-EQUAL", Lstring_equal);
	make_function("STRING<", Lstring_l);
	make_function("STRING>", Lstring_g);
	make_function("STRING<=", Lstring_le);
	make_function("STRING>=", Lstring_ge);
	make_function("STRING/=", Lstring_neq);
	make_function("STRING-LESSP", Lstring_lessp);
	make_function("STRING-GREATERP", Lstring_greaterp);
	make_function("STRING-NOT-LESSP", Lstring_not_lessp);
	make_function("STRING-NOT-GREATERP", Lstring_not_greaterp);
	make_function("STRING-NOT-EQUAL", Lstring_not_equal);
	make_function("MAKE-STRING", Lmake_string);
	make_function("STRING-TRIM", Lstring_trim);
	make_function("STRING-LEFT-TRIM", Lstring_left_trim);
	make_function("STRING-RIGHT-TRIM", Lstring_right_trim);
	make_function("STRING-UPCASE", Lstring_upcase);
	make_function("STRING-DOWNCASE", Lstring_downcase);
	make_function("STRING-CAPITALIZE", Lstring_capitalize);
	make_function("NSTRING-UPCASE", Lnstring_upcase);
	make_function("NSTRING-DOWNCASE", Lnstring_downcase);
	make_function("NSTRING-CAPITALIZE", Lnstring_capitalize);
	make_function("STRING", Lstring);

	make_si_function("STRING-CONCATENATE",
			 siLstring_concatenate);
}
