/*
    num_sfun.c  -- Trascendental functions.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

#include "config.h"
#include "num_include.h"

object imag_unit, minus_imag_unit, imag_two;

int
fixnum_expt(int x, int y)
{
	int z;

	z = 1;
	while (y > 0)
		if (y%2 == 0) {
			x *= x;
			y /= 2;
		} else {
			z *= x;
			--y;
		}
	return(z);
}

object
number_exp(object x)
{
	double exp();

	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
	case t_ratio:
		return(make_shortfloat((shortfloat)exp(number_to_double(x))));

	case t_shortfloat:
		return(make_shortfloat((shortfloat)exp((double)(sf(x)))));

	case t_longfloat:
		return(make_longfloat(exp(lf(x))));

	case t_complex:
	{
		object y, y1;
		object number_sin(), number_cos();
	
		y = x->cmp.cmp_imag;
		x = x->cmp.cmp_real;
		x = number_exp(x);
		y1 = number_cos(y);
		y = number_sin(y);
		y = make_complex(y1, y);
		x = number_times(x, y);
		return(x);
	}

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
number_expt(object x, object y)
{
	enum type tx, ty;
	object z, number_nlog();

	tx = type_of(x);
	ty = type_of(y);
	if (ty == t_fixnum && fix(y) == 0)
		switch (tx) {
		case t_fixnum:  case t_bignum:  case t_ratio:
			return(MAKE_FIXNUM(1));

		case t_shortfloat:
			return(make_shortfloat((shortfloat)1.0));

		case t_longfloat:
			return(make_longfloat(1.0));

		case t_complex:
			z = number_expt(x->cmp.cmp_real, y);
			z = make_complex(z, MAKE_FIXNUM(0));
			return(z);

		default:
			FEwrong_type_argument(Snumber, x);
		}
	if (number_zerop(x)) {
		if (!number_plusp(ty==t_complex?y->cmp.cmp_real:y))
			FEerror("Cannot raise zero to the power ~S.", 1, y);
		return(number_times(x, y));
	}
	if (ty == t_fixnum || ty == t_bignum) {
		if (number_minusp(y)) {
			z = number_negate(y);
			z = number_expt(x, z);
			z = number_divide(MAKE_FIXNUM(1), z);
			return(z);
		}
		z = MAKE_FIXNUM(1);
		while (number_plusp(y))
			if (number_evenp(y)) {
				x = number_times(x, x);
				y = integer_divide1(y, MAKE_FIXNUM(2));
			} else {
				z = number_times(z, x);
				y = number_minus(y, MAKE_FIXNUM(1));
			}
		return(z);
	}
	z = number_nlog(x);
	z = number_times(z, y);
	z = number_exp(z);
	return(z);
}

object
number_nlog(object x)
{
	double log();
	object r, i, a, p, number_sqrt(), number_atan2();

	if (type_of(x) == t_complex) {
		r = x->cmp.cmp_real;
		i = x->cmp.cmp_imag;
		goto COMPLEX;
	}
	if (number_zerop(x))
		FEerror("Zero is the logarithmic singularity.", 0);
	if (number_minusp(x)) {
		r = x;
		i = MAKE_FIXNUM(0);
		goto COMPLEX;
	}
	switch (type_of(x)) {
	case t_fixnum:
	case t_bignum:
	case t_ratio:
		return(make_shortfloat((shortfloat)log(number_to_double(x))));

	case t_shortfloat:
		return(make_shortfloat((shortfloat)log((double)(sf(x)))));

	case t_longfloat:
		return(make_longfloat(log(lf(x))));

	default:
		FEwrong_type_argument(Snumber, x);
	}

COMPLEX:
	a = number_times(r, r);
	p = number_times(i, i);
	a = number_plus(a, p);
	a = number_nlog(a);
	a = number_divide(a, MAKE_FIXNUM(2));
	p = number_atan2(i, r);
	x = make_complex(a, p);
	return(x);
}

object
number_log(object x, object y)
{
	if (number_zerop(y))
		FEerror("Zero is the logarithmic singularity.", 0);
	return(number_divide(number_nlog(y), number_nlog(x)));
}

object
number_sqrt(object x)
{
	object z;
	double sqrt();

	if (type_of(x) == t_complex)
		goto COMPLEX;
	if (number_minusp(x))
		goto COMPLEX;
	switch (type_of(x)) {
	case t_fixnum:
	case t_bignum:
	case t_ratio:
		return(make_shortfloat(
			(shortfloat)sqrt(number_to_double(x))));

	case t_shortfloat:
		return(make_shortfloat((shortfloat)sqrt((double)(sf(x)))));

	case t_longfloat:
		return(make_longfloat(sqrt(lf(x))));

	default:
		FEwrong_type_argument(Snumber, x);
	}

COMPLEX:
	z = make_ratio(MAKE_FIXNUM(1), MAKE_FIXNUM(2));
	z = number_expt(x, z);
	return(z);
}

object
number_atan2(object y, object x)
{
	object z;
	double atan(), dy, dx, dz;

	dy = number_to_double(y);
	dx = number_to_double(x);
	if (dx > 0.0)
		if (dy > 0.0)
			dz = atan(dy / dx);
		else if (dy == 0.0)
			dz = 0.0;
		else
			dz = -atan(-dy / dx);
	else if (dx == 0.0)
		if (dy > 0.0)
			dz = PI / 2.0;
		else if (dy == 0.0)
			FEerror("Logarithmic singularity.", 0);
		else
			dz = -PI / 2.0;
	else
		if (dy > 0.0)
			dz = PI - atan(dy / -dx);
		else if (dy == 0.0)
			dz = PI;
		else
			dz = -PI + atan(-dy / -dx);
	if (type_of(x) == t_longfloat || type_of(y) == t_longfloat)
		z = make_longfloat(dz);
	else
		z = make_shortfloat((shortfloat)dz);
	return(z);
}

object
number_atan(object y)
{
	object z, z1;

	if (type_of(y) == t_complex) {
#ifdef ANSI
		z = number_times(imag_unit, y);
		z = number_nlog(one_plus(z)) +
		  number_nlog(number_minus(MAKE_FIXNUM(1), z));
		z = number_divide(z, number_times(MAKE_FIXNUM(2), imag_unit));
#else
		z = number_times(imag_unit, y);
		z = one_plus(z);
		z1 = number_times(y, y);
		z1 = one_plus(z1);
		z1 = number_sqrt(z1);
		z = number_divide(z, z1);
		z = number_nlog(z);
		z = number_times(minus_imag_unit, z);
#endif ANSI
		return(z);
	}
	return(number_atan2(y, MAKE_FIXNUM(1)));
}

object
number_sin(object x)
{
	double sin();

	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
	case t_ratio:
		return(make_shortfloat((shortfloat)sin(number_to_double(x))));

	case t_shortfloat:
		return(make_shortfloat((shortfloat)sin((double)(sf(x)))));

	case t_longfloat:
		return(make_longfloat(sin(lf(x))));

	case t_complex:
	{
		object	r;
		object	x0, x1, x2;

		x0 = number_times(imag_unit, x);
		x0 = number_exp(x0);
		x1 = number_times(minus_imag_unit, x);
		x1 = number_exp(x1);
		x2 = number_minus(x0, x1);
		r = number_divide(x2, imag_two);

		return(r);
	}

	default:
		FEwrong_type_argument(Snumber, x);

	}
}

object
number_cos(object x)
{
	double cos();

	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
	case t_ratio:
		return(make_shortfloat((shortfloat)cos(number_to_double(x))));

	case t_shortfloat:
		return(make_shortfloat((shortfloat)cos((double)(sf(x)))));

	case t_longfloat:
		return(make_longfloat(cos(lf(x))));

	case t_complex:
	{
		object r;
		object x0, x1, x2;

		x0 = number_times(imag_unit, x);
		x0 = number_exp(x0);
		x1 = number_times(minus_imag_unit, x);
		x1 = number_exp(x1);
		x2 = number_plus(x0, x1);
		r = number_divide(x2, MAKE_FIXNUM(2));

		return(r);
	}

	default:
		FEwrong_type_argument(Snumber, x);

	}
}

object
number_tan(object x)
{
	object r, s, c;

	s = number_sin(x);
	c = number_cos(x);
	if (number_zerop(c) == TRUE)
		FEerror("Cannot compute the tangent of ~S.", 1, x);
	r = number_divide(s, c);
	return(r);
}

Lexp(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = number_exp(x);
	RETURN(1);
}

Lexpt(int narg, object x, object y)
{
	check_arg(2);
	check_type_number(&x);
	check_type_number(&y);
	VALUES(0) = number_expt(x, y);
	RETURN(1);
}

Llog(int narg, object x, object y)
{
	if (narg < 1)
		FEtoo_few_arguments(&narg);
	else if (narg == 1) {
		check_type_number(&x);
		VALUES(0) = number_nlog(x);
	} else if (narg == 2) {
		check_type_number(&x);
		check_type_number(&y);
		VALUES(0) = number_log(y, x);
	} else
		FEtoo_many_arguments(&narg);
	RETURN(1);
}

Lsqrt(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = number_sqrt(x);
	RETURN(1);
}

Lsin(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = number_sin(x);
	RETURN(1);
}

Lcos(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = number_cos(x);
	RETURN(1);
}

Ltan(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = number_tan(x);
	RETURN(1);
}

Latan(int narg, object x, object y)
{
	if (narg < 1)
		FEtoo_few_arguments(&narg);
	if (narg == 1) {
		check_type_number(&x);
		VALUES(0) = number_atan(x);
	} else if (narg == 2) {
		check_type_or_rational_float(&x);
		check_type_or_rational_float(&y);
		VALUES(0) = number_atan2(x, y);
	} else
		FEtoo_many_arguments(&narg);
	RETURN(1);
}

init_num_sfun()
{
	imag_unit
	= make_complex(make_shortfloat((shortfloat)0.0),
		       make_shortfloat((shortfloat)1.0));
	enter_mark_origin(&imag_unit);
	minus_imag_unit
	= make_complex(make_shortfloat((shortfloat)0.0),
		       make_shortfloat((shortfloat)-1.0));
	enter_mark_origin(&minus_imag_unit);
	imag_two
	= make_complex(make_shortfloat((shortfloat)0.0),
		       make_shortfloat((shortfloat)2.0));
	enter_mark_origin(&imag_two);

	make_constant("PI", make_longfloat(PI));

	make_function("EXP", Lexp);
	make_function("EXPT", Lexpt);
	make_function("LOG", Llog);
	make_function("SQRT", Lsqrt);
	make_function("SIN", Lsin);
	make_function("COS", Lcos);
	make_function("TAN", Ltan);
	make_function("ATAN", Latan);
}
