/*
    num_comp.c  -- Comparisons on numbers.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

#include "config.h"
#include "num_include.h"

/*
	The value of number_compare(x, y) is

		-1	if	x < y
		0	if	x = y
		1	if	x > y.

	If x or y is complex, 0 or 1 is returned.
*/
int
number_compare(object x, object y)
{
	int ix, iy;
	double dx, dy;

	switch (type_of(x)) {

	case t_fixnum:
	  	ix = fix(x);
		switch (type_of(y)) {
		case t_fixnum:
		  	iy = fix(y);
			if (ix < iy)
				return(-1);
			else if (ix == iy)
				return(0);
			else
				return(1);
		case t_bignum:
			if (big_sign((struct bignum *)y) < 0)
				return(1);
			else
				return(-1);
		case t_ratio:
			x = number_times(x, y->rat.rat_den);
			y = y->rat.rat_num;
			return(number_compare(x, y));
		case t_shortfloat:
			dx = (double)(ix);
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dx = (double)(ix);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto Y_COMPLEX;
		default:
			wrong_type_argument(Snumber, y);
		}

	case t_bignum:
		switch (type_of(y)) {
		case t_fixnum:
			if (big_sign((struct bignum *)x) < 0)
				return(-1);
			else
				return(1);
		case t_bignum:
			return(big_compare((struct bignum *)x,
					   (struct bignum *)y));
		case t_ratio:
			x = number_times(x, y->rat.rat_den);
			y = y->rat.rat_num;
			return(number_compare(x, y));
		case t_shortfloat:
			dx = number_to_double(x);
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto Y_COMPLEX;
		default:
			wrong_type_argument(Snumber, y);
		}

	case t_ratio:
		switch (type_of(y)) {
		case t_fixnum:
		case t_bignum:
			y = number_times(y, x->rat.rat_den);
			x = x->rat.rat_num;
			return(number_compare(x, y));
		case t_ratio:
			return(number_compare(number_times(x->rat.rat_num,
							   y->rat.rat_den),
					      number_times(y->rat.rat_num,
							   x->rat.rat_den)));
		case t_shortfloat:
			dx = number_to_double(x);
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto Y_COMPLEX;
		default:
			wrong_type_argument(Snumber, y);
		}

	case t_shortfloat:
		dx = (double)(sf(x));
		goto LONGFLOAT0;

	case t_longfloat:
		dx = lf(x);
	LONGFLOAT0:
		switch (type_of(y)) {
		case t_fixnum:
			dy = (double)(fix(y));
			goto LONGFLOAT;
		case t_bignum:
		case t_ratio:
			dy = number_to_double(y);
			goto LONGFLOAT;
		case t_shortfloat:
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto Y_COMPLEX;
		}
	LONGFLOAT:
		if (dx == dy)
			return(0);
		else if (dx < dy)
			return(-1);
		else
			return(1);

	Y_COMPLEX:
		if (number_zerop(y->cmp.cmp_imag))
			if (number_compare(x, y->cmp.cmp_real) == 0)
				return(0);
			else
				return(1);
		else
			return(1);

	case t_complex:
		if (type_of(y) != t_complex)
			if (number_zerop(x->cmp.cmp_imag))
				if (number_compare(x->cmp.cmp_real, y) == 0)
					return(0);
				else
					return(1);
			else
				return(1);
		if (number_compare(x->cmp.cmp_real, y->cmp.cmp_real) == 0 &&
		    number_compare(x->cmp.cmp_imag, y->cmp.cmp_imag) == 0 )
			return(0);
		else
			return(1);

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

Lall_the_same(int narg, object num, ...)
{
	int i;
	va_list nums; object numi;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	check_type_number(&num);
	va_start(nums, num);
	for (i = 1; i < narg; i++) {
	  numi = va_arg(nums, object);
	  check_type_number(&numi);
	  if (number_compare(num, numi) != 0) {
	    VALUES(0) = Cnil;
	    RETURN(1);
	  }
	}
	VALUES(0) = Ct;
	RETURN(1);
}

Lall_different(int narg, ...)
{
	int i, j;
	va_list nums; object numi;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	else if (narg == 1) {
		VALUES(0) = Ct;
		RETURN(1);
	}
	va_start(nums, narg);
	for (i = 0; i < narg; i++, va_arg(nums, object)) {
	  numi = va_arg(nums, object);
	  check_type_number(&numi);
	  va_start(nums, narg);
	  for (j = 0; j < i; j++)
	    if (number_compare(numi, va_arg(nums, object)) == 0) {
	      VALUES(0) = Cnil;
	      RETURN(1);
	    }
	}
	VALUES(0) = Ct;
	RETURN(1);
}

monotonic(int s, int t, int narg, object *nums)
{
	int i;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	for (i = 0; i < narg; i++)
		check_type_or_rational_float(&nums[i]);
	for (i = 1; i < narg; i++)
		if (s*number_compare(nums[i], nums[i-1]) < t) {
			VALUES(0) = Cnil;
			RETURN(1);
		}
	VALUES(0) = Ct;
	RETURN(1);
}

#define MONOTONIC(i, j) (int narg, ...) \
{ va_list nums; va_start(nums, narg); \
  RETURN(monotonic(i, j, narg, (object *)nums)); }

Lmonotonically_increasing	MONOTONIC( 1, 1)
Lmonotonically_decreasing	MONOTONIC(-1, 1)
Lmonotonically_nondecreasing	MONOTONIC( 1, 0)
Lmonotonically_nonincreasing	MONOTONIC(-1, 0)

Lmax(int narg, object max, ...)
{
	object numi; va_list nums;
	int i;
	
	if (narg == 0)
		FEtoo_few_arguments(&narg);
	check_type_or_rational_float(&max);
	va_start(nums, max);
	for (i = 1;  i < narg;  i++) {
	  numi = va_arg(nums, object);
	  check_type_or_rational_float(&numi);
	  if (number_compare(max, numi) < 0)
			max = numi;
	}
	VALUES(0) = max;
	RETURN(1);
}

Lmin(int narg, object min, ...)
{
	object numi; va_list nums;
	int i;
	
	if (narg == 0)
		FEtoo_few_arguments(&narg);
	check_type_or_rational_float(&min);
	va_start(nums, min);
	for (i = 1;  i < narg;  i++) {
	  numi = va_arg(nums, object);
	  check_type_or_rational_float(&numi);
	  if (number_compare(min, numi) > 0)
			min = numi;
	}
	VALUES(0) = min;
	RETURN(1);
}

init_num_comp()
{
	make_function("=", Lall_the_same);
	make_function("/=", Lall_different);
	make_function("<", Lmonotonically_increasing);
	make_function(">", Lmonotonically_decreasing);
	make_function("<=", Lmonotonically_nondecreasing);
	make_function(">=", Lmonotonically_nonincreasing);
	make_function("MAX", Lmax);
	make_function("MIN", Lmin);
}
