/*
    lwp.c -- Light weight processes.
*/
/*
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

lpd main_lpd;
lpd *clwp = &main_lpd;

int critical_level = 0;
int critical_semaphore = FREE;
bool was_interrupted = FALSE;
bool timer_state = INACTIVE;

pd *running_head;		/* front of running pd's  */
pd *running_tail;		/* back of running pd's   */
pd main_pd;

/******************************* IMPORTS ******************************/

extern int writec_PRINTstream();
extern object readc();
extern GC();
extern enum type garbage_parameter;

/******************************* ------- ******************************/


#define BREAK 7
#define DEACTIVATE 8
#define call_scheduler(entry)  intha(entry, 0, 0)

static bool scheduler_disabled = FALSE;
static int scheduler_level = 0;            /* tito */
static int real_scheduler_level = 0;       /* tito */
static bool reset_timer = FALSE;

static running_processes = 1;

object Srunning;
object Ssuspended;
object Swaiting;
object Sstopped;
object Sdead;
object siSthread_top_level;

/* static pd *new_pd;  */
static object main_thread;

static int val;
static int val1;

static int absolute_time = 0;

#ifdef BSD
virtual_ualarm(time)
 int time;
{
 struct itimerval virt_oldtimer;
 struct itimerval virt_itimer;
 virt_itimer.it_value.tv_sec = 0;
 virt_itimer.it_value.tv_usec = time;
 virt_itimer.it_interval.tv_sec = 0;
 virt_itimer.it_interval.tv_usec = 0;
 setitimer(ITIMER_REAL, &virt_itimer, &virt_oldtimer);
}
#else SYSV

#define VIRT_SEC 600000

void virt_tick()
{
 int i = 0;
 while (i < VIRT_SEC)
   i++;
}

virtual_ualarm(int time)
{ static int pid, childpid;
  if (time == 0)
    kill(childpid, 9);
  else {
    pid = getpid ();
    childpid = fork ();
    switch (childpid)
      {
      case -1:
	perror("Couldn't set timer due to");
	break;
      case 0:			/* virtual timer */
	virt_tick();
	kill (pid,SIGALRM);
	exit ();
      default:
	break;
      }
  }
}
#endif BSD

pd *dequeue()
{
  pd *tmp;
  tmp = running_head;
  if  (running_head != NULL)
    running_head = running_head->pd_next;
  return tmp;
}


pd *make_pd()
{
  pd *new_pd; lpd *npd;

#ifdef DEBUG2
  printf("entro nella make_pd\n");
  fflush(stdout);
#endif
  
  /* Allocate a new descriptor for the new lwp */
  new_pd = (pd *)malloc(sizeof(pd));
  
  /* create a new stack ... */
  new_pd->pd_base = (int *)malloc(STACK_SIZE*4); /* tito */

  new_pd->pd_status = SUSPENDED;

  /* allocate a lisp descriptor:
   * using the calloc here it's possible to avoid the
   * critical section in the various push operations
   */
  npd = new_pd->pd_lpd = (lpd *)calloc(sizeof(lpd), 1);

  /* initialize it */
  
				/* bind stack */
  npd->lwp_bds_top = npd->lwp_bind_stack - 1;
  npd->lwp_bds_limit = npd->lwp_bind_stack + BDSSIZE;

				/* c stack */
  /* cssize is different now for the main thread only, but you might
     want to create threads with different stack sizes             */

/*  npd->lwp_cssize = CSSIZE; moved in the new stack creation part  */

#ifdef DOWN_STACK
  npd->lwp_cs_org = new_pd->pd_base + STACK_SIZE - 1;
  npd->lwp_cs_limit = new_pd->pd_base - 1;
#else
  npd->lwp_cs_org = new_pd->pd_base;
  npd->lwp_cs_limit = npd->lwp_cs_org + STACK_SIZE;
#endif    
				/* invocation history stack */
  npd->lwp_ihs_top = npd->lwp_ihs_stack - 1;
  npd->lwp_ihs_limit = npd->lwp_ihs_stack + IHSSIZE;
				/* frame stack */
  npd->lwp_frs_top = npd->lwp_frame_stack - 1;
  npd->lwp_frs_limit = npd->lwp_frame_stack + FRSSIZE;
				/* value stack */
  npd->lwp_vs_top = npd->lwp_vs_base = npd->lwp_Values;
  npd->lwp_vs_limit = npd->lwp_Values + VSSIZE;
  
  npd->lwp_alloc_temporary = OBJNULL;
  npd->lwp_backq_level = 0;
  npd->lwp_bind_temporary = Cnil;
  npd->lwp_eval1 = 0;
				/* for gc */
  npd->lwp_fmt_temporary_stream = OBJNULL;
  npd->lwp_fmt_temporary_string = OBJNULL;
  
  npd->lwp_PRINTstream = Cnil;
  npd->lwp_PRINTescape = TRUE;
  npd->lwp_PRINTpretty = FALSE;
  npd->lwp_PRINTcircle = FALSE;
  npd->lwp_PRINTbase = 10;
  npd->lwp_PRINTradix = FALSE;
  npd->lwp_PRINTcase = Kdowncase;
  npd->lwp_PRINTgensym = TRUE;
  npd->lwp_PRINTlevel = -1;
  npd->lwp_PRINTlength = -1;
  npd->lwp_PRINTarray = FALSE;
  npd->lwp_write_ch_fun = writec_PRINTstream;
  npd->lwp_output_ch_fun = writec_PRINTstream;
  npd->lwp_read_ch_fun = readc;
  
  npd->lwp_READtable  = symbol_value(Vreadtable);
  npd->lwp_READdefault_float_format = 'F';
  npd->lwp_READbase = 10;
  npd->lwp_READsuppress = FALSE;
  npd->lwp_sharp_eq_context_max = 0;
  npd->lwp_delimiting_char = OBJNULL;
  npd->lwp_detect_eos_flag = FALSE;
  npd->lwp_in_list_flag = FALSE;
  npd->lwp_dot_flag = FALSE;
  
				/* for gc */
  npd->lwp_big_register_0 = OBJNULL;
  
  npd->lwp_string_register = OBJNULL;
  npd->lwp_gensym_prefix = OBJNULL;
  npd->lwp_gentemp_prefix = OBJNULL;
  npd->lwp_token = OBJNULL;
  
  /* ihs_push(Cnil) */
  (++npd->lwp_ihs_top)->ihs_function = Cnil;
  npd->lwp_ihs_top->ihs_base = npd->lwp_vs_base;
  
				/* lex_env copy */
  npd->lwp_ihs_top->ihs_base = npd->lwp_vs_top;
  npd->lwp_lex_env = npd->lwp_vs_top;
  npd->lwp_vs_top[0] = lex_env[0];
  npd->lwp_vs_top[1] = lex_env[1];
  npd->lwp_vs_top[2] = lex_env[2];
  npd->lwp_vs_top += 3;
  
  npd->lwp_vs_base = npd->lwp_vs_top;
  
  /* Now the allocation. If the gc is invoked we are able to mark
   * the objects already allocated
   */

  PUSH(new_pd);			/* so that it can be reached */

  npd->lwp_fmt_temporary_stream = make_string_output_stream(64);
  npd->lwp_fmt_temporary_string =
    npd->lwp_fmt_temporary_stream->sm.sm_object0;
  
  npd->lwp_big_register_0 = alloc_object(t_bignum);
  npd->lwp_big_register_0->big.big_car = 0;
  npd->lwp_big_register_0->big.big_cdr = NULL;
  
  npd->lwp_string_register = alloc_simple_string(0);
  npd->lwp_gensym_prefix = make_simple_string("G");
  npd->lwp_gentemp_prefix = make_simple_string("T");
  npd->lwp_token = alloc_simple_string(LISP_PAGESIZE);
  npd->lwp_token->st.st_self = alloc_contblock(LISP_PAGESIZE);
  npd->lwp_token->st.st_fillp = 0;
  npd->lwp_token->st.st_hasfillp = TRUE;
  npd->lwp_token->st.st_adjustable = TRUE;
  
  /* no need to mark it any more */
  dequeue();
  return new_pd;
}

activate_thread(object thread)
{
  pd *npd = thread->th.th_self;

  /* push the initial function on the stack of thread.
     Resume will push the arguments. */
  npd->pd_lpd->lwp_vs_base[0] = thread->th.th_fun;
  (npd->pd_lpd->lwp_vs_top)++;

#define STACK_MARGIN 160	/* longjmp writes also under the sp level */

  /* jump on the new c stack */
  val = sigsetjmp(npd->pd_env, 1);
  if (val == 0) {
#ifdef DOWN_STACK
    npd->pd_env[SP_INDEX] =
      stack_align((int)(npd->pd_base) + 4*STACK_SIZE - STACK_MARGIN);
    npd->pd_lpd->lwp_cssize =
      npd->pd_env[SP_INDEX] - (int)npd->pd_base;
#else
    npd->pd_env[SP_INDEX] =
      stack_align((int)(npd->pd_base));
    npd->pd_lpd->lwp_cssize = 4*STACK_SIZE - STACK_MARGIN;
#endif DOWN_STACK
    return;
  } 
#undef STACK_MARGIN  
  
  /* WARNING: args and locals are no more accessible from here on,
   * since reentering with longjmp does not restore previous frame
   */

#ifdef DEBUG1
  printf("ora parte il nuovo thread\n");
  fflush(stdout);
#endif
  
  flush_stream(symbol_value(Vstandard_output));
  
  super_funcall(siSthread_top_level);

  enable_scheduler();

  /* KILL */
  
  /* stop the timer because you are going to call the scheduler */
  
  virtual_ualarm(0);
  terpri(Cnil);			/* Lfinish_output uses VS. Beppe */
  running_head->pd_status = DEAD;
  running_head->pd_thread->th.th_self = NULL;
  running_processes--;
  
  system_enable_scheduler();
  
  /* call the scheduler now to switch to another process */
  running_tail->pd_next = running_head;
  running_head = running_head->pd_next;
  running_tail = running_tail->pd_next;
  running_tail->pd_next = NULL;
  
  /*    ROTQUEUE();     memory allocated for the thread will be freed by the
   * scheduler at the next round to avoid to free the stack
   * you are working on
   */

  /* I think this is useless and the scheduler could be simplified 
     eliminating the DEAD case.
   */
  
#ifdef DEBUG1
  printf("\n ************************ KILL ************************* \n");
  fflush (stdout);
#endif
  
  intha(DEAD, 0, 0);
}

intha(int sig, int code, struct sigcontext *scp)
{
  register pd *dead_pd;
  pd *old_head = running_head;

#ifdef DEBUG2
printf("entro nella intha %d \n",sig);
fflush(stdout);
#endif

#ifdef SYSV
  signal(SIGALRM, intha);
#endif SYSV

  if (running_processes == 0) {
    sleep(1);			/* are there possible interferences with SIGALRM ? */
    absolute_time += 1000000 / REALQUANTUM;
  }
  else
    absolute_time++;

#ifdef DEBUG1
  printf("scheduler %d %d\n", critical_semaphore, was_interrupted);
  fflush(stdout);
#endif

  /* case of the critical region */
  if (critical_semaphore == BUSY)  {
      was_interrupted = TRUE;
      return;
    } 

  switch (sig)   {
    /* examine the queue */
  case CRITICAL  :		/* called directly from end_critical_section */
  case SUSPENDED :		/* called directly when a thread suspends */
  case BREAK     :		/* called during the SIGINT handling */
  case DEACTIVATE:		/* called directly when a thread deactivates */
  case SIGALRM   :		/* called by the handler of timer interrupt */
#ifdef BSD
  case SIGVTALRM :		/* case of virtual timer. not used */
  case SIGPROF   :
#endif BSD

    val1 = sigsetjmp(running_head->pd_env, 1);

    if (val1 == 1) {		/* coming from longjmp in intha from another thread */
#ifdef DEBUG
      printf("scheduler: after longjmp\n");
      siLfree_list();  
#endif
      return;			/* from interrupt, resuming thread */
    }

    if (val1 == 2)              /* coming from longjmp in GC */
      GC(garbage_parameter);	/* GC will return to the previous thread */

    /* unwind the bind stack */
    lwp_bds_unwind(clwp->lwp_bind_stack, clwp->lwp_bds_top);

    /* save Values pointers */
    clwp->lwp_vs_base = vs_base;
    clwp->lwp_vs_top = vs_top;

    ROTQUEUE();

    /* just an attempt. it is better to hold the stopped lwp's
     * in a different queue?
     */

  case DEAD:			/* called directly when a thread terminates
				 * execution or go on in the other cases to
				 * dequeue killed threads
				 */
    do
      switch (running_head->pd_status) {

      case DEAD:
	/* remove the dead process */
	dead_pd = dequeue();
	  
	/* free the lisp descriptor */
	free (dead_pd->pd_lpd);
	  
	/* free the memory allocated for the stack and the descriptor */
	free (dead_pd->pd_base);
	free (dead_pd);
	break;

      case SUSPENDED:

	if (running_head->pd_slice != 0) 
	  if (absolute_time > running_head->pd_slice) {

	    /* the time slice has expired */
	    running_head->pd_slice = 0;

	    if ((running_head->pd_thread->th.th_cont) != OBJNULL) {
	      /* in this case a continuation was created before %delay */
	      running_head->pd_thread->th.th_cont->cn.cn_timed_out = TRUE;
	      running_head->pd_thread->th.th_cont = OBJNULL;
	    }
	    running_head->pd_status = RUNNING;
	    goto OUT;		/* now you are a running process */
	  }
	ROTQUEUE();
	break;
	  
      case WAITING:		/* waiting processes need to be scheduled  */

      case RUNNING:
	goto OUT;		/* found schedulable process */
	
      default:			/* currently is only STOPPED */
	ROTQUEUE();
	break;
      }

    while (running_head != old_head);

  OUT:
				/* switch clwp */
    clwp = running_head->pd_lpd;

				/* restore Values pointers */
    vs_base = clwp->lwp_vs_base;
    vs_top = clwp->lwp_vs_top;

				/* wind the bind stack */
    lwp_bds_wind(clwp->lwp_bind_stack, clwp->lwp_bds_top);

				/* reset the timer */
    timer_state = INACTIVE;

    for (dead_pd = running_head->pd_next; dead_pd != NULL;
	 dead_pd = dead_pd->pd_next)
      if (dead_pd->pd_status == RUNNING || /* reasons to start the timer are: */
	  dead_pd->pd_slice != 0        || /* another running or suspended with */
	  dead_pd->pd_status == WAITING) { /* time_out slice or waiting */

	timer_state = ACTIVE;

#ifdef DEBUG1
	printf("sto facendo la setitimer dello scheduler\n");
	fflush(stdout);
#endif

	virtual_ualarm(REALQUANTUM);
	break;
      }
  }

#ifdef DEBUG1
  printf("sto uscendo dallo scheduler\n");
  fflush(stdout);
#endif
#ifdef DEBUG    
  printf("sheduler-prelongjmp\n");
  siLfree_list();  
#endif DEBUG

  siglongjmp(running_head->pd_env, 1); 
}


lwp_bds_wind(bds_ptr base, bds_ptr top)
{
  object temp;

  for (; top >= base; base++) {
    temp = (base->bds_sym)->s.s_dbind;
    (base->bds_sym)->s.s_dbind = base->bds_val;
    base->bds_val = temp;
  }
}

lwp_bds_unwind(bds_ptr base, bds_ptr top)
{
  object temp;

  for (; top >= base; top--) {
    temp = (top->bds_sym)->s.s_dbind;
    (top->bds_sym)->s.s_dbind = top->bds_val;
    top->bds_val = temp;
  }
}

scheduler_reconfig()
{
  if (real_scheduler_level > 0) {
    scheduler_disabled = TRUE;
    scheduler_level = real_scheduler_level;
    real_scheduler_level = 0;
    /* this is for the critical_section implementing the disable_scheduler */
    start_critical_section();
  }
}

resume(pd *rpd)
{
  register pd *p;

  /* This must be called inside a critical section */

  if (timer_state != ACTIVE)  {
     timer_state = ACTIVE;

     /* this is for the not executed critical_section in Lresume */
     start_critical_section();

     /* this is to correctly reconfig the scheduler situation */
     scheduler_reconfig();

#ifdef DEBUG1
       printf("sto facendo la setitimer della resume\n");
       fflush(stdout);
#endif

       virtual_ualarm(REALQUANTUM);
   }

  rpd->pd_status = RUNNING;
  running_processes++;

  for (p = running_head; (p != rpd) && (p != NULL); p = p->pd_next)
    ;
  
  if (p == NULL) ENQUEUE(rpd);
}




/* ----------------------------------------------------------------------

 The following is the convention used by C routines on stack usage
 on the MC680x0:

		|//////////|
	sp --->	| old reg m|  \
		|    .	   |   > Previous values of the registers
		| old reg 1|  /  used by the subroutine
		| Local n  |
		|    .	   |
		| Local 3  |
		| Local 2  |
		| Local 1  |
	a6 ---> | old a6   |
		| ret adr  |
		| Param. 1 |
		| Param. 2 |
		| Param. 3 |

 Notes:
 1. Locals are allocated immediately after the call
 2. parameters are popped by the caller after return

   ---------------------------------------------------------------------- */




/***********
 *
 * THREADS
 *
 ***********/




siLthread_break_in(int narg)
{
/*   start_critical_section(); */
  alarm(0);
  VALUES(0) = Cnil;
  RETURN(1);
}

siLthread_break_quit(int narg)
{
  /* reset everything in MT */
  pd *p;

  /* this is done in any case to remedy the problem with C-c handling */
  signal(SIGALRM, intha);

  if (timer_state == ACTIVE) {
     /* reset the critical and disable-scheduler environment    */
      scheduler_disabled = FALSE;
      scheduler_level = 0;
      critical_semaphore = FREE;
      critical_level = 0;
      was_interrupted = 0;

      for (p = running_head; (p != NULL); p = p->pd_next)
	if (p != &main_pd)
	  p->pd_status = DEAD;
	else {
	  p->pd_status = RUNNING;
	  p->pd_thread->th.th_cont = OBJNULL;
	}



      if (running_head != &main_pd) {
	ROTQUEUE();		/* this is to avoid to free the stack of
				 * the executing thread
				 */
	  
	intha(DEAD,0,0);	/* here is freed memory of the threads that
				 * are enqueued before the main thread
				 */
 
	/*  We don't use call_scheduler to avoid passing control
	 *  to another thread
	 */

	/*  here is necessary to have the main-thread function     **
	 *  deallocation                                           **
	 */
      }
      else
	intha(BREAK,0,0);  /* Here the intha call is to free memory */

      /*  We don't use call_scheduler to avoid passing control
       *  to another thread
       */
    }
  VALUES(0) = Cnil;
  RETURN(1);
}

siLthread_break_resume(int narg)
{
  /* this is to restart the alarmclock conditions that could be 
   * changed by the C-c handling
   */
  
  signal(SIGALRM, intha);  
/*  end_critical_section(); */
  if (timer_state == ACTIVE)
    call_scheduler(BREAK);
  VALUES(0) = Cnil;
  RETURN(1);
}

Lthread_list(int narg)
{
  pd *p;
  object tmp, x = CONS(running_head->pd_thread, Cnil);

  tmp = x;

  start_critical_section();

  for (p = running_head->pd_next; (p != NULL); p = p->pd_next) {
      CDR(tmp) = CONS(p->pd_thread, Cnil);
      tmp = CDR(tmp);
    }

  end_critical_section();

  VALUES(0) = x;
  RETURN(1);
}

Lmake_thread(int narg, object fun)
{
  object x;
  pd *npd;
  /* Just one argument for the time being */
  check_arg(1);

  if (type_of(fun) == t_symbol) {
    if (fun->s.s_sfdef != NOT_SPECIAL ||
	fun->s.s_mflag)
      FEinvalid_function(fun);
    if (fun->s.s_gfdef == OBJNULL)
      FEundefined_function(fun);
    fun = fun->s.s_gfdef;
  }

  x = alloc_object(t_thread);
  x->th.th_fun = fun;
  x->th.th_size = sizeof (pd);

  x->th.th_self = npd = make_pd();
  activate_thread(x);

  npd->pd_thread = x;
  npd->pd_slice = 0;
  x->th.th_cont = OBJNULL;

  /* Backpointer to thread */
  npd->pd_lpd->lwp_thread = x;

  VALUES(0) = x;
  RETURN(1);
}

Ldeactivate(int narg, object thread)
{
  check_arg(1);
  if (type_of(thread) != t_thread)
    FEwrong_type_argument(Sthread, thread);

  if (thread->th.th_self == NULL ||
      ((pd *)(thread->th.th_self))->pd_status != RUNNING)
    FEerror("Cannot deactivate a thread not running", 0);

  ((pd *)(thread->th.th_self))->pd_status = STOPPED;
  running_processes--;
  if (((pd *)(thread->th.th_self)) == running_head) {
    system_enable_scheduler();
    call_scheduler(DEACTIVATE);
  }
  VALUES(0) = Cnil;
  RETURN(1);
}

Lreactivate(int narg, object thread)
{
  check_arg(1);
  start_critical_section();

  if (type_of(thread) != t_thread) {
    FEwrong_type_argument(Sthread, thread);
  }

  if (thread->th.th_self == NULL ||
      ((pd *)(thread->th.th_self))->pd_status != STOPPED)
    FEerror("Cannot reactivate a thread not stopped", 0);

  ((pd *)(thread->th.th_self))->pd_status = RUNNING;
  running_processes++;

  if (timer_state != ACTIVE)  {

     timer_state = ACTIVE;

     /* this is for the not executed critical_section in Lreactivate */
     start_critical_section();

     /* this is to correctly reconfigure the scheduler situation     */
     scheduler_reconfig();

     virtual_ualarm(REALQUANTUM);
   }

  end_critical_section();
  VALUES(0) = Cnil;
  RETURN(1);

}

Lkill_thread(int narg, object thread)
{

/* The following code is not enough.
   Consider: The scheduler can be disabled
             What about killing the current thread?
 */
  check_arg(1);

  if (type_of(thread) != t_thread)
    FEwrong_type_argument(Sthread, thread);

  if (thread->th.th_self != NULL) {
    ((pd *)(thread->th.th_self))->pd_status = DEAD;
    thread->th.th_self = NULL;
  }

  VALUES(0) = Cnil;
  RETURN(1);
}

Lcurrent_thread(int narg)
{
  check_arg(0);
  VALUES(0) = clwp->lwp_thread;
  RETURN(1);
}

Lthread_status(int narg, object thread)
{
  check_arg(1);

  if (type_of(thread) != t_thread) {
    FEwrong_type_argument(Sthread, thread);
  }

  if (thread->th.th_self != NULL) {
    switch (((pd *)(thread->th.th_self))->pd_status) {
    case RUNNING:
      VALUES(0) = Srunning;
      break;
    case SUSPENDED:
      VALUES(0) = Ssuspended;
      break;
    case WAITING:
      VALUES(0) = Swaiting;
      break;
    case STOPPED:
      VALUES(0) = Sstopped;
      break;
    case DEAD:
      VALUES(0) = Sdead;
      break;
    default:
      fprintf(stderr, "Unexecpected type for thread\n");
      exit (-1);
    }
  }
  else {
    VALUES(0) = Sdead;
  }
  RETURN(1);
}


/***************
 *
 * CONTINUATIONS
 *
 ***************/

Lmake_continuation(int narg, object thread)
{
  object x;
  check_arg(1);

  if (type_of(thread) != t_thread)
    FEwrong_type_argument(Sthread, thread);

  if (thread->th.th_cont)
    FEerror("A continuation for thread ~A already exists.", 1, thread);

  if ((thread->th.th_self == NULL) ||
      ((pd *)(thread->th.th_self))->pd_status == DEAD) {
    FEerror("Thread ~A is DEAD.", 1, thread);
  }

  x = alloc_object(t_cont);

  x->cn.cn_thread = thread;
  x->cn.cn_resumed = FALSE;
  x->cn.cn_timed_out = FALSE;

  VALUES(0) = thread->th.th_cont = x;
  RETURN(1);
}

/* Returns the thread associated to a continuation */
Lthread_of(int narg, object cont)
{
  check_arg(1);
  if (type_of(cont) != t_cont)
    FEwrong_type_argument(Scont, cont);
  VALUES(0) = cont->cn.cn_thread;
  RETURN(1);
}

/* Returns the continuation associated to a thread, if it exists */
Lcontinuation_of(int narg, object thread)
{
  check_arg(1);
  if (type_of(thread) != t_thread)
    FEwrong_type_argument(Sthread, thread);
  if (thread->th.th_cont)
    VALUES(0) = thread->th.th_cont;
  else 
    VALUES(0) = Cnil;
  RETURN(1);
}

Lresume()
{ int i, narg;
  object *thread_vs_top;
  object cont = vs_base[0];

  narg = vs_top - vs_base;
  if (narg < 1) FEtoo_few_arguments(&narg);

  if (Null(cont)) {
    vs_base++;
    return;
  }

  if (type_of(cont) != t_cont)
    FEwrong_type_argument(Scont, cont);

  if (cont->cn.cn_resumed)
    FEerror("The continuation has already been resumed.", 0);
  
  if (cont->cn.cn_timed_out)
    FEerror("The continuation has been timed out.", 0);

  start_critical_section();

  if (cont->cn.cn_thread->th.th_self != NULL) {
    if (((pd *)(cont->cn.cn_thread->th.th_self))->pd_status != SUSPENDED)
      FEerror("The continuation cannot be resumed. Its thread isn't suspended", 0);

    /* Push the arguments on the value stack of thread */

    thread_vs_top =
      (((pd *)(cont->cn.cn_thread->th.th_self))->pd_lpd)->lwp_vs_top;

    for (i = 1; i < narg; i++)
      *(thread_vs_top++) = vs_base[i];

    (((pd *)(cont->cn.cn_thread->th.th_self))->pd_lpd)->lwp_vs_top =
      thread_vs_top;

    cont->cn.cn_resumed = TRUE;
    cont->cn.cn_thread->th.th_cont = OBJNULL;

    /* If you are waiting on a slice expiring I reset your slice    */
    ((pd *)(cont->cn.cn_thread->th.th_self))->pd_slice = 0;

    resume((pd *)(cont->cn.cn_thread->th.th_self));

    VALUES(0) = cont->cn.cn_thread;;
  }
  else
    FEerror("The continuation cannot be resumed. Its thread is DEAD.", 0);

  end_critical_section();
  RETURN(1);
}


/***************
 *
 * SCHEDULING
 *
 ***************/


Ldisable_scheduler(int narg)
{
  check_arg(0);
  disable_scheduler();
  VALUES(0) = Cnil;
  RETURN(1);
}

disable_scheduler()
{
  start_critical_section();

  if (timer_state != ACTIVE)  {
    real_scheduler_level++;
    end_critical_section();
    return;
  }

  if (scheduler_level++ > 0) end_critical_section();

  scheduler_disabled = TRUE;
}

Lenable_scheduler(int narg)
{
  check_arg(0);
  enable_scheduler();
  VALUES(0) = Cnil;
  RETURN(1);
}

enable_scheduler()
{
  start_critical_section();

  if (timer_state != ACTIVE)  {
    if (real_scheduler_level > 0) 
      real_scheduler_level--;
    end_critical_section();
    return;
  }

  if (scheduler_disabled == TRUE)
     if (--scheduler_level == 0)  {
        scheduler_disabled = FALSE;
        end_critical_section();
      }
  end_critical_section();
}

system_enable_scheduler()
{
  /* this is to brutally re-enable the scheduler at any level */

  if (scheduler_disabled)  {
    scheduler_disabled = FALSE;
    scheduler_level = 0;
    real_scheduler_level = 0;

    /* this is for the critical_section implementing disable_scheduler  */
    was_interrupted = FALSE;
    critical_level--;
    critical_semaphore = FREE;
  }
}

Lsuspend(int narg)
{
  check_arg(0);

  if (timer_state == ACTIVE) {
      /* stop the timer because you are going to call the scheduler */

#ifdef DEBUG1
       printf("sto facendo la setitimer a zero della suspend\n");
       fflush(stdout);
#endif

       virtual_ualarm(0);

       running_head->pd_status = SUSPENDED;
       running_processes--;

    /* We have to re-enable the scheduler anyway */
       system_enable_scheduler();

       intha(SUSPENDED, 0, 0);
    
    /* When resumed it will be provided with the values to return */
     }

  else 
    FEerror("You might wait forever for lack of active processes.",0);
}

Ldelay(int narg, object interval)
{ int z;
  
  check_arg(1);
  check_type_non_negative_integer(&interval);
  z = fix(interval);
  
  if (timer_state == ACTIVE) {
    /* stop the timer because you are going to call the scheduler */
    
#ifdef DEBUG1
       printf("sto facendo la setitimer a zero della delay\n");
       fflush(stdout);
#endif

       virtual_ualarm(0);
       
       running_head->pd_status = SUSPENDED;
       
       /* We have to translate seconds in intha call number    */
       running_head->pd_slice = z * 10 + absolute_time;

       /* We have to re-enable the scheduler anyway */

       system_enable_scheduler();
       
       intha(SUSPENDED, 0, 0);
       
       /* When resumed it will be provided with the values to return */
     }
  else
    sleep(z);
}

Lthread_wait(int narg, object fun, ...)
{ int nr;
  va_list args;
  va_start(args, fun);

  if (narg < 1) FEtoo_few_arguments(&narg);

  running_head->pd_status = WAITING;
  running_processes--;

  for (;;) {
    
    nr = apply(narg-1, fun, args);
    
    if (VALUES(0) != Cnil)
      break;
    else if (timer_state == ACTIVE) {
      virtual_ualarm(0);
      system_enable_scheduler();

      /* the time slice has not been used */
      absolute_time--;

      intha(14,0,0);
    } else
      FEerror("The condition will never be satisfied for lack of active processes", 0);
  }
  running_head->pd_status = RUNNING;
  RETURN(nr);
}
  
      
Lthread_wait_with_timeout(int narg, object timeout, object fun, ...)
{
  int nr;
  va_list args;
  va_start(args, fun);

  if (narg < 2) FEtoo_few_arguments(&narg);
  check_type_non_negative_integer(&timeout);

  /* We have to translate seconds in intha call number */
  running_head->pd_slice = fix(timeout) * 10 + absolute_time;
  
  running_head->pd_status = WAITING;
  running_processes--;
  
  for (;;) {
    
    if (absolute_time > running_head->pd_slice) {
      /* the time slice has expired */
      VALUES(0) = Cnil;
      nr = 1;
      break;
    }

    nr = apply(narg-1, fun, args);

    if (VALUES(0) != Cnil)
      break;
    else {
      virtual_ualarm(0);
      system_enable_scheduler();

      /* the time slice has not been used */
      absolute_time--;

      intha(14,0,0);
    }
  }
  
  running_head->pd_slice = 0;
  
  running_head->pd_status = RUNNING;
  running_processes++;
  RETURN(nr);
}

enable_lwp()
{
  signal(SIGALRM, intha);
}
  

init_lwp()
{ pd *temp_pd;

  temp_pd = &main_pd;
  PUSH(temp_pd);

  main_thread = alloc_object (t_thread);
  main_pd.pd_thread = main_thread;
  main_thread->th.th_fun = Cnil;
  main_thread->th.th_size = sizeof (pd);
  main_thread->th.th_self = &main_pd;
  main_thread->th.th_cont = OBJNULL;
  /* Backpointer to thread */
  main_pd.pd_status = RUNNING;
  main_pd.pd_lpd = &main_lpd;
  main_lpd.lwp_thread = main_thread;
  enter_mark_origin(&main_thread);

  Srunning = make_ordinary("RUNNING");
  Ssuspended = make_ordinary("SUSPENDED");
  Swaiting = make_ordinary("WAITING");
  Sstopped = make_ordinary("STOPPED");
  Sdead = make_ordinary("DEAD");
  siSthread_top_level = make_si_ordinary("THREAD-TOP-LEVEL");

  make_si_function("THREAD-BREAK-IN", siLthread_break_in);
  make_si_function("THREAD-BREAK-QUIT", siLthread_break_quit);
  make_si_function("THREAD-BREAK-RESUME", siLthread_break_resume);

  make_function("MAKE-THREAD", Lmake_thread);
  make_function("DEACTIVATE", Ldeactivate);
  make_function("REACTIVATE", Lreactivate);
  make_function("KILL-THREAD", Lkill_thread);
  
  make_function("CURRENT-THREAD", Lcurrent_thread);
  make_function("THREAD-STATUS", Lthread_status);
  make_function("THREAD-LIST", Lthread_list);
  
  make_function("MAKE-CONTINUATION", Lmake_continuation);
  make_function("THREAD-OF", Lthread_of);
  make_function("CONTINUATION-OF", Lcontinuation_of);
  make_function("RESUME", Lresume);

  make_function("%DISABLE-SCHEDULER", Ldisable_scheduler);
  make_function("%ENABLE-SCHEDULER", Lenable_scheduler);

  make_function("%SUSPEND", Lsuspend);
  make_function("%DELAY", Ldelay);
  make_function("%THREAD-WAIT", Lthread_wait);
  make_function("%THREAD-WAIT-WITH-TIMEOUT", Lthread_wait_with_timeout);

}
