/*
    unify.d -- Support for unification.
*/
/*
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

object *slot;			/* scanning pointer within object */
int (*slotf)();			/* read/write mode accessor */

extern object allocate_instance();
extern object list();
extern object Sprogn;

/* -------------------- Trail Instructions -------------------- */

object *trail[VSSIZE];
object **trail_top = trail;
#define	trail_push(loc)		(*trail_top++ = (loc))
#define	trail_pop		(**--trail_top = OBJNULL)

#define BIND(loc, val)		{loc = val; trail_push(&loc);}

#define trail_mark		trail_push((object *)NULL)

@(defun trail_mark ()
@
  trail_mark;
@)

#define trail_restore		{while (trail_top[-1] != (object *)NULL) \
				   trail_pop;}

@(defun trail_restore ()
@
  trail_restore;
  @(return Cnil)
@)

#define trail_unmark		{trail_restore; trail_top--;}

@(defun trail_unmark ()
@
  trail_unmark;
  @(return Cnil)
@)

/* -------------------- Mode Operators -------------------- */

bool get_slot(object x)		/* read mode */
{
  if (x == *slot || unify(x, *slot))
    if (*slot == OBJNULL)
      return((bool)MAKE_LOCATIVE(slot++));
    else
      return((bool)*slot++);	/* dereference */
  else
    return(FALSE);
}

bool set_slot(object x)		/* write mode */
{
  /* NOTE: slot contains OBJNULL */
  *slot = x;
  return((bool)MAKE_LOCATIVE(slot++));
}


/* -------------------- Get Instructions -------------------- */

/* get_variable is just setq */

#define get_value(v, x)		unify(x, v)

@(defun get_value (v x)
@
	@(return `get_value(v, x)?Ct:Cnil`)
@)

#define get_constant(c, x)	(c == x || unify(x, c))

@(defun get_constant (c x)
@
	@(return `get_constant(c, x)?Ct:Cnil`)
@)

#define get_nil(x)		(Cnil == x || unify(x, Cnil))

@(defun get_nil (arg)
@
	@(return `get_nil(arg)?Ct:Cnil`)
@)

bool
get_cons(object x)
{

RETRY:	switch (type_of(x)) {
	case t_cons:
	  slot = &CDR(x);	/* cdr slot is first in struct cons */
	  slotf = get_slot;
	  return(TRUE);

	case t_locative:
	  if (UNBOUNDP(x)) {
	    object new = CONS(OBJNULL, OBJNULL);
	    BIND(DEREF(x), new);
	    slot = &CDR(new);
	    slotf = set_slot;
	    return(TRUE);
	  }
	  else {
	    x = DEREF(x);
	    goto RETRY;
	  }

	default: return(FALSE);
	}
	    
}

@(defun get_cons (arg)
@
	@(return `get_cons(arg)?Ct:Cnil`)
@)

bool
get_instance(object x, object class, int arity)
{
RETRY:	switch (type_of(x)) {
	case t_instance: 
	if (CLASS_OF(x) == class) {
	  slot = x->in.in_slots;
	  slotf = get_slot;
	  return(TRUE);
	} else
	  return(FALSE);

	case t_locative:
	  if (UNBOUNDP(x)) {
	    object new = allocate_instance(class, arity);
	    BIND(DEREF(x), new);
	    slot = new->in.in_slots;
	    slotf = set_slot;
	    return(TRUE);
	  }
	  else {
	    x = DEREF(x);
	    goto RETRY;
	  }
	default: return(FALSE);
	}
}

@(defun get_instance (x class arity)
@
	@(return `get_instance(x, class, fix(arity))?Ct:Cnil`)
@)


/* -------------------- Unify Instructions --------------------  */

#define UNIFY_LOCATIVE(x, y, L)	{object *p = &DEREF(x); \
				   if (*p == OBJNULL) { \
				     BIND(*p, y); return(TRUE); } \
				       else { x = *p; goto L;}}
/*
#define UNIFY_LOCATIVE(x, y, L) {if (UNBOUNDP(x)) { \
				     BIND(DEREF(x), y); return(TRUE); } \
				       else { x = DEREF(x); goto L;}}
*/

bool
unify(object x, object y)
{
  /* NOTE: x <- y */

  L: switch (type_of(x)) {

  case t_locative: UNIFY_LOCATIVE(x, y, L);

  case t_cons:
       L1: switch (type_of(y)) {

       case t_cons: return(unify(CAR(x), CAR(y)) &&
			   unify(CDR(x), CDR(y)));

       case t_locative: UNIFY_LOCATIVE(y, x, L1);

       default: return(FALSE);
       }

  case t_instance:
       L2: switch (type_of(y)) {

       case t_instance:
	 if (CLASS_OF(x) == CLASS_OF(y)) {
	   int l = x->in.in_length; int i;
	   object *slotx = x->in.in_slots;
	   object *sloty = y->in.in_slots;
	   for (i = 0; i < l; i++) {
	     if (!unify(*slotx++, *sloty++))
	       return(FALSE);
	   }
	   return(TRUE);
	 } else
	   return(FALSE);

       case t_locative: UNIFY_LOCATIVE(y, x, L2);

       default: return(FALSE);
       }

  default:
    L3: if (LOCATIVEP(y))
           UNIFY_LOCATIVE(y, x, L3)
        else if (equal(x,y))
	  return(TRUE);
	else
	  return(FALSE);
  }
}

/* Internal function. One should use unify_variable, which always returns T */

#define unify_slot	(*slotf)(*slot)

@(defun unify_slot ()
@
	@(return `(object)unify_slot`)
@)


#define unify_value(loc)		(*slotf)(loc)

@(defun unify_value (loc)
  object x;
@
	x = (object)unify_value(loc);
	@(return `(x == Cnil || x)?Ct:Cnil`)
@)

#define unify_constant(c)	(*slotf)(c)

@(defun unify_constant (c)
  object x;
@
	x = (object)unify_constant(c);
	@(return `(x == Cnil || x)?Ct:Cnil`)
@)

#define unify_nil	(*slotf)(Cnil)

@(defun unify_nil ()
  object x;
@
	x = (object)unify_nil;
	@(return `(x == Cnil || x)?Ct:Cnil`)
@)

/* -------------------- Test Functions -------------------- */

@(defun make_locative (&optional (n 0))
@
	@(return `MAKE_LOCATIVE(fix(n))`)
@)

@(defun locativep (obje)
@
        @(return `LOCATIVEP(obje)?Ct:Cnil`)
@)

@(defun unboundp (loc)
@
        @(return `UNBOUNDP(loc)?Ct:Cnil`)
@)

@(defun dereference (x)
  extern object Slocative;
@
	while (type_of(x) != t_locative)
	  x = wrong_type_argument(Slocative, x);
	@(return `DEREF(x)`)
@)

@(defun make_variable (name)
@
	@(return `CONS(name, OBJNULL)`)
@)

/* (defmacro unify-variable (v) `(progn (setq ,v (si:unify-slot)) t) */

object Ssetq, Sunify_slot;

Lunify_variable (object var)
{ VALUES(0) = list(3, Sprogn,
			    list(3, Ssetq, CADR(var),
				 CONS(Sunify_slot, Cnil)),
			    Ct);
  RETURN(1);
}

#define make_si_macro(name, cfun)	\
	{object x = make_si_ordinary(name); \
	   x->s.s_gfdef = make_cfun(cfun, Cnil, NULL); \
	   x->s.s_mflag = TRUE; \
	 }

init_unify()
{	extern Fsetq();

	make_si_function("TRAIL-MARK", Ltrail_mark);
	make_si_function("TRAIL-UNMARK", Ltrail_unmark);
	make_si_function("TRAIL-RESTORE", Ltrail_restore);
	make_special_form("GET-VARIABLE", Fsetq);
	make_si_function("GET-VALUE", Lget_value);
	make_si_function("GET-CONSTANT", Lget_constant);
	make_si_function("GET-NIL", Lget_nil);
	make_si_function("GET-CONS", Lget_cons);
	make_si_function("GET-INSTANCE", Lget_instance); /* Mauro */
	make_si_function("UNIFY-SLOT", Lunify_slot);
	make_si_function("UNIFY-VALUE", Lunify_value);
	make_si_function("UNIFY-CONSTANT", Lunify_constant);
	make_si_function("UNIFY-NIL", Lunify_nil);
	make_si_function("MAKE-LOCATIVE", Lmake_locative);
	make_si_function("LOCATIVEP",Llocativep);
	make_si_function("UNBOUNDP",Lunboundp);
	make_si_function("MAKE-VARIABLE", Lmake_variable);
	make_si_function("DEREFERENCE", Ldereference);
	Ssetq = make_ordinary("SETQ");
	Sunify_slot = make_si_ordinary("UNIFY-SLOT");
	make_si_macro("UNIFY-VARIABLE", Lunify_variable);
}
